/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.oauth2.client.storage.token.dao;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.oauth2.client.api.ClientToken;
import com.atlassian.oauth2.client.api.storage.token.ClientTokenEntity;
import com.atlassian.oauth2.client.api.storage.token.exception.TokenNotFoundException;
import com.atlassian.oauth2.client.lib.ClientTokenImpl;
import com.atlassian.oauth2.client.security.ClientPluginSecretServicePath;
import com.atlassian.oauth2.client.storage.AbstractStore;
import com.atlassian.oauth2.client.storage.token.dao.ClientTokenStore;
import com.atlassian.oauth2.client.storage.token.dao.ClientTokenStoreImpl;
import com.atlassian.oauth2.common.IdGenerator;
import com.atlassian.secrets.api.SecretService;
import com.atlassian.secrets.api.SecretServiceException;
import com.atlassian.secrets.api.SecretServiceState;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecuredClientTokenStore
extends AbstractStore
implements ClientTokenStore {
    private static final Logger logger = LoggerFactory.getLogger(SecuredClientTokenStore.class);
    private final SecretService secretService;
    private final ClientTokenStore baseClientTokenStore;

    public SecuredClientTokenStore(ActiveObjects activeObjects, IdGenerator idGenerator, SecretService secretService) {
        super(activeObjects);
        this.baseClientTokenStore = new ClientTokenStoreImpl(activeObjects, idGenerator);
        this.secretService = secretService;
    }

    @Override
    @Nonnull
    public ClientTokenEntity create(@Nonnull ClientTokenEntity clientTokenEntity) {
        return (ClientTokenEntity)this.activeObjects.executeInTransaction(() -> {
            if (this.isSecretServiceDisabled()) {
                logger.debug("Warning: SecretService is disabled, creating client token without encryption");
                return this.baseClientTokenStore.create(clientTokenEntity);
            }
            String accessToken = clientTokenEntity.getAccessToken();
            String refreshToken = clientTokenEntity.getRefreshToken();
            ClientTokenEntity createdToken = this.baseClientTokenStore.create(this.sanitizeTokenFields(clientTokenEntity));
            this.storeCreatedSecrets(createdToken.getId(), accessToken, refreshToken);
            return this.populateSecretFields(createdToken, accessToken, refreshToken);
        });
    }

    @Override
    @Nonnull
    public ClientTokenEntity create(@Nonnull ClientToken clientToken, @Nullable String externalId, @Nonnull String configId) {
        return (ClientTokenEntity)this.activeObjects.executeInTransaction(() -> {
            if (this.isSecretServiceDisabled()) {
                logger.debug("Warning: SecretService is disabled, creating client token without encryption");
                return this.baseClientTokenStore.create(clientToken, externalId, configId);
            }
            String accessToken = clientToken.getAccessToken();
            String refreshToken = clientToken.getRefreshToken();
            ClientTokenEntity createdToken = this.baseClientTokenStore.create(this.sanitizeTokenFields(clientToken), externalId, configId);
            this.storeCreatedSecrets(createdToken.getId(), accessToken, refreshToken);
            return this.populateSecretFields(createdToken, accessToken, refreshToken);
        });
    }

    private void storeCreatedSecrets(String tokenId, String accessToken, String refreshToken) {
        this.secretService.put(ClientPluginSecretServicePath.accessTokenPath(tokenId), accessToken);
        if (refreshToken != null) {
            this.secretService.put(ClientPluginSecretServicePath.refreshTokenPath(tokenId), refreshToken);
        }
    }

    @Override
    @Nonnull
    public Optional<ClientTokenEntity> update(@Nonnull ClientTokenEntity clientTokenEntity) {
        return (Optional)this.activeObjects.executeInTransaction(() -> {
            if (this.isSecretServiceDisabled()) {
                logger.debug("Warning: SecretService is disabled, updating client token without encryption");
                return this.baseClientTokenStore.update(clientTokenEntity);
            }
            String accessToken = clientTokenEntity.getAccessToken();
            String refreshToken = clientTokenEntity.getRefreshToken();
            Optional<ClientTokenEntity> updatedToken = this.baseClientTokenStore.update(this.sanitizeTokenFields(clientTokenEntity));
            updatedToken.ifPresent(token -> this.updateSecretFields(token.getId(), accessToken, refreshToken));
            return updatedToken.map(token -> this.populateSecretFields((ClientTokenEntity)token, accessToken, refreshToken));
        });
    }

    @Override
    @Nonnull
    public ClientTokenEntity updateOrFail(@Nonnull ClientTokenEntity clientTokenEntity) throws TokenNotFoundException {
        return this.executeInTransaction(() -> {
            if (this.isSecretServiceDisabled()) {
                logger.debug("Warning: SecretService is disabled, updating client token without encryption");
                return this.baseClientTokenStore.updateOrFail(clientTokenEntity);
            }
            String accessToken = clientTokenEntity.getAccessToken();
            String refreshToken = clientTokenEntity.getRefreshToken();
            ClientTokenEntity updatedToken = this.baseClientTokenStore.updateOrFail(this.sanitizeTokenFields(clientTokenEntity));
            this.updateSecretFields(updatedToken.getId(), accessToken, refreshToken);
            return this.populateSecretFields(updatedToken, accessToken, refreshToken);
        }, TokenNotFoundException.class);
    }

    private void updateSecretFields(String tokenId, String accessToken, String refreshToken) {
        this.secretService.put(ClientPluginSecretServicePath.accessTokenPath(tokenId), accessToken);
        if (refreshToken != null) {
            this.secretService.put(ClientPluginSecretServicePath.refreshTokenPath(tokenId), refreshToken);
        } else {
            this.secretService.delete(ClientPluginSecretServicePath.refreshTokenPath(tokenId));
        }
    }

    @Override
    public void delete(@Nonnull String id) throws TokenNotFoundException {
        this.executeInTransaction(() -> {
            this.baseClientTokenStore.delete(id);
            this.deleteTokenFromSecretStorage(id);
            return null;
        }, TokenNotFoundException.class);
    }

    @Override
    public List<String> deleteWithConfigId(@Nonnull String configId) {
        return (List)this.activeObjects.executeInTransaction(() -> this.baseClientTokenStore.deleteWithConfigId(configId).stream().peek(this::deleteTokenFromSecretStorage).collect(Collectors.toList()));
    }

    @Override
    public List<String> deleteTokensExpiringBefore(@Nonnull Instant timestamp) {
        return (List)this.activeObjects.executeInTransaction(() -> this.baseClientTokenStore.deleteTokensExpiringBefore(timestamp).stream().peek(this::deleteTokenFromSecretStorage).collect(Collectors.toList()));
    }

    @Override
    public List<String> deleteTokensUnrecoverableSince(@Nonnull Instant timestamp) {
        return (List)this.activeObjects.executeInTransaction(() -> this.baseClientTokenStore.deleteTokensUnrecoverableSince(timestamp).stream().peek(this::deleteTokenFromSecretStorage).collect(Collectors.toList()));
    }

    private void deleteTokenFromSecretStorage(String tokenId) {
        this.secretService.delete(ClientPluginSecretServicePath.accessTokenPath(tokenId));
        this.secretService.delete(ClientPluginSecretServicePath.refreshTokenPath(tokenId));
    }

    @Override
    @Nullable
    public ClientTokenEntity getById(@Nonnull String id) {
        return Optional.ofNullable(this.baseClientTokenStore.getById(id)).map(this::populateSecretFields).orElse(null);
    }

    @Override
    @Nonnull
    public ClientTokenEntity getByIdOrFail(@Nonnull String id) throws TokenNotFoundException {
        return this.populateSecretFields(this.baseClientTokenStore.getByIdOrFail(id));
    }

    @Override
    public Optional<ClientTokenEntity> getByExternalIdAndConfigId(@Nonnull String externalId, @Nonnull String configId) {
        return this.baseClientTokenStore.getByExternalIdAndConfigId(externalId, configId).map(this::populateSecretFields);
    }

    @Override
    @Nonnull
    public List<ClientTokenEntity> getAccessTokensExpiringBefore(@Nonnull Instant timestamp) {
        return this.baseClientTokenStore.getAccessTokensExpiringBefore(timestamp).stream().map(this::populateSecretFields).collect(Collectors.toList());
    }

    @Override
    @Nonnull
    public List<ClientTokenEntity> getRefreshTokensExpiringBefore(@Nonnull Instant timestamp) {
        return this.baseClientTokenStore.getRefreshTokensExpiringBefore(timestamp).stream().map(this::populateSecretFields).collect(Collectors.toList());
    }

    @Override
    @Nonnull
    public List<ClientTokenEntity> list() {
        return this.baseClientTokenStore.list().stream().map(this::populateSecretFields).collect(Collectors.toList());
    }

    @Override
    public boolean deleteByTokenValue(String tokenValue) {
        return (Boolean)this.activeObjects.executeInTransaction(() -> {
            if (this.isSecretServiceDisabled()) {
                return this.baseClientTokenStore.deleteByTokenValue(tokenValue);
            }
            logger.error("Warning: token deletion by value is not supported as Secret Service is enabled.");
            return false;
        });
    }

    private ClientToken sanitizeTokenFields(ClientToken clientToken) {
        String sanitizedRefreshToken = clientToken.getRefreshToken() != null ? "{ATL_SECURED}" : null;
        return ClientTokenImpl.builder().accessToken("{ATL_SECURED}").accessTokenExpiration(clientToken.getAccessTokenExpiration()).refreshToken(sanitizedRefreshToken).refreshTokenExpiration(clientToken.getRefreshTokenExpiration()).build();
    }

    private ClientTokenEntity sanitizeTokenFields(ClientTokenEntity clientTokenEntity) {
        String sanitizedRefreshToken = clientTokenEntity.getRefreshToken() != null ? "{ATL_SECURED}" : null;
        return clientTokenEntity.toBuilder().accessToken("{ATL_SECURED}").refreshToken(sanitizedRefreshToken).build();
    }

    private ClientTokenEntity populateSecretFields(ClientTokenEntity token, String accessToken, String refreshToken) {
        return token.toBuilder().accessToken(accessToken).refreshToken(refreshToken).build();
    }

    private ClientTokenEntity populateSecretFields(ClientTokenEntity token) {
        if (this.isSecretServiceDisabled()) {
            if ("{ATL_SECURED}".equals(token.getAccessToken())) {
                throw new SecretServiceException("Secret service is disabled, but client secret is marked as secured.");
            }
            if ("{ATL_SECURED}".equals(token.getRefreshToken())) {
                throw new SecretServiceException("Secret service is disabled, but client certificate is marked as secured.");
            }
            return token;
        }
        Optional maybeAccessToken = this.secretService.get(ClientPluginSecretServicePath.accessTokenPath(token.getId()));
        String refreshToken = this.secretService.get(ClientPluginSecretServicePath.refreshTokenPath(token.getId())).orElse(null);
        if (maybeAccessToken.isEmpty()) {
            logger.error("Failed to retrieve secrets from secret service for client configuration id [{}].", (Object)token.getId());
        }
        return this.populateSecretFields(token, maybeAccessToken.orElse(""), refreshToken);
    }

    private boolean isSecretServiceDisabled() {
        return SecretService.getState() == SecretServiceState.DISABLED;
    }
}

