/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.oauth2.provider.core.token;

import com.atlassian.oauth2.provider.api.token.access.AccessToken;
import com.atlassian.oauth2.provider.api.token.refresh.RefreshToken;
import com.atlassian.oauth2.provider.core.credentials.ClientCredentialsGenerator;
import com.atlassian.oauth2.provider.core.event.OAuth2ProviderEventPublisher;
import com.atlassian.oauth2.provider.core.properties.SystemProperty;
import com.atlassian.oauth2.provider.core.security.Hasher;
import com.atlassian.oauth2.provider.core.token.TokenService;
import com.atlassian.oauth2.provider.core.token.access.dao.AccessTokenDao;
import com.atlassian.oauth2.provider.core.token.access.dao.AccessTokenEntity;
import com.atlassian.oauth2.provider.core.token.access.exception.UserKeyNotFoundException;
import com.atlassian.oauth2.provider.core.token.refresh.dao.RefreshTokenDao;
import com.atlassian.oauth2.provider.core.token.refresh.dao.RefreshTokenEntity;
import com.atlassian.oauth2.scopes.api.Scope;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.sal.api.user.UserKey;
import com.atlassian.sal.api.user.UserManager;
import jakarta.annotation.Nonnull;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultTokenService
implements TokenService {
    private static final Logger logger = LoggerFactory.getLogger(DefaultTokenService.class);
    private final AccessTokenDao accessTokenDao;
    private final RefreshTokenDao refreshTokenDao;
    private final ClientCredentialsGenerator clientCredentialsGenerator;
    private final UserManager userManager;
    private final Clock clock;
    private final I18nResolver i18nResolver;
    private final Hasher hasher;
    private final OAuth2ProviderEventPublisher oAuth2ProviderEventPublisher;

    public DefaultTokenService(AccessTokenDao accessTokenDao, RefreshTokenDao refreshTokenDao, ClientCredentialsGenerator clientCredentialsGenerator, UserManager userManager, Clock clock, I18nResolver i18nResolver, Hasher hasher, OAuth2ProviderEventPublisher oAuth2ProviderEventPublisher) {
        this.accessTokenDao = accessTokenDao;
        this.refreshTokenDao = refreshTokenDao;
        this.clientCredentialsGenerator = clientCredentialsGenerator;
        this.userManager = userManager;
        this.clock = clock;
        this.i18nResolver = i18nResolver;
        this.hasher = hasher;
        this.oAuth2ProviderEventPublisher = oAuth2ProviderEventPublisher;
    }

    @Override
    @Nonnull
    public AccessToken createAccessToken(String clientId, String userKey, String authorizationCode, long authorizationDate, Set<Scope> scopes) {
        if (this.userManager.getUserProfile(new UserKey(userKey)) != null) {
            AccessTokenEntity accessTokenEntity = AccessTokenEntity.builder().id(this.clientCredentialsGenerator.generate(ClientCredentialsGenerator.Length.THIRTY_TWO)).clientId(clientId).userKey(userKey).authorizationCode(authorizationCode).scopes(scopes).authorizationDate(authorizationDate).createdAt(this.clock.millis()).grantTypeUsedToCreate("authorization_code").build();
            logger.debug("Creating an access token for client id [{}].", (Object)clientId);
            this.accessTokenDao.create(accessTokenEntity.toBuilder().id(this.hasher.hash(accessTokenEntity.getId())).build());
            return accessTokenEntity;
        }
        throw new UserKeyNotFoundException(this.i18nResolver.getText("oauth2.service.user.key.not.found"));
    }

    @Override
    @Nonnull
    public RefreshToken createRefreshToken(String clientId, String userKey, long authorizationDate, String accessTokenId, Set<Scope> scopes, String authorizationCode, int refreshCount) {
        if (this.userManager.getUserProfile(new UserKey(userKey)) != null) {
            RefreshTokenEntity refreshTokenEntity = RefreshTokenEntity.builder().id(this.clientCredentialsGenerator.generate(ClientCredentialsGenerator.Length.THIRTY_TWO)).clientId(clientId).userKey(userKey).accessTokenId(accessTokenId).scopes(scopes).authorizationCode(authorizationCode).refreshCount(refreshCount).authorizationDate(authorizationDate).createdAt(this.clock.millis()).build();
            logger.debug("Creating refresh token for client id [{}].", (Object)clientId);
            this.refreshTokenDao.create(refreshTokenEntity.toBuilder().id(this.hasher.hash(refreshTokenEntity.getId())).accessTokenId(this.hasher.hash(refreshTokenEntity.getAccessTokenId())).build());
            logger.debug("Removing oldest refresh token if limit exceeded for client id [{}] and user key [{}].", (Object)clientId, (Object)userKey);
            this.refreshTokenDao.removeOldestIfLimitExceeded(clientId, userKey, SystemProperty.MAX_REFRESH_TOKEN_LIMIT_PER_CLIENT_ID_AND_USER.getValue());
            this.oAuth2ProviderEventPublisher.publishTokenCreatedEvent(clientId, refreshTokenEntity);
            return refreshTokenEntity;
        }
        throw new UserKeyNotFoundException(this.i18nResolver.getText("oauth2.service.user.key.not.found"));
    }

    @Override
    public AccessToken createClientCredentialsAccessToken(String clientId, Set<Scope> scopes) {
        AccessTokenEntity accessTokenEntity = AccessTokenEntity.builder().id(this.clientCredentialsGenerator.generate(ClientCredentialsGenerator.Length.THIRTY_TWO)).clientId(clientId).scopes(scopes).createdAt(this.clock.millis()).userKey("").authorizationCode("").authorizationDate(this.clock.millis()).grantTypeUsedToCreate("client_credentials").build();
        logger.debug("Creating a client credentials access token for client id [{}].", (Object)clientId);
        this.accessTokenDao.create(accessTokenEntity.toBuilder().id(this.hasher.hash(accessTokenEntity.getId())).build());
        return accessTokenEntity;
    }

    @Override
    public void updateAccessTokenLastAccessed(String tokenId) {
        this.accessTokenDao.updateTokenLastAccessed(this.hasher.hash(tokenId));
    }

    @Override
    @Nonnull
    public Optional<AccessToken> findByAccessTokenId(String id) {
        return this.accessTokenDao.findByToken(this.hasher.hash(id));
    }

    @Override
    @Nonnull
    public Optional<RefreshToken> findByRefreshTokenId(String id) {
        return this.refreshTokenDao.findByTokenId(this.hasher.hash(id));
    }

    @Override
    @Nonnull
    public Optional<AccessToken> removeAccessTokenById(String tokenId) {
        return this.accessTokenDao.removeTokenById(this.hasher.hash(tokenId));
    }

    @Override
    @Nonnull
    public List<RefreshToken> removeTokensByAuthorizationCode(String authorizationCode) {
        List<RefreshToken> removedRefreshTokens = this.refreshTokenDao.removeAllByAuthorizationCode(authorizationCode);
        removedRefreshTokens.forEach(token -> {
            this.accessTokenDao.removeTokenById(token.getAccessTokenId());
            this.oAuth2ProviderEventPublisher.publishTokenRevokedEvent(token.getClientId(), token.getUserKey());
        });
        return removedRefreshTokens;
    }

    @Override
    @Nonnull
    public void removeByClientId(String clientId) {
        this.accessTokenDao.removeAllByClientId(clientId);
        this.refreshTokenDao.removeByClientId(clientId);
    }

    @Override
    @Nonnull
    public List<AccessToken> removeByUserKey(String userKey) {
        return this.accessTokenDao.removeAllByUserKey(userKey);
    }

    @Override
    public void removeExpiredAccessTokens(@Nonnull Duration expirationPeriod) {
        this.accessTokenDao.removeExpiredTokensAfter(expirationPeriod);
    }

    @Override
    public void removeExpiredRefreshTokens(@Nonnull Duration expirationPeriod) {
        this.refreshTokenDao.removeExpiredTokensAfter(expirationPeriod);
    }

    @Override
    @Nonnull
    public List<String> findUserKeysByClientId(String clientId) {
        return this.accessTokenDao.findUserKeysByClientId(clientId);
    }

    @Override
    @Nonnull
    public List<RefreshToken> findRefreshTokensForClientId(String clientId) {
        return this.refreshTokenDao.findByClientId(clientId);
    }

    @Override
    public List<AccessToken> findAccessTokensByUserKey(@Nonnull UserKey userKey) {
        return this.accessTokenDao.findByUserKey(userKey.getStringValue());
    }

    @Override
    public Optional<Instant> findClientLastAccessedDate(String clientId) {
        return this.accessTokenDao.findLastAccessedByClientId(clientId).filter(accessToken -> accessToken.getLastAccessed() != null && accessToken.getLastAccessed() > 0L).map(accessToken -> Instant.ofEpochMilli(accessToken.getLastAccessed()));
    }

    @Override
    public List<RefreshToken> findRefreshTokensByUserKey(@Nonnull UserKey userKey) {
        return this.refreshTokenDao.findByUserKey(userKey.getStringValue());
    }

    @Override
    public Optional<AccessToken> removeAccessTokenAssociatedWith(String refreshTokenJti) {
        String refreshTokenId = this.hasher.hash(refreshTokenJti);
        return this.removeAccessTokenByRefreshTokenId(refreshTokenId);
    }

    private Optional<AccessToken> removeAccessTokenByRefreshTokenId(String refreshTokenId) {
        Optional<AccessToken> accessToken = this.refreshTokenDao.findByTokenId(refreshTokenId).flatMap(refreshToken -> this.accessTokenDao.findByToken(refreshToken.getAccessTokenId()));
        this.refreshTokenDao.findByTokenId(refreshTokenId).ifPresent(actualRefreshToken -> this.accessTokenDao.removeTokenById(actualRefreshToken.getAccessTokenId()));
        return accessToken;
    }

    @Override
    public Optional<RefreshToken> removeRefreshTokenAssociatedWith(String accessTokenId) {
        return this.refreshTokenDao.removeByAccessTokenId(this.hasher.hash(accessTokenId)).map(token -> {
            this.oAuth2ProviderEventPublisher.publishTokenRevokedEvent(token.getClientId(), token.getUserKey());
            return token;
        });
    }

    @Override
    public boolean isAccessTokenValid(@Nonnull String clientId, @Nonnull String accessTokenId) {
        return this.findByAccessTokenId(accessTokenId).map(token -> token.getClientId().equals(clientId)).orElse(false);
    }

    @Override
    public boolean isRefreshTokenValid(@Nonnull String clientId, @Nonnull String refreshTokenId) {
        return this.findByRefreshTokenId(refreshTokenId).map(token -> token.getClientId().equals(clientId)).orElse(false);
    }

    @Override
    public Optional<RefreshToken> removeRefreshToken(String refreshTokenJti) {
        String refreshTokenId = this.hasher.hash(refreshTokenJti);
        return this.removeRefreshTokenById(refreshTokenId);
    }

    private Optional<RefreshToken> removeRefreshTokenById(String refreshTokenId) {
        return this.refreshTokenDao.removeByRefreshTokenId(refreshTokenId).map(token -> {
            this.oAuth2ProviderEventPublisher.publishTokenRevokedEvent(token.getClientId(), token.getUserKey());
            return token;
        });
    }

    @Override
    public void removeAllRefreshTokens() {
        UserKey user = this.userManager.getRemoteUserKey();
        if (!this.userManager.isSystemAdmin(user)) {
            if (logger.isDebugEnabled()) {
                logger.debug("User [{}] is not a system admin. Cannot revoke all refresh tokens", (Object)user);
            }
            return;
        }
        logger.info("Revoking all refresh tokens");
        this.refreshTokenDao.removeAll();
        this.accessTokenDao.removeAll();
        this.oAuth2ProviderEventPublisher.publishAllRefreshTokensRevokedEvent();
    }

    @Override
    public boolean isCodeRedeemed(String authorizationCode) {
        return !this.refreshTokenDao.findByAuthorizationCode(authorizationCode).isEmpty();
    }

    @Override
    public boolean removeCurrentUsersTokensById(@Nonnull String tokenId) {
        return this.removeTokensByAccessTokenId(tokenId) || this.removeTokensByRefreshTokenId(tokenId);
    }

    private boolean removeTokensByAccessTokenId(String tokenId) {
        return this.findByAccessTokenId(tokenId).filter(token -> this.canDeleteToken(token.getUserKey())).map(accessToken -> {
            this.removeRefreshTokenAssociatedWith(tokenId);
            return this.removeAccessTokenById(tokenId);
        }).isPresent();
    }

    private boolean removeTokensByRefreshTokenId(String tokenId) {
        return this.findByRefreshTokenId(tokenId).filter(token -> this.canDeleteToken(token.getUserKey())).map(refreshToken -> {
            this.removeRefreshToken(tokenId);
            return this.removeAccessTokenById(refreshToken.getAccessTokenId());
        }).isPresent();
    }

    private boolean canDeleteToken(String userKey) {
        return this.isTokenForCurrentUserOrIsCurrentUserSystemAdmin().test(userKey);
    }

    @Override
    public void removeCurrentUserRefreshTokens() {
        this.removeUserRefreshTokens(this.userManager.getRemoteUserKey());
    }

    @Override
    public void removeUserRefreshTokens(UserKey userKey) {
        boolean canDeleteTokens = this.isTokenForCurrentUserOrIsCurrentUserSystemAdmin().test(userKey.getStringValue());
        if (!canDeleteTokens) {
            logger.debug("User [{}] is not the owner of refresh tokens nor a system admin. Cannot revoke refresh tokens for user [{}]", (Object)this.userManager.getRemoteUserKey(), (Object)userKey);
            return;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Revoking all refresh tokens for user [{}]", (Object)userKey);
        }
        List<RefreshToken> refreshTokens = this.findRefreshTokensByUserKey(userKey);
        refreshTokens.forEach(refreshToken -> {
            this.removeAccessTokenByRefreshTokenId(refreshToken.getId());
            this.removeRefreshTokenById(refreshToken.getId());
        });
    }

    @Override
    public boolean removeTokensById(@Nonnull String tokenId) {
        Predicate<String> isTokenForCurrentUserOrIsCurrentUserSystemAdmin = this.isTokenForCurrentUserOrIsCurrentUserSystemAdmin();
        boolean removedTokens = this.accessTokenDao.findByToken(tokenId).filter(accessToken -> isTokenForCurrentUserOrIsCurrentUserSystemAdmin.test(accessToken.getUserKey())).map(accessToken -> {
            this.accessTokenDao.removeTokenById(tokenId);
            this.refreshTokenDao.removeByAccessTokenId(tokenId);
            this.oAuth2ProviderEventPublisher.publishTokenRevokedEvent(accessToken.getClientId(), accessToken.getUserKey());
            return true;
        }).orElse(false);
        if (!removedTokens) {
            removedTokens = this.refreshTokenDao.findByTokenId(tokenId).filter(refreshToken -> isTokenForCurrentUserOrIsCurrentUserSystemAdmin.test(refreshToken.getUserKey())).map(refreshToken -> {
                this.refreshTokenDao.removeByRefreshTokenId(tokenId);
                this.accessTokenDao.removeTokenById(refreshToken.getAccessTokenId());
                this.oAuth2ProviderEventPublisher.publishTokenRevokedEvent(refreshToken.getClientId(), refreshToken.getUserKey());
                return true;
            }).orElse(false);
        }
        return removedTokens;
    }

    private Predicate<String> isTokenForCurrentUserOrIsCurrentUserSystemAdmin() {
        UserKey currentLoggedInUserKey = this.userManager.getRemoteUserKey();
        boolean isSystemAdmin = currentLoggedInUserKey != null && this.userManager.isSystemAdmin(currentLoggedInUserKey);
        return userKey -> currentLoggedInUserKey != null && (isSystemAdmin || currentLoggedInUserKey.getStringValue().equals(userKey));
    }
}

