/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.plugins.authentication.tsv.service;

import com.atlassian.crowd.embedded.api.CrowdService;
import com.atlassian.crowd.exception.FailedAuthenticationException;
import com.atlassian.plugins.authentication.api.tsv.internal.exception.EnrollmentNotFoundException;
import com.atlassian.plugins.authentication.api.tsv.internal.service.AuthAuditService;
import com.atlassian.plugins.authentication.api.tsv.internal.service.RecoveryCodeService;
import com.atlassian.plugins.authentication.api.tsv.internal.service.VerificationMethod;
import com.atlassian.plugins.authentication.sso.web.CookieService;
import com.atlassian.plugins.authentication.tsv.exception.ElevationMeansIncorrectException;
import com.atlassian.plugins.authentication.tsv.exception.SessionElevationRequiredException;
import com.atlassian.plugins.authentication.tsv.model.ActionType;
import com.atlassian.plugins.authentication.tsv.model.ElevationMethod;
import com.atlassian.plugins.authentication.tsv.rest.filters.ratelimit.RateLimit;
import com.atlassian.plugins.authentication.tsv.service.EnrollmentService;
import com.atlassian.plugins.authentication.tsv.service.InternalRecoveryCodeService;
import com.atlassian.plugins.authentication.tsv.service.InternalTotpService;
import com.atlassian.plugins.rest.api.security.exception.AuthenticationRequiredException;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.sal.api.user.UserKey;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sal.api.user.UserProfile;
import com.google.common.primitives.Longs;
import io.atlassian.fugue.Either;
import jakarta.annotation.Nonnull;
import jakarta.inject.Inject;
import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.servlet.http.HttpSession;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ElevatedSessionService {
    private static final Logger log = LoggerFactory.getLogger(ElevatedSessionService.class);
    private static final Long ELEVATED_SESSION_VALIDITY = Long.getLong(ElevatedSessionService.class.getName() + ".ELEVATED_SESSION_VALIDITY", Duration.of(15L, ChronoUnit.MINUTES).toMillis());
    static final String ELEVATED_SESSION_TIMESTAMP_ATTRIBUTE = "com.atlassian.plugins.authentication.elevated-session-timestamp";
    public static final String ELEVATED_SESSION_COOKIE_NAME = "com.atlassian.plugins.authentication.elevated-session";
    private final Clock clock;
    private final CookieService cookieService;
    private final CrowdService crowdService;
    private final UserManager userManager;
    private final I18nResolver i18nResolver;
    private final InternalRecoveryCodeService recoveryCodeService;
    private final EnrollmentService enrollmentService;
    private final InternalTotpService totpService;
    private final AuthAuditService authAuditService;

    @Inject
    public ElevatedSessionService(Clock clock, CookieService cookieService, CrowdService crowdService, UserManager userManager, I18nResolver i18nResolver, InternalRecoveryCodeService recoveryCodeService, EnrollmentService enrollmentService, InternalTotpService totpService, AuthAuditService authAuditService) {
        this.clock = clock;
        this.cookieService = cookieService;
        this.crowdService = crowdService;
        this.userManager = userManager;
        this.i18nResolver = i18nResolver;
        this.recoveryCodeService = recoveryCodeService;
        this.enrollmentService = enrollmentService;
        this.totpService = totpService;
        this.authAuditService = authAuditService;
    }

    public void checkElevatedSession(ActionType actionType, HttpServletRequest request) throws SessionElevationRequiredException {
        boolean hasElevatedSession = this.hasElevatedSession(actionType, request);
        if (hasElevatedSession) {
            return;
        }
        String userKey = Optional.ofNullable(this.userManager.getRemoteUserKey()).map(UserKey::getStringValue).orElseThrow(AuthenticationRequiredException::new);
        ElevationMethod elevationMethod = this.enrollmentService.isAlreadyEnrolled(userKey) ? ElevationMethod.TOTP : ElevationMethod.PASSWORD;
        throw new SessionElevationRequiredException(elevationMethod);
    }

    @RateLimit(value=RateLimit.MethodType.PASSWORD)
    public void elevateSessionUsingPassword(ActionType actionType, String password, HttpServletRequest request, HttpServletResponse response) throws ElevationMeansIncorrectException {
        this.elevateSessionInternal(actionType, user -> this.authenticate(user.getUsername(), password), request, response);
    }

    public String elevateSessionUsingRecoveryCode(ActionType actionType, String recoveryCode, HttpServletRequest request, HttpServletResponse response) throws ElevationMeansIncorrectException {
        return (String)this.elevateSessionInternal(actionType, user -> this.verifyRecoveryCode((UserProfile)user, recoveryCode), request, response);
    }

    public void elevateSessionUsingTotpCode(ActionType actionType, String totpCode, HttpServletRequest request, HttpServletResponse response) throws ElevationMeansIncorrectException {
        this.elevateSessionInternal(actionType, user -> this.verifyTotpCode((UserProfile)user, totpCode), request, response);
    }

    public void removeElevatedSession(@Nonnull HttpServletResponse response, @Nonnull HttpSession session) {
        session.removeAttribute(ELEVATED_SESSION_TIMESTAMP_ATTRIBUTE);
        this.cookieService.removeCookie(ELEVATED_SESSION_COOKIE_NAME, response);
    }

    private static void checkSessionPresent(HttpSession session) {
        if (session == null) {
            log.error("Cannot elevate session as there's no session associated with the request");
            throw new IllegalStateException("Cannot elevate session as there's no session associated with the request");
        }
    }

    private boolean hasElevatedSession(ActionType actionType, HttpServletRequest request) {
        HttpSession session = request.getSession(false);
        if (session == null) {
            log.debug("User has no elevated session because there's not HTTP session associated with the request");
            return false;
        }
        Optional<UserProfile> maybeUser = this.getCurrentUser();
        if (maybeUser.isEmpty()) {
            log.debug("No user associated with the session");
            return false;
        }
        String username = maybeUser.get().getUsername();
        Optional<Instant> maybeElevatedSessionTimestamp = this.getElevatedSessionTimestamp(session);
        if (maybeElevatedSessionTimestamp.isEmpty()) {
            log.debug("User {} does not have an elevated session due to missing elevated session attribute", (Object)username);
            return false;
        }
        Instant elevatedSessionTimestamp = maybeElevatedSessionTimestamp.get();
        Boolean isElevatedSessionAttributeValid = this.isElevatedSessionAttributeValid(elevatedSessionTimestamp, username);
        Boolean elevatedSessionCookieValid = this.isElevatedSessionCookieValid(request, elevatedSessionTimestamp, username);
        return isElevatedSessionAttributeValid != false && elevatedSessionCookieValid != false;
    }

    private void elevateSession(HttpServletResponse response, HttpSession session, String username) {
        long elevatedSessionTimestamp = this.clock.millis();
        log.trace("Setting elevated session timestamp of {} for user {}", (Object)elevatedSessionTimestamp, (Object)username);
        session.setAttribute(ELEVATED_SESSION_TIMESTAMP_ATTRIBUTE, (Object)elevatedSessionTimestamp);
        Cookie cookie = new Cookie(ELEVATED_SESSION_COOKIE_NAME, String.valueOf(elevatedSessionTimestamp));
        cookie.setPath(this.cookieService.buildCookiePath());
        response.addCookie(cookie);
    }

    private Optional<Instant> getElevatedSessionTimestamp(HttpSession session) {
        Object elevatedSessionAttribute = session.getAttribute(ELEVATED_SESSION_TIMESTAMP_ATTRIBUTE);
        if (elevatedSessionAttribute == null) {
            return Optional.empty();
        }
        if (!(elevatedSessionAttribute instanceof Long)) {
            return Optional.empty();
        }
        return Optional.of(Instant.ofEpochMilli((Long)elevatedSessionAttribute));
    }

    private Either<ElevationError, Boolean> authenticate(String username, String password) {
        log.debug("Elevating session using credentials for user {}", (Object)username);
        try {
            if (this.enrollmentService.isAlreadyEnrolled(this.userManager.getRemoteUserKey().getStringValue())) {
                this.authAuditService.logTsvSessionElevationFailure(username, VerificationMethod.PASSWORD);
                return Either.left((Object)new ElevationError(String.format("User %s used password while being enrolled", username), "authentication.two-step-verification.elevate-permissions.already-enrolled"));
            }
            this.crowdService.authenticate(username, password);
            log.debug("User {} provided valid credentials", (Object)username);
            this.authAuditService.logTsvSessionElevationSuccess(username, VerificationMethod.PASSWORD);
            return Either.right((Object)true);
        }
        catch (FailedAuthenticationException e) {
            this.authAuditService.logTsvSessionElevationFailure(username, VerificationMethod.PASSWORD);
            return Either.left((Object)new ElevationError(String.format("User %s supplied invalid credentials during session elevation", username), "authentication.two-step-verification.elevate-permissions.incorrect-password"));
        }
    }

    private Either<ElevationError, String> verifyRecoveryCode(UserProfile userProfile, String recoveryCode) {
        log.debug("Elevating session using recovery code for user {}", (Object)userProfile.getUsername());
        return this.recoveryCodeService.useRecoveryCode(userProfile, recoveryCode).bimap(v -> switch (v) {
            default -> throw new MatchException(null, null);
            case RecoveryCodeService.RecoveryCodeConsumptionError.USER_NOT_ENROLLED -> new ElevationError(String.format("User %s is not enrolled to 2SV and cannot elevate session with a recovery code", userProfile.getUsername()), "authentication.two-step-verification.no.enrollment.found.error.message");
            case RecoveryCodeService.RecoveryCodeConsumptionError.RECOVERY_CODE_INVALID -> new ElevationError(String.format("User %s supplied invalid recovery code during session elevation", userProfile.getUsername()), "authentication.two-step-verification.recovery.error.message");
        }, v -> {
            log.debug("User {} provided valid recovery code", (Object)userProfile.getUsername());
            this.authAuditService.logTsvSessionElevationSuccess(userProfile.getUsername(), VerificationMethod.RECOVERY_KEY);
            return v;
        }).leftMap(e -> {
            this.authAuditService.logTsvSessionElevationFailure(userProfile.getUsername(), VerificationMethod.RECOVERY_KEY);
            return e;
        });
    }

    private Either<ElevationError, Boolean> verifyTotpCode(UserProfile userProfile, String totpCode) {
        boolean isTotpCodeValid;
        try {
            isTotpCodeValid = this.totpService.isTotpCodeValid(userProfile.getUserKey().getStringValue(), totpCode);
        }
        catch (EnrollmentNotFoundException e) {
            log.debug("User {} is not enrolled to 2SV and cannot elevate session with a TOTP code", (Object)userProfile.getUsername());
            return Either.left((Object)new ElevationError(String.format("User %s is not enrolled to 2SV and cannot elevate session with a TOTP code", userProfile.getUsername()), "authentication.two-step-verification.no.enrollment.found.error.message"));
        }
        if (!isTotpCodeValid) {
            this.authAuditService.logTsvSessionElevationFailure(userProfile.getUsername(), VerificationMethod.TOTP);
            return Either.left((Object)new ElevationError(String.format("User %s supplied invalid TOTP code during session elevation", userProfile.getUsername()), "authentication.two-step-verification.totp.error.message"));
        }
        log.debug("User {} provided valid TOTP code", (Object)userProfile.getUsername());
        this.authAuditService.logTsvSessionElevationSuccess(userProfile.getUsername(), VerificationMethod.TOTP);
        return Either.right((Object)true);
    }

    private Optional<UserProfile> getCurrentUser() {
        return Optional.ofNullable(this.userManager.getRemoteUser());
    }

    private Boolean isElevatedSessionCookieValid(HttpServletRequest request, Instant elevatedSessionTimestamp, String username) {
        Optional<Cookie> elevatedSessionCookie = this.cookieService.getCookieFromRequest(ELEVATED_SESSION_COOKIE_NAME, request);
        Boolean elevatedSessionCookieValid = elevatedSessionCookie.map(Cookie::getValue).map(Longs::tryParse).map(Instant::ofEpochMilli).map(timestamp -> Objects.equals(timestamp, elevatedSessionTimestamp)).orElse(false);
        log.debug("Elevated session cookie value for user {} {} session value", (Object)username, (Object)(elevatedSessionCookieValid != false ? "matches" : "does not match"));
        if (log.isTraceEnabled() && elevatedSessionCookie.isPresent()) {
            log.trace("Elevated session cookie value for user {} was {}", (Object)username, (Object)elevatedSessionCookie.get().getValue());
        }
        return elevatedSessionCookieValid;
    }

    private Boolean isElevatedSessionAttributeValid(Instant elevatedSessionTimestamp, String username) {
        Instant now = Instant.ofEpochMilli(this.clock.millis());
        Instant validityThreshold = elevatedSessionTimestamp.plus((long)ELEVATED_SESSION_VALIDITY, ChronoUnit.MILLIS);
        boolean elevatedSessionValid = validityThreshold.isAfter(now);
        log.debug("Elevated session attribute for user {} considered {} after comparing with current time", (Object)username, (Object)(elevatedSessionValid ? "valid" : "expired"));
        if (log.isTraceEnabled()) {
            log.trace("Compared elevation timestamp {} to current time of {} with {} being validity threshold for the session", new Object[]{elevatedSessionTimestamp, now, validityThreshold});
        }
        return elevatedSessionValid;
    }

    private <T> T elevateSessionInternal(ActionType actionType, Function<UserProfile, Either<ElevationError, T>> elevationMeansVerifier, HttpServletRequest request, HttpServletResponse response) throws ElevationMeansIncorrectException {
        HttpSession session = request.getSession(false);
        ElevatedSessionService.checkSessionPresent(session);
        Optional<UserProfile> maybeUser = this.getCurrentUser();
        if (maybeUser.isEmpty()) {
            log.error("Cannot elevate session as there's no user associated with the session");
            throw new IllegalStateException("Cannot elevate session as there's no user associated with the session");
        }
        UserProfile user = maybeUser.get();
        String username = user.getUsername();
        Either<ElevationError, T> elevationResult = elevationMeansVerifier.apply(user);
        if (elevationResult.isRight()) {
            log.debug("User {} passed the elevation check", (Object)username);
            if (!this.hasElevatedSession(actionType, request)) {
                log.debug("User {} does not have an elevated session, elevating", (Object)username);
                this.elevateSession(response, session, username);
            } else {
                log.debug("User {} already had an elevated session, no action will be taken", (Object)username);
            }
            return (T)elevationResult.right().get();
        }
        ElevationError elevationError = (ElevationError)elevationResult.left().get();
        log.info("{}, invalidating session elevation", (Object)elevationError.logMessage);
        session.removeAttribute(ELEVATED_SESSION_TIMESTAMP_ATTRIBUTE);
        this.cookieService.removeCookie(ELEVATED_SESSION_COOKIE_NAME, response);
        throw new ElevationMeansIncorrectException(this.i18nResolver.getText(elevationError.i18nErrorKey));
    }

    private record ElevationError(String logMessage, String i18nErrorKey) {
    }
}

