/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.tunnel.configuration;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.tunnel.audit.event.TunnelAuditEvent;
import com.atlassian.tunnel.audit.event.TunnelCreatedEvent;
import com.atlassian.tunnel.audit.event.TunnelDeletedEvent;
import com.atlassian.tunnel.audit.event.TunnelUpdatedEvent;
import com.atlassian.tunnel.configuration.Tunnel;
import com.atlassian.tunnel.configuration.TunnelStatus;
import com.atlassian.tunnel.configuration.dao.TunnelDao;
import com.atlassian.tunnel.configuration.exception.InvalidTunnelConfigurationException;
import com.atlassian.tunnel.monitoring.TunnelConnection;
import com.atlassian.tunnel.monitoring.TunnelConnectionService;
import com.atlassian.tunnel.process.TunnelProcessState;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TunnelService {
    private static final Logger log = LoggerFactory.getLogger(TunnelService.class);
    private final TunnelDao cachedTunnelDao;
    private final TunnelConnectionService tunnelConnectionService;
    private final EventPublisher eventPublisher;

    public TunnelService(TunnelDao cachedTunnelDao, TunnelConnectionService tunnelConnectionService, EventPublisher eventPublisher) {
        this.cachedTunnelDao = cachedTunnelDao;
        this.tunnelConnectionService = tunnelConnectionService;
        this.eventPublisher = eventPublisher;
    }

    public Optional<Tunnel> updateTunnel(Tunnel newTunnel) {
        if (this.hasTunnelChanged(newTunnel)) {
            this.validateTunnel(newTunnel);
            log.debug("Tunnel has changed, updating.");
            Optional<Tunnel> existingTunnel = this.cachedTunnelDao.getTunnel();
            if (!existingTunnel.isPresent()) {
                this.tunnelConnectionService.deleteTunnelConnections();
            }
            Tunnel savedTunnel = this.cachedTunnelDao.saveTunnel(newTunnel);
            this.eventPublisher.publish((Object)this.buildAuditEvent(existingTunnel, newTunnel));
            return Optional.of(savedTunnel);
        }
        return this.cachedTunnelDao.getTunnel();
    }

    private TunnelAuditEvent buildAuditEvent(Optional<Tunnel> previousTunnel, Tunnel newTunnel) {
        if (!previousTunnel.isPresent()) {
            return TunnelCreatedEvent.builder().tunnelName(newTunnel.getName()).build();
        }
        return TunnelUpdatedEvent.builder().tunnelName(newTunnel.getName()).build();
    }

    public Optional<Tunnel> getTunnel() {
        return this.cachedTunnelDao.getTunnel();
    }

    public TunnelStatus getTunnelStatus() {
        List<TunnelConnection> tunnelConnectionList = this.tunnelConnectionService.getTunnelConnections();
        Map<TunnelProcessState.Status, Long> statusCounts = tunnelConnectionList.stream().map(TunnelConnection::getStatus).collect(Collectors.groupingBy(Function.identity(), Collectors.counting()));
        long runningCount = this.getStatusCount(statusCounts, TunnelProcessState.Status.RUNNING);
        long authenticationFailedCount = this.getStatusCount(statusCounts, TunnelProcessState.Status.AUTHENTICATION_FAILED);
        long restartingCount = this.getStatusCount(statusCounts, TunnelProcessState.Status.RESTARTING);
        long notRunningCount = this.getStatusCount(statusCounts, TunnelProcessState.Status.NOT_RUNNING);
        long startingCount = this.getStatusCount(statusCounts, TunnelProcessState.Status.STARTING);
        if (tunnelConnectionList.isEmpty() || startingCount > 0L) {
            return TunnelStatus.CONNECTING;
        }
        if (runningCount > 0L) {
            if (authenticationFailedCount > 0L || restartingCount > 0L || notRunningCount > 0L) {
                return TunnelStatus.DEGRADED;
            }
            return TunnelStatus.CONNECTED;
        }
        if (authenticationFailedCount > 0L) {
            return TunnelStatus.ERROR;
        }
        return TunnelStatus.UNAVAILABLE;
    }

    public void deleteTunnel() {
        this.cachedTunnelDao.getTunnel().ifPresent(tunnel -> {
            this.cachedTunnelDao.deleteTunnel();
            this.eventPublisher.publish((Object)TunnelDeletedEvent.builder().tunnelName(tunnel.getName()).build());
        });
    }

    public void syncTunnel() {
        this.cachedTunnelDao.syncTunnel();
    }

    private long getStatusCount(Map<TunnelProcessState.Status, Long> statusCounts, TunnelProcessState.Status status) {
        return statusCounts.containsKey((Object)status) ? statusCounts.get((Object)status) : 0L;
    }

    private void validateTunnel(Tunnel tunnel) {
        if (StringUtils.isBlank((CharSequence)tunnel.getName())) {
            throw new InvalidTunnelConfigurationException(InvalidTunnelConfigurationException.Reason.INVALID, "Missing connection name");
        }
        if (StringUtils.isBlank((CharSequence)tunnel.getCloudOrgName())) {
            throw new InvalidTunnelConfigurationException(InvalidTunnelConfigurationException.Reason.INVALID, "Missing cloud org name");
        }
        if (!this.isValidUrl(tunnel.getCloudPageUrl())) {
            throw new InvalidTunnelConfigurationException(InvalidTunnelConfigurationException.Reason.INVALID, "Invalid cloud page URL");
        }
        if (StringUtils.isBlank((CharSequence)tunnel.getToken())) {
            throw new InvalidTunnelConfigurationException(InvalidTunnelConfigurationException.Reason.INVALID, "Missing token");
        }
        if (StringUtils.isBlank((CharSequence)tunnel.getUpstreamDnsUrl())) {
            throw new InvalidTunnelConfigurationException(InvalidTunnelConfigurationException.Reason.INVALID, "Missing upstream DNS URL");
        }
    }

    private boolean isValidUrl(String cloudPageUrl) {
        if (StringUtils.isBlank((CharSequence)cloudPageUrl)) {
            return false;
        }
        try {
            URL url = new URL(cloudPageUrl);
            return url.getProtocol().equalsIgnoreCase("http") || url.getProtocol().equalsIgnoreCase("https");
        }
        catch (MalformedURLException e) {
            return false;
        }
    }

    private boolean hasTunnelChanged(Tunnel tunnel) {
        return this.cachedTunnelDao.getTunnel().map(cachedTunnel -> !cachedTunnel.equals(tunnel)).orElse(true);
    }
}

