/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.confluence.impl.system.runner;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.confluence.api.model.journal.JournalIdentifier;
import com.atlassian.confluence.cluster.ClusterManager;
import com.atlassian.confluence.cluster.ClusterNodeInformation;
import com.atlassian.confluence.event.events.admin.RestoreIndexSnapshotStartedEvent;
import com.atlassian.confluence.impl.journal.JournalStateStore;
import com.atlassian.confluence.impl.system.MaintenanceTaskExecutionException;
import com.atlassian.confluence.impl.system.SystemMaintenanceTaskRegistry;
import com.atlassian.confluence.impl.system.runner.SystemMaintenanceTaskRunner;
import com.atlassian.confluence.impl.system.task.RestoreIndexSnapshotMaintenanceTask;
import com.atlassian.confluence.impl.system.task.SystemMaintenanceTaskType;
import com.atlassian.confluence.index.status.ReIndexJob;
import com.atlassian.confluence.index.status.ReIndexJobManager;
import com.atlassian.confluence.internal.index.Index;
import com.atlassian.confluence.internal.index.event.IndexSnapshotRestorationFailedEvent;
import com.atlassian.confluence.internal.index.event.IndexSnapshotRestorationSkippedEvent;
import com.atlassian.confluence.internal.index.event.IndexSnapshotRestoredSuccessfullyEvent;
import com.atlassian.confluence.internal.index.lucene.snapshot.IndexSnapshotError;
import com.atlassian.confluence.internal.index.lucene.snapshot.LuceneIndexSnapshot;
import com.atlassian.confluence.internal.index.lucene.snapshot.LuceneIndexSnapshotException;
import com.atlassian.confluence.internal.index.lucene.snapshot.LuceneIndexSnapshotManager;
import com.atlassian.confluence.search.SearchPlatformConfig;
import com.atlassian.event.api.EventPublisher;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.PreDestroy;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RestoreIndexSnapshotMaintenanceTaskRunner
implements SystemMaintenanceTaskRunner<RestoreIndexSnapshotMaintenanceTask> {
    static final long SNAPSHOT_CREATION_MAX_WAIT_TIME_MS = 60000L;
    private static final Logger log = LoggerFactory.getLogger(RestoreIndexSnapshotMaintenanceTaskRunner.class);
    private static final String SEMAPHORE_NAME = "snapshot_restoration_semaphore";
    @VisibleForTesting
    static final String MAX_LOCK_ATTEMPTS_PROP = "node.reindex.propagation.max.lock.attempts";
    @VisibleForTesting
    static final String PARALLEL_PROPAGATION_PERMITS_PROP = "node.reindex.propagation.parallel.permits";
    private static final int MAX_LOCK_ATTEMPTS = Integer.getInteger("node.reindex.propagation.max.lock.attempts", 10000);
    private static final int PARALLEL_PROPAGATION_PERMITS = Integer.getInteger("node.reindex.propagation.parallel.permits", 3);
    private static final long LOCK_WAITING_SECS = 100L;
    private final SystemMaintenanceTaskRegistry registry;
    private final LuceneIndexSnapshotManager luceneIndexSnapshotManager;
    private final ReIndexJobManager reIndexJobManager;
    private final ClusterManager clusterManager;
    private final EventPublisher eventPublisher;
    private final JournalStateStore journalStateStore;
    private final SearchPlatformConfig searchPlatformConfig;

    public RestoreIndexSnapshotMaintenanceTaskRunner(SystemMaintenanceTaskRegistry registry, LuceneIndexSnapshotManager luceneIndexSnapshotManager, ReIndexJobManager reIndexJobManager, ClusterManager clusterManager, EventPublisher eventPublisher, JournalStateStore journalStateStore, SearchPlatformConfig searchPlatformConfig) {
        this.registry = Objects.requireNonNull(registry);
        this.luceneIndexSnapshotManager = Objects.requireNonNull(luceneIndexSnapshotManager);
        this.reIndexJobManager = Objects.requireNonNull(reIndexJobManager);
        this.clusterManager = Objects.requireNonNull(clusterManager);
        this.eventPublisher = Objects.requireNonNull(eventPublisher);
        this.searchPlatformConfig = Objects.requireNonNull(searchPlatformConfig);
        this.journalStateStore = Objects.requireNonNull(journalStateStore);
    }

    @PostConstruct
    public void register() {
        this.registry.register(SystemMaintenanceTaskType.RESTORE_INDEX_SNAPSHOT, this);
    }

    @PreDestroy
    public void unregister() {
        this.registry.unregister(SystemMaintenanceTaskType.RESTORE_INDEX_SNAPSHOT);
    }

    @Override
    public void execute(RestoreIndexSnapshotMaintenanceTask task) throws MaintenanceTaskExecutionException {
        Optional<ReIndexJob> currentJobOptional = this.reIndexJobManager.getRunningOrMostRecentReIndex();
        String currentJobId = currentJobOptional.map(ReIndexJob::getId).orElse(null);
        if (this.shouldSkip(task)) {
            this.eventPublisher.publish((Object)new IndexSnapshotRestorationSkippedEvent(currentJobOptional.map(ReIndexJob::getId).orElse(null), this.getCurrentNodeId()));
            return;
        }
        if (!this.shouldRestoreSnapshot(task, currentJobOptional)) {
            return;
        }
        this.eventPublisher.publish((Object)new RestoreIndexSnapshotStartedEvent());
        boolean permitted = false;
        int attemptsRemaining = MAX_LOCK_ATTEMPTS;
        Instant startTime = Instant.now();
        while (!permitted && attemptsRemaining > 0) {
            --attemptsRemaining;
            try {
                if (!this.clusterManager.tryAcquire(SEMAPHORE_NAME, PARALLEL_PROPAGATION_PERMITS, 100L, TimeUnit.SECONDS)) continue;
                permitted = true;
                try {
                    this.doRestore(task.getIndexSnapshots());
                    this.eventPublisher.publish((Object)new IndexSnapshotRestoredSuccessfullyEvent(currentJobId, this.getCurrentNodeId(), startTime.until(Instant.now(), ChronoUnit.SECONDS)));
                }
                catch (Exception e) {
                    IndexSnapshotError error = e instanceof LuceneIndexSnapshotException ? ((LuceneIndexSnapshotException)e).getError() : IndexSnapshotError.UNKNOWN;
                    this.eventPublisher.publish((Object)new IndexSnapshotRestorationFailedEvent(currentJobId, this.getCurrentNodeId(), startTime.until(Instant.now(), ChronoUnit.SECONDS), error));
                    throw e;
                }
                finally {
                    log.info("releasing the semaphore");
                    this.clusterManager.release(SEMAPHORE_NAME);
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new MaintenanceTaskExecutionException("This thread was interrupted while waiting to perform an index replacement. The operation will be retried later", e);
            }
        }
        if (!permitted) {
            this.eventPublisher.publish((Object)new IndexSnapshotRestorationFailedEvent(currentJobId, this.getCurrentNodeId(), startTime.until(Instant.now(), ChronoUnit.SECONDS), IndexSnapshotError.UNKNOWN));
            String exceptionMessage = String.format("This node was unable to obtain a permit to update its index. The index on this node has not been updated. Consider increasing the %s property to increase the number of attempts or %s property to increase the permits for more nodes to process this task simultaneously.", MAX_LOCK_ATTEMPTS_PROP, PARALLEL_PROPAGATION_PERMITS_PROP);
            throw new RuntimeException(exceptionMessage);
        }
    }

    @VisibleForTesting
    boolean shouldRestoreSnapshot(RestoreIndexSnapshotMaintenanceTask task, Optional<ReIndexJob> currentJobOptional) {
        ClusterNodeInformation thisNodeInformation = this.clusterManager.getThisNodeInformation();
        if (!this.clusterManager.isClustered() || thisNodeInformation == null) {
            log.warn("Index propagation is a DC-only feature");
            return false;
        }
        String currentNodeId = thisNodeInformation.getAnonymizedNodeIdentifier();
        if (currentNodeId.equals(task.getSourceNodeId())) {
            log.info("Index snapshot will not be restored on the origin node. Skipping node: '{}'", (Object)currentNodeId);
            return false;
        }
        if (task.getIndexSnapshots() == null || task.getIndexSnapshots().isEmpty()) {
            log.info("Task has no index snapshots, skipping task");
            return false;
        }
        if (currentJobOptional.isEmpty()) {
            log.info("current reindex job is empty, skipping task");
            return false;
        }
        ReIndexJob currentJob = currentJobOptional.get();
        if (currentJob.getIndexJournalEntryIdMap() == null || currentJob.getIndexJournalEntryIdMap().isEmpty()) {
            log.info("current reindex job has no index journal id mapping, skipping task");
            return false;
        }
        boolean indexSnapshotMatched = this.snapshotsMatch(task.getIndexSnapshots(), currentJob.getIndexJournalEntryIdMap());
        if (!indexSnapshotMatched) {
            log.info("index snapshots of task and current job do not match, skipping task");
            return false;
        }
        return this.isIndexJournalEntryIdGreaterThanRecoveredEntryId(task);
    }

    private boolean isIndexJournalEntryIdGreaterThanRecoveredEntryId(RestoreIndexSnapshotMaintenanceTask task) {
        for (RestoreIndexSnapshotMaintenanceTask.IndexSnapshot indexSnapshot : task.getIndexSnapshots()) {
            JournalIdentifier newJournalIdentifier = new JournalIdentifier("recovered_" + indexSnapshot.getIndex().getJournalIdentifier().getJournalName());
            long recoveredEntryId = this.journalStateStore.getMostRecentId(newJournalIdentifier);
            if (indexSnapshot.getJournalId() >= recoveredEntryId) continue;
            log.warn("index snapshot journal entry id is lesser than recovered entry id. Skipping task");
            return false;
        }
        return true;
    }

    private boolean snapshotsMatch(Collection<RestoreIndexSnapshotMaintenanceTask.IndexSnapshot> taskSnapshots, Map<String, Long> indexJournalIdMap) {
        if (taskSnapshots.size() != indexJournalIdMap.size()) {
            log.warn("size of task snapshots does not match index journal id mapping, skipping task");
            return false;
        }
        return taskSnapshots.stream().allMatch(taskSnapshot -> {
            if (taskSnapshot == null) {
                log.warn("taskSnapshot is null, skipping task");
                return false;
            }
            Index index = taskSnapshot.getIndex();
            Long snapshotJournalId = taskSnapshot.getJournalId();
            Long mapJournalId = (Long)indexJournalIdMap.get(index.name());
            return mapJournalId != null && snapshotJournalId >= mapJournalId;
        });
    }

    private String getCurrentNodeId() {
        ClusterNodeInformation thisNodeInformation = this.clusterManager.getThisNodeInformation();
        return thisNodeInformation != null ? thisNodeInformation.getAnonymizedNodeIdentifier() : "";
    }

    private void doRestore(Collection<RestoreIndexSnapshotMaintenanceTask.IndexSnapshot> indexSnapshots) throws MaintenanceTaskExecutionException {
        log.info("Restoring index snapshots");
        for (RestoreIndexSnapshotMaintenanceTask.IndexSnapshot indexSnapshot : indexSnapshots) {
            Optional<LuceneIndexSnapshot> maybeSnapshot;
            try {
                maybeSnapshot = this.luceneIndexSnapshotManager.find(indexSnapshot.getIndex().getJournalIdentifier(), indexSnapshot.getJournalId(), 60000L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new MaintenanceTaskExecutionException(String.format("Interrupted while waiting for index snapshot %s : %s", new Object[]{indexSnapshot.getIndex(), indexSnapshot.getJournalId()}));
            }
            if (maybeSnapshot.isEmpty()) {
                throw new LuceneIndexSnapshotException(String.format("No snapshot found in shared home for index %s with journal entry id %s.", indexSnapshot.getIndex().name(), indexSnapshot.getJournalId()), IndexSnapshotError.SNAPSHOT_NOT_EXIST);
            }
            this.luceneIndexSnapshotManager.restore(maybeSnapshot.get());
            log.info("Index snapshot {} has been restored", (Object)maybeSnapshot.get());
        }
        log.info("All index snapshots have been restored successfully");
    }

    private boolean shouldSkip(RestoreIndexSnapshotMaintenanceTask task) {
        if (this.searchPlatformConfig.isSharedIndex()) {
            log.warn("This node is configured with a shared index - skipping snapshot restoration triggered by {}", (Object)task.getSourceNodeId());
            return true;
        }
        return false;
    }
}

