/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.applinks.internal.uniconn;

import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.internal.uniconn.BackboneConnectionHealthcheckResult;
import com.atlassian.applinks.internal.uniconn.BackboneConnectionHealthcheckService;
import com.atlassian.applinks.internal.uniconn.BackboneConnectionService;
import com.atlassian.applinks.internal.uniconn.UniconnApplinkProperties;
import com.atlassian.applinks.internal.uniconn.domain.BackboneConnectionStatus;
import java.util.List;
import java.util.Objects;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class BackboneConnectionMonitoringService {
    private static final Logger LOG = LoggerFactory.getLogger(BackboneConnectionMonitoringService.class);
    private final BackboneConnectionService backboneConnectionService;
    private final BackboneConnectionHealthcheckService healthcheckService;

    @Autowired
    public BackboneConnectionMonitoringService(BackboneConnectionService backboneConnectionService, BackboneConnectionHealthcheckService healthcheckService) {
        this.backboneConnectionService = Objects.requireNonNull(backboneConnectionService, "backboneConnectionService");
        this.healthcheckService = Objects.requireNonNull(healthcheckService, "healthcheckService");
    }

    public void updateConnectionStatusForBackboneConnections() {
        LOG.trace("Checking backbone connections status");
        List<ApplicationLink> backboneConnections = this.backboneConnectionService.getApplinks();
        if (backboneConnections.isEmpty()) {
            LOG.debug("No backbone connections to check");
            return;
        }
        for (ApplicationLink backboneConnection : backboneConnections) {
            try {
                BackboneConnectionStatus newStatus;
                BackboneConnectionStatus currentStatus = this.getCurrentStatus(backboneConnection);
                if (Objects.equals((Object)currentStatus, (Object)(newStatus = this.determineNewStatus(backboneConnection, currentStatus)))) continue;
                this.backboneConnectionService.updateBackboneConnectionStatus(backboneConnection, newStatus);
                LOG.debug("Successfully updated backbone connection status from {} to {} for application link {}", new Object[]{currentStatus, newStatus, backboneConnection.getId()});
            }
            catch (Exception e) {
                LOG.error("Error updating status for backbone connection with id: {}", (Object)backboneConnection.getId(), (Object)e);
            }
        }
    }

    private BackboneConnectionStatus determineNewStatus(ApplicationLink backboneConnection, BackboneConnectionStatus currentStatus) {
        if (this.isPermanentlyDisabled(currentStatus)) {
            return currentStatus;
        }
        if (this.areCredentialsExpired(backboneConnection)) {
            return BackboneConnectionStatus.EXPIRED;
        }
        BackboneConnectionHealthcheckResult healthcheckResult = this.healthcheckService.checkConnectionHealthcheck(backboneConnection);
        this.updateCredentialsExpiryFromHealthcheck(backboneConnection, healthcheckResult);
        return this.mapHealthcheckResultToStatus(healthcheckResult, currentStatus);
    }

    private BackboneConnectionStatus mapHealthcheckResultToStatus(BackboneConnectionHealthcheckResult result, BackboneConnectionStatus currentStatus) {
        switch (result.getStatus()) {
            case HEALTHY: {
                return BackboneConnectionStatus.CONNECTED;
            }
            case AUTH_FAILURE: 
            case NETWORK_FAILURE: 
            case UNKNOWN: {
                return BackboneConnectionStatus.INTERRUPTED;
            }
        }
        return currentStatus;
    }

    private boolean areCredentialsExpired(ApplicationLink backboneConnection) {
        Object expiryObject = backboneConnection.getProperty(UniconnApplinkProperties.CREDENTIALS_EXPIRY.key());
        if (expiryObject == null) {
            return false;
        }
        String expiryProperty = expiryObject.toString();
        if (StringUtils.isBlank((CharSequence)expiryProperty)) {
            return false;
        }
        try {
            boolean isExpired;
            long expiryTime = Long.parseLong(expiryProperty);
            boolean bl = isExpired = System.currentTimeMillis() > expiryTime;
            if (isExpired) {
                LOG.debug("Credentials expired for backbone connection {}: expiry={}, current={}", new Object[]{backboneConnection.getId(), expiryTime, System.currentTimeMillis()});
            }
            return isExpired;
        }
        catch (NumberFormatException e) {
            LOG.warn("Invalid credentials expiry format for backbone connection {}: {}", (Object)backboneConnection.getId(), (Object)expiryProperty);
            return false;
        }
    }

    private BackboneConnectionStatus getCurrentStatus(ApplicationLink backboneConnection) {
        Object statusObject = backboneConnection.getProperty(UniconnApplinkProperties.CONNECTION_STATUS.key());
        if (statusObject == null || statusObject.toString().trim().isEmpty()) {
            return BackboneConnectionStatus.CREATED;
        }
        try {
            return BackboneConnectionStatus.valueOf(statusObject.toString());
        }
        catch (IllegalArgumentException e) {
            LOG.warn("Invalid connection status for backbone connection {}: {}", (Object)backboneConnection.getId(), (Object)statusObject.toString());
            return BackboneConnectionStatus.CREATED;
        }
    }

    private boolean isPermanentlyDisabled(BackboneConnectionStatus status) {
        return status == BackboneConnectionStatus.DISCONNECTED;
    }

    private void updateCredentialsExpiryFromHealthcheck(ApplicationLink backboneConnection, BackboneConnectionHealthcheckResult healthcheckResult) {
        if (healthcheckResult.getTokenExpirationTime().isPresent()) {
            Long tokenExpirationTime = healthcheckResult.getTokenExpirationTime().get();
            try {
                backboneConnection.putProperty(UniconnApplinkProperties.CREDENTIALS_EXPIRY.key(), (Object)tokenExpirationTime.toString());
                LOG.debug("Updated credentials expiry time to {} for backbone connection {}", (Object)tokenExpirationTime, (Object)backboneConnection.getId());
            }
            catch (Exception e) {
                LOG.warn("Failed to update credentials expiry time for backbone connection {}: {}", (Object)backboneConnection.getId(), (Object)e.getMessage());
            }
        }
    }
}

