/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.secrets.service;

import com.atlassian.secrets.api.SecretServiceException;
import com.atlassian.secrets.api.SecretServiceType;
import com.atlassian.secrets.service.SecretServiceParams;
import com.atlassian.secrets.service.aes.AESEncryptionBackend;
import com.atlassian.secrets.service.aes.DefaultAESBackendGenerator;
import com.atlassian.secrets.service.config.BackendConfig;
import com.atlassian.secrets.service.config.SecretServiceConfig;
import com.atlassian.secrets.service.file.FileWriteRequest;
import com.fasterxml.jackson.core.JacksonException;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Collections;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecretConfigManager {
    public static final boolean isPosixFilesystem = FileSystems.getDefault().supportedFileAttributeViews().contains("posix");
    private static final YAMLFactory yamlFactory = new YAMLFactory().disable(YAMLGenerator.Feature.WRITE_DOC_START_MARKER);
    private static final ObjectMapper objectMapper = new ObjectMapper((JsonFactory)yamlFactory).enable(new JsonParser.Feature[]{JsonParser.Feature.STRICT_DUPLICATE_DETECTION});
    private static final Logger log = LoggerFactory.getLogger(SecretConfigManager.class);

    private SecretConfigManager() {
    }

    public static SecretServiceConfig readConfigFile(Path configFile) throws SecretServiceException {
        try {
            return (SecretServiceConfig)objectMapper.readValue(Files.readAllBytes(configFile), SecretServiceConfig.class);
        }
        catch (JacksonException e) {
            String exceptionMessage = e.getOriginalMessage();
            if (exceptionMessage.toLowerCase().contains("duplicate field")) {
                String errorMessage = String.format("%s is invalid: %s", configFile.getFileName(), exceptionMessage);
                log.error(errorMessage);
                throw new SecretServiceException(errorMessage);
            }
            String errorMessage = String.format("Secret service configuration file is invalid, review the %s file.", configFile.getFileName());
            log.error(errorMessage);
            throw new SecretServiceException(errorMessage);
        }
        catch (Exception e) {
            String errorMessage = "Problem when reading secret service configuration.";
            log.error(errorMessage, (Throwable)e);
            throw new SecretServiceException(errorMessage, (Throwable)e);
        }
    }

    static Path maybeGenerateDefaultConfigFile(SecretServiceParams params) throws SecretServiceException {
        try {
            Path secretConfigFilePath = params.getSecretsConfigFile();
            Path secretKeyDirPath = params.getKeyDirectory();
            AtomicBoolean newConfigCreated = new AtomicBoolean(false);
            AESEncryptionBackend encryptionBackend = new DefaultAESBackendGenerator().generate(secretKeyDirPath);
            LinkedHashMap<String, BackendConfig> backends = new LinkedHashMap<String, BackendConfig>();
            backends.put(encryptionBackend.getId(), encryptionBackend.getBackendConfig());
            SecretServiceConfig secretServiceConfig = new SecretServiceConfig(encryptionBackend.getId(), Collections.unmodifiableMap(backends));
            FileWriteRequest request = new FileWriteRequest(secretConfigFilePath, () -> {
                try {
                    return objectMapper.writeValueAsString((Object)secretServiceConfig);
                }
                catch (JsonProcessingException e) {
                    throw new SecretServiceException((Throwable)e);
                }
            }, true, newConfigCreated::set);
            params.getFileWriter().write(request);
            if (newConfigCreated.get()) {
                SecretConfigManager.applyReadOnlyPermissionsTo(isPosixFilesystem, secretConfigFilePath);
            }
            log.debug("Secret config file does not exist, generated default config in {}.", (Object)request.file());
            return secretConfigFilePath;
        }
        catch (Exception e) {
            String errorMessage = "Problem when generating a default secret service configuration.";
            log.error(errorMessage);
            throw new SecretServiceException(errorMessage, (Throwable)e);
        }
    }

    public static BackendConfig convertToBackendConfig(SecretServiceType type, Object backendConfigPojo) {
        return new BackendConfig(type.toString(), (Map)objectMapper.convertValue(backendConfigPojo, (TypeReference)new TypeReference<Map<String, Object>>(){}));
    }

    public static void applyReadOnlyPermissionsTo(boolean isPosixFilesystem, Path ... canonicalFilePaths) {
        if (isPosixFilesystem) {
            for (Path path : canonicalFilePaths) {
                EnumSet<PosixFilePermission> permissions = EnumSet.of(PosixFilePermission.OWNER_READ);
                try {
                    Files.setPosixFilePermissions(path, permissions);
                }
                catch (Exception e) {
                    log.error(String.format("Error applying read only permissions to file [%s]. To keep things secure update this file manually with read only permissions.", path), (Throwable)e);
                }
            }
        }
    }

    public static void updateConfigFile(SecretServiceParams params, SecretServiceConfig newConfig) {
        try {
            FileWriteRequest request = new FileWriteRequest(params.getSecretsConfigFile(), () -> {
                try {
                    return objectMapper.writeValueAsString((Object)newConfig);
                }
                catch (JsonProcessingException e) {
                    throw new SecretServiceException((Throwable)e);
                }
            }, false, ignored -> {});
            params.getFileWriter().write(request);
            SecretConfigManager.applyReadOnlyPermissionsTo(isPosixFilesystem, params.getSecretsConfigFile());
        }
        catch (Exception exception) {
            String errorMessage = "There was a problem writing Secret Service configuration file.";
            log.error(errorMessage);
            throw new SecretServiceException(errorMessage, (Throwable)exception);
        }
    }
}

