/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.secrets.service.aes;

import com.atlassian.secrets.api.SealedSecret;
import com.atlassian.secrets.api.SecretServiceException;
import com.atlassian.secrets.api.SecretServiceType;
import com.atlassian.secrets.service.EncryptionBasedSecret;
import com.atlassian.secrets.service.SecretConfigManager;
import com.atlassian.secrets.service.SecretServiceBackend;
import com.atlassian.secrets.service.aes.AESConfig;
import com.atlassian.secrets.service.aes.SecurityProvider;
import com.atlassian.secrets.service.config.BackendConfig;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import java.util.Arrays;
import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

public class AESEncryptionBackend
implements SecretServiceBackend {
    public static final String AES_ALGORITHM = "AES";
    public static final String ALGORITHM_SPECIFICATION = "AES/GCM/NoPadding";
    public static final int IV_LENGTH = 12;
    public static final int MAX_AES_KEY_SIZE = 256;
    public static final int TAG_LENGTH = 128;
    private static final int MIN_KEY_BYTES = 32;
    private static final Logger log = LoggerFactory.getLogger(AESEncryptionBackend.class);
    protected final String backendId;
    private final AESConfig config;
    private final SecretKey key;
    private final byte[] keyHash;
    private final ThreadLocal<Cipher> cipherThreadLocal;
    private final GeneratedIv generatedIv;

    public AESEncryptionBackend(String backendId, AESConfig config) throws SecretServiceException {
        this(backendId, config, AESEncryptionBackend::generateIV);
    }

    AESEncryptionBackend(String backendId, AESConfig config, GeneratedIv generatedIv) throws SecretServiceException {
        this.backendId = backendId;
        this.config = config;
        try {
            this.generatedIv = generatedIv;
            byte[] keyBytes = Files.readAllBytes(config.getKey());
            if (keyBytes.length < 32) {
                throw new SecretServiceException(String.format("The provided key size is less than %d bits, use more secure key", 256));
            }
            this.key = new SecretKeySpec(keyBytes, AES_ALGORITHM);
            this.keyHash = AESEncryptionBackend.calculateSha256(keyBytes);
            this.cipherThreadLocal = ThreadLocal.withInitial(this::buildCipher);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new SecretServiceException((Throwable)e);
        }
    }

    private Cipher buildCipher() {
        try {
            return Cipher.getInstance(ALGORITHM_SPECIFICATION, SecurityProvider.get());
        }
        catch (Exception e) {
            throw new SecretServiceException("Error creating Cipher", (Throwable)e);
        }
    }

    @Override
    public SealedSecret seal(String identifier, String plainTextToSecure) throws SecretServiceException {
        Assert.hasText((String)identifier, (String)"Secret identifier must not be empty.");
        try {
            byte[] iv = this.generatedIv.get();
            Cipher cipher = this.cipherThreadLocal.get();
            GCMParameterSpec algoParams = new GCMParameterSpec(128, iv);
            cipher.init(1, (Key)this.key, algoParams);
            byte[] encryptedData = cipher.doFinal(plainTextToSecure.getBytes(StandardCharsets.UTF_8));
            return new EncryptionBasedSecret(identifier, this.backendId, ALGORITHM_SPECIFICATION, encryptedData, iv, this.keyHash);
        }
        catch (GeneralSecurityException e) {
            throw new SecretServiceException("Error when sealing the secret", (Throwable)e);
        }
    }

    @Override
    public String unseal(SealedSecret sealedSecret) throws SecretServiceException {
        if (!(sealedSecret instanceof EncryptionBasedSecret)) {
            throw new SecretServiceException("Expecting encrypted secret but secret identifier was passed in");
        }
        EncryptionBasedSecret encryptionBasedSecret = (EncryptionBasedSecret)sealedSecret;
        if (!Arrays.equals(encryptionBasedSecret.getKeyHash(), this.keyHash)) {
            throw new SecretServiceException("This secret cannot be decrypted with the configured encryption key");
        }
        if (!ALGORITHM_SPECIFICATION.equals(encryptionBasedSecret.getAlgorithm())) {
            throw new SecretServiceException(String.format("This secret cannot be decrypted with the %s algorithm", ALGORITHM_SPECIFICATION));
        }
        try {
            byte[] encryptedData = encryptionBasedSecret.getEncryptedData();
            byte[] iv = encryptionBasedSecret.getIv();
            Cipher cipher = this.cipherThreadLocal.get();
            GCMParameterSpec algoParams = new GCMParameterSpec(128, iv);
            cipher.init(2, (Key)this.key, algoParams);
            byte[] decryptedText = cipher.doFinal(encryptedData);
            return new String(decryptedText, StandardCharsets.UTF_8);
        }
        catch (GeneralSecurityException e) {
            throw new SecretServiceException("Error when unsealing the secret", (Throwable)e);
        }
    }

    @Override
    public void delete(String identifier) throws SecretServiceException {
    }

    @Override
    public SecretServiceType getType() {
        return SecretServiceType.AES;
    }

    @Override
    public String getId() {
        return this.backendId;
    }

    @Override
    public BackendConfig getBackendConfig() {
        return SecretConfigManager.convertToBackendConfig(SecretServiceType.AES, this.config);
    }

    protected static byte[] generateIV() {
        byte[] iv = new byte[12];
        SecureRandom secureRandom = AESEncryptionBackend.getStrongSecureRandom();
        secureRandom.nextBytes(iv);
        return iv;
    }

    private static SecureRandom getStrongSecureRandom() {
        try {
            SecureRandom secureRandom = SecureRandom.getInstance("DRBG", "SUN");
            log.debug("Using {} algorithm for IV generation", (Object)secureRandom.getAlgorithm());
            return secureRandom;
        }
        catch (NoSuchAlgorithmException | NoSuchProviderException e) {
            SecureRandom secureRandom = new SecureRandom();
            log.error("Error when creating preferred Secure Random, using {}", (Object)secureRandom.getAlgorithm());
            return secureRandom;
        }
    }

    private static byte[] calculateSha256(byte[] bytes) throws NoSuchAlgorithmException {
        MessageDigest digest = MessageDigest.getInstance("SHA-256");
        return digest.digest(bytes);
    }

    public AESConfig getConfig() {
        return this.config;
    }

    @FunctionalInterface
    static interface GeneratedIv {
        public byte[] get();
    }
}

