/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.secrets.service.dao;

import com.atlassian.secrets.api.SealedSecret;
import com.atlassian.secrets.api.SecretDao;
import com.atlassian.secrets.api.SecretServiceException;
import com.atlassian.secrets.service.SecretServiceParams;
import com.atlassian.secrets.service.file.FileWriteRequest;
import com.atlassian.secrets.service.file.FileWriter;
import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

public class MultiFileSecretDao
implements SecretDao {
    public static final String SECRET_DIRECTORY_NAME = "secured";
    private static final Logger log = LoggerFactory.getLogger(MultiFileSecretDao.class);
    private static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;
    private final Path secretDirectoryPath;
    private final FileWriter fileWriter;

    public MultiFileSecretDao(Path homeDirectoryPath) {
        this(homeDirectoryPath, SecretServiceParams.DEFAULT_FILE_WRITER);
    }

    public MultiFileSecretDao(Path homeDirectoryPath, FileWriter fileWriter) {
        this.secretDirectoryPath = homeDirectoryPath.resolve(SECRET_DIRECTORY_NAME);
        this.fileWriter = fileWriter;
        this.createSecretDirectory();
    }

    public void put(Set<SealedSecret> sealedSecrets) {
        for (SealedSecret secret : sealedSecrets) {
            this.writeSealedSecret(secret);
        }
    }

    public Optional<SealedSecret> get(String identifier) {
        Path secretFile = this.getFileForSecret(identifier);
        if (!Files.exists(secretFile, new LinkOption[0])) {
            return Optional.empty();
        }
        return Optional.of(this.parseSealedSecret(secretFile));
    }

    public void delete(String identifier) {
        Path secretFile = this.getFileForSecret(identifier);
        try {
            Files.delete(secretFile);
        }
        catch (IOException e) {
            log.error("Error deleting data for secret '{}', {}", (Object)secretFile.toAbsolutePath(), (Object)e);
        }
    }

    public Set<String> getIdsForBackend(String backendId) {
        HashSet<String> result = new HashSet<String>();
        try (Stream<Path> secretFiles = Files.list(this.secretDirectoryPath);){
            secretFiles.forEach(file -> {
                if (this.hasFileExtension((Path)file)) {
                    return;
                }
                SealedSecret secret = this.parseSealedSecret((Path)file);
                Path fileName = file.getFileName();
                Path expectedFileName = this.getFileForSecret(secret.getIdentifier()).getFileName();
                Assert.isTrue((boolean)fileName.equals(expectedFileName), (String)"Secret file name should match secret");
                if (secret.getBackendId().equals(backendId)) {
                    result.add(secret.getIdentifier());
                }
            });
        }
        catch (IOException e) {
            throw new SecretServiceException("Error getting secrets stored by backend", (Throwable)e);
        }
        return result;
    }

    Path getSecretDirectoryPath() {
        return this.secretDirectoryPath;
    }

    private void createSecretDirectory() {
        File secretDirectoryFile = this.secretDirectoryPath.toFile();
        if (!secretDirectoryFile.exists()) {
            try {
                Files.createDirectories(this.secretDirectoryPath, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new SecretServiceException("Error creating secretDirectoryPath", (Throwable)e);
            }
        }
        if (!secretDirectoryFile.isDirectory()) {
            throw new SecretServiceException("secretDirectoryPath must be a directory");
        }
    }

    private void writeSealedSecret(SealedSecret secret) throws SecretServiceException {
        Path secretFile = this.getFileForSecret(secret.getIdentifier());
        FileWriteRequest request = new FileWriteRequest(secretFile, () -> ((SealedSecret)secret).toString(), false, ignored -> {});
        this.fileWriter.write(request);
    }

    private SealedSecret parseSealedSecret(Path secretFile) {
        try {
            byte[] fileBytes = Files.readAllBytes(secretFile);
            String asString = new String(fileBytes, DEFAULT_CHARSET);
            return SealedSecret.from((String)asString);
        }
        catch (IOException e) {
            throw new SecretServiceException("Error reading secret from file", (Throwable)e);
        }
    }

    Path getFileForSecret(String identifier) {
        BigInteger asNumber = new BigInteger(1, identifier.getBytes(DEFAULT_CHARSET));
        String asHexString = String.format("%040x", asNumber);
        return this.secretDirectoryPath.resolve(asHexString);
    }

    private boolean hasFileExtension(Path filePath) {
        int lastDotIndex = filePath.getFileName().toString().lastIndexOf(46);
        return lastDotIndex != -1;
    }
}

