/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.secrets.service.lock;

import com.atlassian.secrets.api.SecretServiceException;
import com.atlassian.secrets.service.dao.Underlock;
import com.atlassian.secrets.service.lock.ConfigLock;
import java.io.IOException;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

public class ConfigFileLock
implements ConfigLock {
    private static final Logger log = LoggerFactory.getLogger(ConfigFileLock.class);
    public static final String LOCK_TIMEOUT_SYS_PROP = "atlassian.secret.service.lock.timeout";
    public static final Long LOCK_TIMEOUT_DEFAULT_IN_MS = 10000L;
    public static final String STALE_LOCK_DURATION_SYS_PROP = "atlassian.secret.service.lock.stale.age";
    public static final Long STALE_LOCK_DURATION_IN_MS = 10000L;
    public static final String LOCK_FILE_SUFFIX = ".lock";
    static final long lockAcquireTimeout = Long.getLong("atlassian.secret.service.lock.timeout", LOCK_TIMEOUT_DEFAULT_IN_MS);
    static final long staleLockAgeLimit = Long.getLong("atlassian.secret.service.lock.stale.age", STALE_LOCK_DURATION_IN_MS);
    private final Path fileToLock;
    private final Underlock underlock;
    private final Clock clock;

    ConfigFileLock(Path fileToLock, Underlock underlock, Clock clock) {
        Assert.isTrue((!fileToLock.endsWith(LOCK_FILE_SUFFIX) ? 1 : 0) != 0, (String)"Original file path should not end with lock file suffix.");
        this.fileToLock = fileToLock;
        this.underlock = underlock;
        this.clock = clock;
    }

    public ConfigFileLock(Path fileToLock) {
        this(fileToLock, Underlock.forFile(fileToLock, Duration.ofMillis(lockAcquireTimeout)), Clock.systemUTC());
    }

    @Override
    public void releaseStaleLock() {
        Path lockFile = this.fileToLock.resolveSibling(String.valueOf(this.fileToLock.getFileName()) + LOCK_FILE_SUFFIX);
        if (Files.exists(lockFile, new LinkOption[0])) {
            try {
                if (this.isLockFileStale(lockFile)) {
                    log.warn("Lock file {} is stale, removing it.", (Object)lockFile);
                    Files.delete(lockFile);
                }
            }
            catch (IOException e) {
                log.warn("Failed to cleanup stale lock file {}", (Object)lockFile, (Object)e);
            }
        }
    }

    @Override
    public void runWithLock(Runnable action) {
        log.debug("Attempting to acquire lock on secrets-config.yaml file.");
        try (Underlock.LockedFile ignored = this.underlock.lock();){
            log.debug("Acquired lock on secrets-config.yaml file.");
            if (!Files.exists(this.fileToLock, new LinkOption[0])) {
                log.debug("Generating default secrets-config.yaml file.");
                action.run();
            } else {
                log.debug("The secrets-config.yaml file already exists. Skipping generation and going to read it");
            }
        }
        catch (IOException e) {
            log.error("Failed to acquire lock on secrets-config.yaml file.", (Throwable)e);
            log.info("Creating default secrets-config.yaml file without lock.");
            action.run();
        }
        catch (OverlappingFileLockException e) {
            throw new SecretServiceException("Another process is holding a lock on secrets-config.yaml file.", (Throwable)e);
        }
    }

    private boolean isLockFileStale(Path lockFile) throws IOException {
        Duration staleLockAge = Duration.ofMillis(staleLockAgeLimit);
        Instant lastModified = Files.getLastModifiedTime(lockFile, new LinkOption[0]).toInstant();
        Duration age = Duration.between(lastModified, Instant.now(this.clock));
        return age.compareTo(staleLockAge) > 0;
    }
}

