/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.sigillo.signature.verifier;

import com.atlassian.sigillo.signature.verifier.CertificateAttributesExtractor;
import com.atlassian.sigillo.signature.verifier.CertificateChainBuilder;
import com.atlassian.sigillo.signature.verifier.Ed25519PublicKeyParametersAdapter;
import com.atlassian.sigillo.signature.verifier.exception.CertificateValidityException;
import com.atlassian.sigillo.signature.verifier.exception.CertificateValidityExceptionType;
import com.atlassian.sigillo.signature.verifier.exception.CertificateVerificationException;
import com.atlassian.sigillo.signature.verifier.exception.UntrustedCertificateChainException;
import com.atlassian.sigillo.signature.verifier.exception.VerifierException;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.security.PublicKey;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.interfaces.EdECPublicKey;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.crypto.params.Ed25519PublicKeyParameters;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemWriter;

class BcCertificateVerifier {
    private static final int NON_CA_FLAG = -1;

    BcCertificateVerifier() {
    }

    public boolean verifyCertificateLeafAsChain(String leafCertificate, List<String> trustedCertificates) throws VerifierException {
        if (leafCertificate == null || leafCertificate.isBlank()) {
            throw new IllegalArgumentException("Certificate data cannot be null or empty");
        }
        List<X509Certificate> certificates = CertificateChainBuilder.buildX509CertificateChain(leafCertificate, trustedCertificates);
        List<Ed25519PublicKeyParameters> trustedPublicKeys = BcCertificateVerifier.getEd25519PublicKeyParameters(trustedCertificates);
        CertificatesHolder certificatesHolder = new CertificatesHolder(certificates, trustedPublicKeys);
        certificatesHolder.verifyCertificates();
        return true;
    }

    public boolean verifyCertificate(String certificate) throws CertificateException {
        CertificateAttributesExtractor certificateAttributesExtractor = new CertificateAttributesExtractor(certificate);
        certificateAttributesExtractor.checkValidity();
        if (!certificateAttributesExtractor.signatureAlgorithm().equals("Ed25519")) {
            throw new CertificateException("Unsupported signature algorithm in certificate, please use Ed25519");
        }
        return true;
    }

    private static List<Ed25519PublicKeyParameters> getEd25519PublicKeyParameters(List<String> trustedCertsOrPublicKeys) {
        ArrayList<Ed25519PublicKeyParameters> trustedPublicKeys = new ArrayList<Ed25519PublicKeyParameters>();
        for (String trustedInput : trustedCertsOrPublicKeys) {
            try (PEMParser pemParser = new PEMParser(new StringReader(trustedInput));){
                Object object = pemParser.readObject();
                String publicKey = BcCertificateVerifier.getPublicKey(trustedInput, object);
                trustedPublicKeys.add(Ed25519PublicKeyParametersAdapter.adapt(publicKey));
            }
            catch (IOException e) {
                throw new VerifierException("Error while extracting public keys", e);
            }
        }
        return trustedPublicKeys;
    }

    private static String getPublicKey(String trustedInput, Object object) {
        if (object instanceof X509CertificateHolder) {
            X509CertificateHolder certificateHolder = (X509CertificateHolder)object;
            return BcCertificateVerifier.getPemFormatedPublicKeyFromCertificate(certificateHolder);
        }
        if (object instanceof SubjectPublicKeyInfo) {
            return trustedInput;
        }
        throw new VerifierException("Unsupported PEM object: " + object.getClass());
    }

    private static String getPemFormatedPublicKeyFromCertificate(X509CertificateHolder certificateHolder) {
        String publicKey;
        try (StringWriter writer = new StringWriter();
             JcaPEMWriter pemWriter = new JcaPEMWriter(writer);){
            JcaX509CertificateConverter converter = new JcaX509CertificateConverter();
            PublicKey publicKeyEncoded = converter.getCertificate(certificateHolder).getPublicKey();
            ((PemWriter)pemWriter).writeObject(new PemObject("PUBLIC KEY", publicKeyEncoded.getEncoded()));
            pemWriter.flush();
            publicKey = writer.toString();
        }
        catch (IOException | CertificateException e) {
            throw new VerifierException("Error parsing trusted certificate public key", e);
        }
        return publicKey;
    }

    private static class CertificatesHolder {
        private final List<SomeCertificate> certificatesInChain = new ArrayList<SomeCertificate>();
        private boolean hasTrustedCertificate;

        public CertificatesHolder(List<X509Certificate> certificates, List<Ed25519PublicKeyParameters> trustedPublicKeys) {
            if (certificates.isEmpty()) {
                throw new IllegalArgumentException("Certificate chain is empty");
            }
            if (trustedPublicKeys.isEmpty()) {
                throw new IllegalArgumentException("No trusted certificates or public keys provided");
            }
            certificates.forEach(certificate -> this.addCertificate((X509Certificate)certificate, trustedPublicKeys));
        }

        private void addCertificate(X509Certificate certificate, List<Ed25519PublicKeyParameters> trustedPublicKeys) {
            Instant currentTimestamp = Instant.now();
            if (this.isExpired(certificate, currentTimestamp)) {
                throw new CertificateValidityException(CertificateValidityExceptionType.EXPIRED, certificate.getSubjectX500Principal().getName());
            }
            if (this.isBefore(certificate, currentTimestamp)) {
                throw new CertificateValidityException(CertificateValidityExceptionType.NOT_YET_VALID, certificate.getSubjectX500Principal().getName());
            }
            if (!(certificate.getPublicKey() instanceof EdECPublicKey)) {
                throw new IllegalArgumentException("Unsupported public key type");
            }
            Ed25519PublicKeyParameters publicKey = CertificatesHolder.getEd25519PublicKeyParameters((EdECPublicKey)certificate.getPublicKey());
            trustedPublicKeys.forEach(trustedKey -> {
                if (Arrays.equals(publicKey.getEncoded(), trustedKey.getEncoded())) {
                    this.hasTrustedCertificate = true;
                }
            });
            this.certificatesInChain.add(new SomeCertificate(certificate));
        }

        private static Ed25519PublicKeyParameters getEd25519PublicKeyParameters(EdECPublicKey publicKey) {
            byte[] encodedPublicKey = publicKey.getEncoded();
            if (encodedPublicKey.length == 32) {
                return new Ed25519PublicKeyParameters(encodedPublicKey, 0);
            }
            byte[] rawKey = Arrays.copyOfRange(encodedPublicKey, encodedPublicKey.length - 32, encodedPublicKey.length);
            return new Ed25519PublicKeyParameters(rawKey, 0);
        }

        boolean verifyCertificates() throws VerifierException {
            if (!this.hasTrustedCertificate) {
                throw new UntrustedCertificateChainException();
            }
            Iterator<SomeCertificate> certificateIterator = this.certificatesInChain.iterator();
            SomeCertificate certificate = certificateIterator.next();
            SomeCertificate issuerCertificate = certificateIterator.next();
            certificate.verifyAndThrowOnError(issuerCertificate.certificate());
            if (certificate.certificate().getBasicConstraints() != -1) {
                throw new VerifierException("Leaf certificate must not be a Certificate Authority certificate");
            }
            while (certificateIterator.hasNext()) {
                certificate = issuerCertificate;
                issuerCertificate = certificateIterator.next();
                certificate.verifyAndThrowOnError(issuerCertificate.certificate());
                if (certificate.certificate().getBasicConstraints() != -1) continue;
                throw new VerifierException("Intermediate certificate must be a Certificate Authority certificate");
            }
            if (issuerCertificate.certificate().getBasicConstraints() == -1) {
                throw new VerifierException("Root certificate must be a Certificate Authority certificate");
            }
            return true;
        }

        private boolean isExpired(X509Certificate certificate, Instant currentTimestamp) {
            return certificate.getNotAfter().toInstant().isBefore(currentTimestamp);
        }

        private boolean isBefore(X509Certificate certificate, Instant currentTimestamp) {
            return certificate.getNotBefore().toInstant().isAfter(currentTimestamp);
        }
    }

    private record SomeCertificate(X509Certificate certificate) {
        void verifyAndThrowOnError(X509Certificate issuerCertificate) throws VerifierException {
            try {
                this.certificate.verify(issuerCertificate.getPublicKey());
            }
            catch (Exception e) {
                throw new CertificateVerificationException(this.certificate.getSubjectX500Principal().getName(), e);
            }
        }
    }
}

