define('confluence/ob/common/common-core', [
    'jquery'
], function ($) {

    var CommonCore = {};

    /**
     * Return a boolean indicating whether the current locale is English.
     * @returns {boolean}
     */
    CommonCore.isEnglishLocale = function () {
        return !!CommonCore.getLocale().match(/^en/);
    };

    CommonCore.getLocale = function () {
        return AJS.Meta.get("user-locale");
    };

    CommonCore.getLocaleSupportedInfo = function () {
        var info = {};
        info.locale = CommonCore.getLocale();
        info.supported = CommonCore.isEnglishLocale();
        return info;
    };

    /**
     * Return the context path. (e.g. '/jira').
     *
     * WARNING: Only confirmed to work for JIRA and Confluence.
     * TODO: Get this to work for other products (e.g. Stash).
     */
    CommonCore.getContextPath = function () {
        return AJS.contextPath();
    };

    /**
     * Send an analytic event.
     * The specified event name has the id automatically appended to it.
     * This function defines both the "AJS.trigger" and "ajax" implementations.
     * @param eventName
     * @param data
     * @param isAjax - a boolean indicating whether to use the "ajax" implementation
     */
    function sendAnalyticsEvent(eventName, data, isAjax) {
        if (isAjax) {
            return $.ajax({
                type: "POST",
                url: CommonCore.getContextPath() + "/rest/analytics/1.0/publish/bulk",
                data: JSON.stringify([
                    {name: eventName, properties: data || {}}
                ]),
                contentType: "application/json",
                dataType: "json"
            });
        } else {
            AJS.trigger("analyticsEvent", {name: eventName, data: data || {}});
        }
    }

    /**
     * Send an analytic event.
     * Note: This may fail if AJS.trigger is not available. For a more reliable method, use sendAnalyticsEventViaAjax().
     * @param eventName
     * @param data
     */
    CommonCore.sendAnalyticsEvent = function (eventName, data) {
        return sendAnalyticsEvent(eventName, data, false);
    };

    CommonCore.sendFlattenedAnalyticsEvent = function (eventName, data) {
        return sendAnalyticsEvent(eventName, flatten(data), false);
    };

    /**
     * Attempt to send an analytic event directly via Ajax, not relying on AJS.trigger.
     *
     * If there was an error when trying to send the analytic event, then try to safely log an error.
     * @param eventName
     * @param data
     */
    CommonCore.sendAnalyticsEventViaAjax = function (eventName, data) {
        var promise = sendAnalyticsEvent(eventName, data, true);
        if (typeof promise === "undefined") {
            // This can occur in debug mode, when the ajax call is not actually performed
            promise = $.Deferred().resolve().promise();
        }
        return promise;
    };

    /**
     * Is a user logged in?
     * @returns {boolean} a boolean which indicates whether the a user is logged in
     */
    CommonCore.isLoggedIn = function () {
        return !!AJS.Meta.get("remote-user");
    };

    /**
     * Return the global object.
     * Source: http://stackoverflow.com/questions/7290086/javascript-use-strict-and-nicks-find-global-function
     */
    CommonCore.getGlobal = function () {
        return (function () {
            return this || (0 || eval)('this');
        }());
    };

    /**
     * Get the token for XSRF protection.
     */
    CommonCore.getAtlToken = function () {
        return $("#atlassian-token").attr("content");
    };

    /**
     * Is the Analytics plugin available?
     *
     * @returns {boolean} a boolean which indicates that the analytics is available.
     */
    CommonCore.isAnalyticsAvailable = function () {
        return !!(typeof AJS !== 'undefined' && AJS.EventQueue);
    };

    /**
     * Expose the specified value in the global scope via GROW, using the specified property.
     *
     * @param property
     * @param value
     */
    CommonCore.setGROW = function (property, value) {
        var global = CommonCore.getGlobal();
        if (!global.GROW) {
            global.GROW = {};
        }
        global.GROW[property] = value;
    };

    return CommonCore;
});
