define('confluence/ob/common/space-utils', [
    'jquery',
    'ajs',
    'confluence/ob/common/common-core',
    'confluence/ob/common/test-evaluator'
], function ($, AJS, CommonCore, TestEvaluator) {
    var ConfluenceSpaces = {};
    var contextPath = AJS.contextPath();
    var DEFAULT_SPACE_KEY = 'SPC';

    ConfluenceSpaces.SPACE_DEMO_DESC = "efionboardingspace";

    ConfluenceSpaces.generateSpaceKey = function (name) {
        var keygen = new KeyGenerator();
        return keygen.generateKey(name, {desiredKeyLength: 4, maxKeyLength: 10}) || DEFAULT_SPACE_KEY;
    };

    ConfluenceSpaces.generateUniqueSpaceKey = function(key) {
        var deferred = $.Deferred();

        $.ajax({
            url: contextPath + "/rest/onboarding/1/space/generate?key=" + key,
            type: "GET",
            contentType: "application/json",
            success: function (data) {
                deferred.resolve(data.key);
            },
            error: function (xhr) {
                if (xhr.status === 401) {
                    deferred.reject({ status: 401, msg: "Unauthorized" });
                } else if (xhr.status === 504) {
                    deferred.reject({ status: 504, msg: "Gateway timeout" });
                } else {
                    var err = { status: xhr.status };
                    deferred.reject(err);
                }
            }
        });
        return deferred.promise();
    };

    ConfluenceSpaces.createSpace = function (name, key) {
        var deferred = $.Deferred();

        if (AJS.DarkFeatures.isEnabled("confluence.efi.onboarding.rich.space.content")) {
            $.ajax({
                url: contextPath + "/rest/onboarding/1/space",
                type: "POST",
                contentType: "application/json",
                dataType: "json",
                data: JSON.stringify({name: name, key: key, temporary: TestEvaluator.isTestingEvaluator()})
            }).done(function(spaceDetails) {
                deferred.resolve(spaceDetails);
            }).fail(function(xhr) {
                if (xhr.status === 401) {
                    deferred.reject({ status: 401, msg: "Unauthorized" });
                } else if (xhr.status === 504) {
                    deferred.reject({ status: 504, msg: "Gateway timeout" });
                } else {
                    var msg = "Unexpected error";
                    try {
                        var responseJson = JSON.parse(xhr.responseText);
                        if (responseJson && responseJson.error) {
                            msg = responseJson.message;
                        }
                    } catch(e) {}
                    deferred.reject({status: xhr.status, msg: msg});
                }
            });
        } else {
            var data = {
                name: name,
                key: key
            };

            // add demo description for find and remove it later
            if (TestEvaluator.isTestingEvaluator()) {
                data.description = {
                    plain: {
                        value: ConfluenceSpaces.SPACE_DEMO_DESC,
                        representation: "plain"
                    }
                
                };
            }
            $.ajax({
                url: contextPath + "/rest/api/space",
                type: "POST",
                contentType: "application/json",
                dataType: "json",
                data: JSON.stringify(data),
                success: function (spaceDetails) {
                    deferred.resolve(spaceDetails);
                },
                error: function (xhr) {
                    if (xhr.status === 401) {
                        deferred.reject({ status: 401, msg: "Unauthorized" });
                    } else if (xhr.status === 504) {
                        deferred.reject({ status: 504, msg: "Gateway timeout" });
                    } else {
                        var msg = "An unexpected error occurred.";
                        try {
                            var data = JSON.parse(xhr.responseText);
                            if (data && data.message) {
                                msg = data.message;
                            }
                        } catch(e) {}
                        deferred.reject({status: xhr.status, msg: msg});
                    }
                }
            });
        }

        return deferred.promise();
    };

    /**
    * Note that the response coming back as a 200 doesn't mean that the space has been deleted,
    * as this is a long running task. You should follow the status link in the response and poll
    * it until the task completes.
    */ 
    ConfluenceSpaces.deleteSpace = function (spaceKey) {
        var deferred = $.Deferred();

        $.ajax({
            type: "DELETE",
            url: contextPath + "/rest/api/space/" + spaceKey,
            success: function(response) {
                deferred.resolve(response);
            },
            error: function(err) {
                deferred.reject(err);
            }
        });

        return deferred.promise(); 
    };

    /**
     * Get all of the available blueprints (as an array) for the specified space.
     * @param spaceKey
     * @returns $.Deferred
     */
    ConfluenceSpaces.getAvailableBlueprints = function(spaceKey) {
        var deferred = $.Deferred();
        var milliseconds = (new Date()).valueOf().toString();
        var url = [CommonCore.getContextPath(), '/rest/create-dialog/1.0/blueprints/web-items?spaceKey=', spaceKey, '&_=', milliseconds].join('');

        $.ajax({
            url: url,
            dataType: "json"
        }).then(function (blueprints) {
            deferred.resolve(blueprints);
        }, function () {
            deferred.reject();
        });
        return deferred;
    };

    return ConfluenceSpaces;
});
