define('confluence/jim/jira/jira-issues-view-mode/fetching-job', [
    'jquery',
    'ajs',
    'confluence/jim/confluence-shim',
    'confluence/jim/util/retry-caller'
], function ($,
             AJS,
             Confluence,
             retryCaller) {
    'use strict';

    const CONNECTION_TIMEOUT = "http-connection-timeout"
    const SOCKET_TIMEOUT = "http-socket-timeout"


    /**
     * Fetching Job object - abstract of ajax call to fetch content
     * @param options
     * @constructor
     */
    var FetchingJob = function (options) {
        this.clientIds = options.clientIds;
        this.maxTimeout = this.getTimeout();

        var TICK = 1000;
        const GRACE_INTERVAL = 13;
        const PREVIOUS_CLOSE = 59

        // we need total ~120 seconds in server side to render timeout error.
        this.TICK_RETRIES = [
            0,
            1 * TICK,
            1 * TICK,
            2 * TICK,
            3 * TICK,
            5 * TICK,
            8 * TICK,
            13 * TICK,
            13 * TICK,
            13 * TICK
        ];

        /**
         The following method call ensure and synchronize the backend timeout with frontend by adding extra ticks,
         if the backend timeout exceeds 59 seconds
         */
        this.computeTickRetries(TICK, PREVIOUS_CLOSE, GRACE_INTERVAL)

        // returned HTTP code which will help to detect whether reloading data.
        this.RETRY_HTTP_CODE = 202;
    };

    FetchingJob.prototype.getTimeout = function () {
        const connectionTimeout = AJS.Meta.get(CONNECTION_TIMEOUT);
        const socketTimeout = AJS.Meta.get(SOCKET_TIMEOUT);

        if (connectionTimeout === undefined ||  socketTimeout === undefined) {
            return null;
        }

        return Number(connectionTimeout) + Number(socketTimeout)

    }

    FetchingJob.prototype.computeTickRetries = function (TICK, startTime, interval) {
        if (this.maxTimeout === null) {
            return ;
        }

        const TICK_RETRIES = []
        const maxTimeoutInSeconds = Math.ceil(this.maxTimeout / 1000);
        if (maxTimeoutInSeconds >= startTime) {
            let startTimeout = startTime;
            while (startTimeout <= maxTimeoutInSeconds) {
                startTimeout = startTimeout + interval;
                TICK_RETRIES.push(interval * TICK);
            }
        }

        this.TICK_RETRIES = [...this.TICK_RETRIES, ...TICK_RETRIES];
    }

    FetchingJob.prototype.startJob = function () {
        return this.fetchSingeJiraServer();
    };

    /**
     * Begin to featch data from a Jira Server
     * @param jiraServerId
     * @returns {Object} a jQuery Deferred object
     */
    FetchingJob.prototype.fetchSingeJiraServer = function () {
        var jimUrl = AJS.contextPath() + '/rest/jiraanywhere/1.0/jira/clientIds/';
        var promise = $.ajax({
            type: 'POST',
            url: jimUrl,
            data: this.clientIds,
            contentType: 'application/json',
            cache: true
        });

        // we need to cache jira server id so that we know which Promise object is rejected later
        // and render error message
        promise.clientIds = this.clientIds;

        return promise;
    };

    /**
     * Start the job with having retry ability
     * @returns {Object} a Promise object
     */
    FetchingJob.prototype.startJobWithRetry = function () {
        return retryCaller(
            this.startJob, {
                name: this.clientIds, // for logging
                delays: this.TICK_RETRIES,
                context: this,
                tester: function (dataOfAServer, successMessage, promise) {
                    // if status is 202, we need to retry to call the same ajax again
                    return promise && promise.status === this.RETRY_HTTP_CODE;
                }
            }
        );
    };

    return FetchingJob;
});
