/**
 * @module confluence-schedule-admin/viewscheduledjobs
 */
define('confluence-schedule-admin/viewscheduledjobs', [
    'jquery',
    'ajs',
    'confluence/templates',
    'window',
    'wrm/data'
], function(
    $,
    AJS,
    confTemplates,
    window,
    wrmData
) {
    'use strict';

    const cronExpressionHelpDoc = wrmData.claim('com.atlassian.confluence.plugins.confluence-schedule-admin:view-scheduled-jobs.help-docs');

    function ViewScheduledJobs() {
        var processCronExpressionChange = function(resultDiv, cronExpInputField, okButton) {
            var initValue = cronExpInputField.val();

            AJS.log('processCronExpressionChange(val=' + initValue + ')');

            $('#edit-cron button.ok').prop('disabled', true);

            $.getJSON('getFutureSchedules.action', {
                cronExpression: initValue
            },
            function(data, textStatus) {
                if (initValue == cronExpInputField.val()) {
                    // text in input has not changed, so response is still useful
                    var html;
                    if (data.futureSchedules) {
                        $('#edit-cron button.ok').prop('disabled', false);

                        const rowData = data.futureSchedules
                            .map((schedule, index) => {
                                const previousDate = index > 0 ? data.futureSchedules[index - 1].date : null;
                                const displayedDate = schedule.date === previousDate ? '' : schedule.date;
                                return { date: displayedDate, time: schedule.time };
                            });
                        html = confTemplates.Jobs.Scheduled.futureSchedulesTable({ rows: rowData });
                    } else {
                        // An error occurred
                        $('#edit-cron button.ok').prop('disabled', true);
                        html = confTemplates.Jobs.Scheduled.validationError({ dataError: (data.reason && (' - ' + data.reason)) || '' });
                    }
                    resultDiv.html(html);
                } else {
                    AJS.log('Ignoring JSON response as input has changed');
                }
            }
            );
        };

        var showEditJobDialog = function(options, label, setup) {
            var onCancel = function() {
                popup.remove();
                return false;
            };

            options = options || {};
            options.onCancel = onCancel;
            var popup = new AJS.ConfluenceDialog(options);

            popup.addHeader(AJS.I18n.getText('scheduledjob.edit.heading', label));
            popup.addPanel(AJS.I18n.getText('scheduledjob.edit.cronDefinition'), 'panel2');

            setup(popup);

            popup.addCancel(AJS.I18n.getText('cancel.name'), onCancel);

            popup.gotoPage(0);
            popup.gotoPanel(0);

            popup.show();
        };

        var editSimpleJob = function(jobId, label, repeatInterval, defaultRepeatInterval) {
            AJS.log('AJS.ScheduledSimpleJobs.edit');
            showEditJobDialog({
                id: 'edit-simple',
                width: 640,
                height: 330
            }, label, function setup(popup) {
                popup.getCurrentPanel()
                    .html(confTemplates.Jobs.Scheduled.simpleSchedule({ repeatInterval: repeatInterval, defaultRepeatInterval: defaultRepeatInterval }));

                var newRepeatInterval = $('#newRepeatInterval');
                var errorMessageArea = $('#errorMessage');

                popup.addButton(AJS.I18n.getText('update.name'), function(dialog) {
                    AJS.safe.ajax({
                        url: 'changeSimpleJobSchedule.action',
                        type: 'POST',
                        data: {
                            id: jobId,
                            repeatInterval: newRepeatInterval.val()
                        },
                        success: function(data) {
                            if (data.error) {
                                const message = confTemplates.Jobs.Scheduled.validationError({ dataError: data.error });
                                errorMessageArea.html(message);
                            } else {
                                dialog.remove();
                                location.reload();
                            }
                        },
                        error: function(req, status, error) {
                            const message = confTemplates.Jobs.Scheduled.generalError({
                                message: AJS.I18n.getText('scheduledjob.edit.saveFailed'),
                                statusText: req.statusText,
                                statusCode: req.status
                            });
                            errorMessageArea.html(message);
                        }
                    });
                });
            });
        };

        var editCronJob = function(jobId, label, cronValue, defaultCronValue) {
            AJS.log('AJS.ScheduledCronJobs.edit');

            showEditJobDialog({
                id: 'edit-cron',
                width: 640,
                height: 470
            }, label, function setup(popup) {
                popup.getCurrentPanel().html(confTemplates.Jobs.Scheduled.cronSchedule({ cron: (cronValue || defaultCronValue), cronExpressionHelpDoc: cronExpressionHelpDoc }));

                var newCronValueInput = $('#newCronValue');
                var previewArea = $('#cronPreview');

                newCronValueInput.keyup(function() {
                    processCronExpressionChange(previewArea, newCronValueInput);
                });
                processCronExpressionChange(previewArea, newCronValueInput);

                popup.addButton(AJS.I18n.getText('default.name'), function(dialog) {
                    newCronValueInput.val(defaultCronValue);
                    processCronExpressionChange(previewArea, newCronValueInput);
                });

                popup.addButton(AJS.I18n.getText('update.name'), function(dialog) {
                    AJS.safe.ajax({
                        url: 'changeCronJobSchedule.action',
                        type: 'POST',
                        data: {
                            id: jobId,
                            cronExpression: newCronValueInput.val()
                        },
                        success: function(data) {
                            if (data.error) {
                                const message = confTemplates.Jobs.Scheduled.validationError({ dataError: data.error });
                                previewArea.html(message);
                            } else {
                                newCronValueInput.unbind();
                                dialog.remove();
                                location.reload();
                            }
                        },
                        error: function(req, status, error) {
                            const message = confTemplates.Jobs.Scheduled.generalError({
                                message: AJS.I18n.getText('scheduledjob.edit.saveFailed'),
                                statusText: req.statusText,
                                statusCode: req.status
                            });
                            previewArea.html(message);
                        }
                    });
                }, 'ok');
            });
        };

        var showHistory = function(jobId, label) {
            AJS.log('AJS.ScheduledJobs.showHistory');
            var onCancel = function() {
                popup.remove();
            };
            var popup = new AJS.ConfluenceDialog({
                width: 640,
                height: 455,
                id: 'job-history',
                onCancel: onCancel
            });

            popup.addHeader(AJS.I18n.getText('scheduledjob.history.heading', label));

            popup.addPanel('Panel 1', 'panel1');

            $.ajax({
                url: 'viewscheduledjobhistory.action',
                data: {
                    id: jobId
                },
                success: function(data) {
                    popup.getCurrentPanel().html(data);
                },
                error: function(req, status, error) {
                    const message = confTemplates.Jobs.Scheduled.generalError({
                        message: AJS.I18n.getText('scheduledjob.history.loadFailed'),
                        statusText: req.statusText,
                        statusCode: req.status
                    });
                    popup.getCurrentPanel().html(message);
                }
            });

            popup.addCancel(AJS.I18n.getText('close.name'), onCancel);

            popup.gotoPage(0);
            popup.gotoPanel(0);

            popup.show();
        };

        /**
         * Gets the current row data for the specified element in the row.
         *
         * @near a node in a row of the schedule-admin table
         * @return the data values for the row as a structure.
         */
        var getRowData = function(near) {
            var dataNode = $(near).closest('tr');
            return {
                jobId: dataNode.attr('data-job-id'),
                label: dataNode.attr('data-job-name'),
                cronValue: dataNode.attr('data-cron-expression'),
                defaultCronValue: dataNode.attr('data-default-cron-expression'),
                isCron: dataNode.attr('data-is-cron') === 'true',
                repeatInterval: dataNode.attr('data-repeat-interval'),
                defaultRepeatInterval: dataNode.attr('data-default-repeat-interval')
            };
        };

        /**
         * Bind's all click events for the specified selector in the schedule-admin table to call
         * the supplied actionFunction.
         */
        var bindLink = function(linkSelector, actionFunction) {
            $('#schedule-admin tbody').delegate(linkSelector, 'click', function(e) {
                var rowData = getRowData(e.target);
                actionFunction(rowData);
                e.preventDefault();
                return false;
            });
        };

        /**
         * Bind's all click events for the specified selector in the schedule-admin table to open
         * the specified action, passing in the id parameter.
         */
        var bindJobLink = function(linkSelector, actionName) {
            bindLink(linkSelector, function(rowData) {
                window.location = actionName + '.action?id=' + encodeURIComponent(rowData.jobId)
                        + '&atl_token=' + $('#atlassian-token').attr('content');
            });
        };

        bindLink('.edit-schedule', function(rowData) {
            if (rowData.isCron === true) {
                editCronJob(rowData.jobId, rowData.label, rowData.cronValue, rowData.defaultCronValue);
            } else {
                editSimpleJob(rowData.jobId, rowData.label, rowData.repeatInterval, rowData.defaultRepeatInterval);
            }
        });

        bindLink('.show-history', function(rowData) {
            showHistory(rowData.jobId, rowData.label);
        });

        bindJobLink('.run-job', 'runJob');
        bindJobLink('.enable-job', 'enableJob');
        bindJobLink('.disable-job', 'disableJob');
    }

    return ViewScheduledJobs;
});

require('confluence/module-exporter').safeRequire('confluence-schedule-admin/viewscheduledjobs', function(ViewScheduledJobs) {
    'use strict';

    require('ajs').toInit(ViewScheduledJobs);
});
