(function ($) {
    const baseStorageKey = 'atlassian-analytics';
    const contextPath =
        typeof AJS.contextPath === 'function' ? AJS.contextPath() : '';

    let storageKey = null;
    let lockKey = null;

    // A unique identifier for this browser tab
    // Source: http://stackoverflow.com/a/2117523
    const uuid = 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
        const r = Math.random() * 16 | 0, v = c === 'x' ? r : (r & 0x3 | 0x8);
        return v.toString(16);
    });

    const determineStorageKey = function () {
        // We need to give each product its own key for events because it's possible to host multiple products
        // at the same url.
        let product = 'unknown';
        if (document.body.id === 'jira') {
            product = 'jira';
        } else if (document.body.id === 'com-atlassian-confluence') {
            product = 'confluence';
        }
        storageKey = baseStorageKey + '.' + product;
        lockKey = storageKey + '.lock';
    };

    const getLock = function () {
        if (store.get(lockKey)) {
            return false;
        }

        store.set(lockKey, uuid);

        // Reduce chance of race condition - read back lock to make sure we still have it
        return (store.get(lockKey) === uuid);
    };

    const releaseLock = function () {
        store.set(lockKey, null);
    };

    const clearEventsInStorage = () => store.remove(storageKey);

    /**
     * Persists the events that have been generated until such time that they can be sent.
     */
    const saveForLater = function () {
        let events = [],
            event,
            e, i, ii;
        if (AJS.EventQueue.length === 0)
            return;
        // Prime our events array with anything that's already saved.
        events = store.get(storageKey) || events;
        // Suck the events out of the event queue and in to our events array.
        for (i = 0, ii = AJS.EventQueue.length; i < ii; ++i) {
            e = AJS.EventQueue[i];
            if (e.name) {
                // the queue could contain anything - shear unusable properties
                event = { name: e.name, properties: e.properties, time: e.time || 0 };
                events.push(event);
            }
        }
        // Empty the event queue
        AJS.EventQueue.length = 0;
        // Save our events for later;
        // if this fails with storage error (quota exceeded), clear the events to not clog the storage
        store.set(storageKey, events, clearEventsInStorage);
    };

    // Variable to track the number of retries to publish
    let bulkPublishRetryCount = 0;

    /**
     * Gets the amount of time that should be waited until the next publish attempt.
     * @param retryCount How many requests failed since the last successful publish.
     * @returns {number} how many ms that should be waited.
     */
    const getBulkPublishBackoff = function (retryCount) {
        return Math.min(5000 * Math.pow(2, retryCount), 5 * 60 * 1000);
    };

    /**
     * As part of bundling this plugin in BTF now, we need to remove the existing JIRA analytics setting if we see it.
     */
    const removeOldAnalytics = function () {
        if (window.location.pathname.indexOf('/secure/admin/ViewApplicationProperties') > -1) {
            AJS.$('[data-property-id=\'analytics-enabled\']').remove();
        } else if (window.location.pathname.indexOf('/secure/admin/EditApplicationProperties') > -1) {
            const $analytics = AJS.$(':contains(Enable Atlassian analytics)');
            if ($analytics.length > 0) {
                const parentElement = $analytics[$analytics.length - 2];
                if (parentElement) {
                    parentElement.remove();
                }
            }
        }
    };

    function valueExists(propertyValue) {
        return typeof propertyValue !== 'undefined' && propertyValue !== null;
    }

    function isAllowedType(propertyValue) {
        return typeof propertyValue === 'number' || typeof propertyValue === 'string' || typeof propertyValue === 'boolean';
    }

    const sanitiseProperties = function (properties) {
        for (const propertyName in properties) {
            if (properties.hasOwnProperty(propertyName)) {
                const propertyValue = properties[propertyName];

                if (valueExists(propertyValue) && isAllowedType(propertyValue)) {
                    // Do nothing - the property value is safe & allowed already
                } else if (valueExists(propertyValue) && propertyValue.toString) {
                    // Sanitise the property value by invoking its "toString"
                    properties[propertyName] = propertyValue.toString();
                } else {
                    // If it's an undefined, null or invalid value - blank it out
                    properties[propertyName] = '';
                }
            }
        }
    };

    /**
     * Check for any invalid events and remove/sanitise them.
     * @param events - the list of events to be published
     * @returns the number of valid events remaining
     */
    const validateEvents = function (events) {
        for (let i = events.length - 1; i >= 0; i--) {
            let validMsg = '';
            const event = events[i];
            const properties = event.properties;
            if (typeof event.name === 'undefined') {
                validMsg = 'you must provide a name for the event.';
            } else if (typeof properties !== 'undefined' && properties !== null) {
                if (properties.constructor !== Object) {
                    validMsg = 'properties must be an object with key value pairs.';
                } else {
                    sanitiseProperties(properties);
                }
            }
            if (validMsg !== '') {
                AJS.log('WARN: Invalid analytics event detected and ignored, ' + validMsg + '\nEvent: ' + JSON.stringify(event));
                events.splice(i, 1);
            }
        }
        return events.length;
    };

    let lockFailures = 0;

    /**
     * Publishes every event that's ready for publishing.
     */
    const bulkPublish = function () {
        let events;

        function reschedule() {
            setTimeout(bulkPublish, getBulkPublishBackoff(bulkPublishRetryCount = 0));
        }

        function rescheduleFailed() {
            setTimeout(bulkPublish, getBulkPublishBackoff(++bulkPublishRetryCount));
        }

        // Avoid multiple browser tabs hitting this all at once
        if (!getLock()) {
            ++lockFailures;
            // if we have been failing to get the lock for a while, just grab it
            if (lockFailures < 20) {
                return reschedule();
            }
        } else {
            lockFailures = 0;
        }
        try {
            // Make sure every event we might have is stored.
            saveForLater();
            // Pull the stored events out and get 'em ready for transmission.
            events = store.get(storageKey);

            if (!events || !events.length) {
                return reschedule();
            }

            // Wipe the stored events.
            store.remove(storageKey);

            // Validate events and remove any dodgy ones
            if (!validateEvents(events)) {
                return reschedule();
            }

            // try to present a rough timing of events that the server can interpret relative to its own time.
            const currentTime = new Date().getTime();
            for (let i = 0; i < events.length; i++) {
                if (events[i].time > 0) {
                    events[i].timeDelta = events[i].time - currentTime;
                } else {
                    // just fake it. This corresponds to a millisecond for each place behind last in the array.
                    // This should be rare. Basically, events added to EventQueue before this script was loaded.
                    events[i].timeDelta = i - events.length;
                }
                delete events[i].time;
            }

            fetch(contextPath + '/rest/analytics/1.0/publish/bulk', {
                method: 'POST',
                body: JSON.stringify(events),
                headers: {
                    'Content-Type': 'application/json'
                },
                cache: 'no-cache',
                credentials: 'same-origin'
            }).then(response => {
                if (!response.ok) {
                    throw new Error('A response outside OK range (200-299)');
                }

                reschedule();
            }).catch(() => {
                // In case the transmission fails, let's keep the events we just attempted to send.
                AJS.EventQueue = AJS.EventQueue.concat(events);

                saveForLater();
                rescheduleFailed();
            }).catch((err) => {
                console.error(err);
            });
        } finally {
            releaseLock();
        }
    };

    AJS.toInit(function () {
        determineStorageKey();
        setTimeout(bulkPublish, 500);
        removeOldAnalytics();
    });

    $(window).on('beforeunload', function () {
        saveForLater();
    });

    /**
     * Binds to an event that developers can trigger without having to do any feature check.
     * If this code is available then the event will get published and if it's not the event
     * will go unnoticed.
     * @example
     * AJS.trigger('analytics', {name: 'pageSaved', data: {pageName: page.name, space: page.spaceKey}});
     */
    AJS.bind('analytics', function (event, data) {
        AJS.EventQueue.push({ name: data.name, properties: data.data });
    });

    // legacy binding until Confluence page layout JS is updated
    AJS.bind('analyticsEvent', function (event, data) {
        AJS.EventQueue.push({ name: data.name, properties: data.data });
    });
}(AJS.$));
