(function($) {
    var contextPath =
        typeof AJS.contextPath === "function" ? AJS.contextPath() : "";

    AJS.toInit(function ($) {
        var restUrl = contextPath + "/rest/analytics/1.0/report";
        var EventList = Backbone.Collection.extend({
            url: restUrl,

            parse: function(response) {
                return response.events;
            },

            comparator: function(event) {
                return new Date(event.get('time'));
            }
        });

        var EventView = Backbone.View.extend({

            el: $("#event-log-section"),

            events: {
                "click .event-report-capturing" : "capturingButtonClicked",
                "click .event-report-clear" : "clearEvents",
                "click .event-report-mode .aui-button" : "setModeToggle",
                "input #event-report-filter" : "filterEvents"
            },

            initialize: function() {
                this.awaitingResult = false;
                this.collection = new EventList();
                this.table = $('table.event-report-table');
                this.fetchEvents(true);
                this.setAutoRefresh();

                this.collection.on("add", this.addEvent, this);
            },

            populateTable: function () {
                var self = this;
                this.clearTable();
                if (this.collection) {
                    this.collection.each(function(event) {
                        self.addEvent(event);
                    });
                }
            },

            capturingButtonClicked: function() {
                var $button = $('.event-report-capturing');
                var capturing = $button.hasClass("capturing");
                this.showLoadingIcon();

                // Toggle state
                capturing = !capturing;

                fetch(restUrl, {
                    method: "PUT",
                    headers: { "Content-Type": "application/json" },
                    cache: "no-cache",
                    credentials: "same-origin",
                    body: JSON.stringify({capturing}),
                })
                    .then(() => {
                        this.updateCapturingState(capturing);
                        this.hideLoadingIcon();
                    }, () => {
                        this.hideLoadingIcon();
                    });
            },

            fetchEvents: function(noIcon) {
                var self = this;
                if (!self.awaitingResult && this.isCapturing()) {
                    if (noIcon) {
                        self.showLoadingIcon();
                    }
                    self.awaitingResult = true;
                    this.collection.fetch({
                        data: {
                            mode: self.getMode()
                        },
                        success: function() {
                            self.awaitingResult = false;
                            self.hideLoadingIcon();
                        },
                        error: function() {
                            self.awaitingResult = false;
                            self.hideLoadingIcon();
                        }});
                }
            },

            clearEvents: function() {
                this.clearTable();
                this.showLoadingIcon();
                fetch(restUrl, { method: "DELETE" }).then(() => {
                    this.hideLoadingIcon();
                }, () => {
                    this.hideLoadingIcon();
                });
            },

            setAutoRefresh: function() {
                var self = this;
                self.refreshTimer = setInterval(function() {
                    if (!document.hidden) {
                        self.fetchEvents();
                    }
                }, 1000);
            },

            filterEvents: function(e) {
                this.populateTable();
            },

            clearTable: function() {
                this.table.find("tbody").html("");
            },

            updateCapturingState: function(capturing) {
                if (capturing) {
                    $(".event-capturing-toggle").addClass("capturing");
                } else {
                    $(".event-capturing-toggle").removeClass("capturing");
                }
            },

            isCapturing: function() {
                return $(".event-capturing-toggle").hasClass("capturing");
            },

            showLoadingIcon: function() {
                $(".event-report-loading").removeClass("hidden");
            },

            hideLoadingIcon: function() {
                $(".event-report-loading").addClass("hidden");
            },

            getMode: function() {
                return $(".event-report-mode .aui-button[aria-pressed=true]").attr("mode");
            },

            setModeToggle: function(e) {
                var $currentMode = $(".event-report-mode .aui-button[aria-pressed=true]");
                var $target = $(e.target);

                if ($currentMode[0] !== $target[0]) {
                    $(".event-report-mode .aui-button").attr("aria-pressed", false);
                    $target.attr("aria-pressed", true);

                    this.clearTable();
                    this.fetchEvents();
                }
            },

            addEvent: function(event) {
                var $tableBody = this.table.find("tbody");
                var filterText = $('#event-report-filter').val();
                var requestCorrelationId = event.get('requestCorrelationId');

                if (!filterText || event.get('name').indexOf(filterText) !== -1) {
                    var $row = $("<tr/>")
                            .append($("<td/>").text(event.get('time')).addClass("event-report-table-nowrap"))
                            .append($("<td/>").text(event.get('name')).addClass("event-report-table-nowrap"))
                            .append($("<td/>").text(event.get('user') || "-").addClass("event-report-table-nowrap"))
                            .append($("<td/>").addClass("event-report-table-fill").append(this.renderProperties(requestCorrelationId, event.get('properties'))));
                    if (event.get('removed')) {
                        $row.addClass("event-report-event-removed");
                    }
                    $tableBody.prepend($row);
                }
            },

            renderProperties: function(requestCorrelationId, properties) {
                var $container = $("<div/>").addClass("event-report-table-expander");

                if (requestCorrelationId != null){
                    $container.append("requestCorrelationId: ", requestCorrelationId);
                }

                if (!properties) {
                    $container.append($("<p/>").text("-"));
                } else {
                    var sortedKeys = Object.keys(properties).sort();
                    $.each(sortedKeys, function(index, key) {
                        $container.append($("<p/>").text(key + ": " + properties[key]));
                    });

                    if ($container.find("p").length >= 1) {
                        $container.append($("<button/>")
                            .addClass("aui-button")
                            .addClass("aui-button-link")
                            .addClass("event-report-table-expander-trigger")
                            .click(function (e) {
                                 e.preventDefault();
                                 var $this = $(this),
                                         $expander = $this.parents(".event-report-table-expander");
                                 if ($expander.hasClass("expanded")) {
                                     $expander.removeClass("expanded");
                                 } else {
                                     $expander.addClass("expanded");
                                 }
                            }).append($("<span/>")
                                .addClass("aui-icon")
                                .addClass("aui-icon-small")
                                .addClass("aui-iconfont-chevron-down")
                            )
                        )
                    }
                }
                return $container;
            }
        });

        var eventsView = new EventView();

        (function initWhitelists() {
            $.get(restUrl + "/whitelist", function(data) {
                $.each(data, function(i, whitelist) {
                    var $row = $("<tr/>")
                            .append($("<td/>").append($("<a/>").attr("href", contextPath + "/plugins/servlet/event/whitelist?whitelistId=" + whitelist.whitelistId).text(whitelist.whitelistId)))
                            .append($("<td/>").text(Object.keys(whitelist.filteredEvents).length));
                    $(".whitelist-expand tbody").append($row);
                });
            });
        })();

        document.querySelector('#expand-all-events').addEventListener('click', (e) => {
            e.preventDefault();
            [...document.querySelectorAll('.event-report-table-expander:not(.expanded)')]
                .forEach(row => row.classList.add('expanded'));
        });
        document.querySelector('#contract-all-events').addEventListener('click', (e) => {
            e.preventDefault();
            [...document.querySelectorAll('.expanded')]
                .forEach(row => row.classList.remove('expanded'));
        });

        $('#expand-whitelists-link').keydown(function (e) {
            if (e.keyCode !== 13 || (e.originalEvent && e.originalEvent.repeat)) {
                return;
            }
            $('.whitelist-expand').slideToggle();
        });

        $('#expand-whitelists-link').click(function (e) {
            e.preventDefault();
            $('.whitelist-expand').slideToggle();
        });

        $(".event-report-mode .aui-button").tooltip();
        $(".event-log-group .aui-button").tooltip();
        $(".event-capturing-toggle .aui-iconfont-help").tooltip();
    });
})(AJS.$);
