/**
 * Provides a store functionality based on the native localStorage implementation.
 */

(function () {
    const errorMessagePrefix = 'atlassian-analytics';
    let storage;

    /**
     * Gets the localStorage size estimated by the length property of each value.
     * It turns out to be the efficient estimator compared to other methods (Blob, TextEncoder, etc.).
     * @returns {number} Estimated size of the localStorage values in bytes.
     */
    function getStorageSize() {
        if (storage && typeof storage === 'object') {
            return Object.values(storage)
                .map(value => value.length)
                .reduce((accumulator, currentValue) => accumulator + currentValue, 0);
        } else {
            return 0;
        }
    }

    /**
     * Converts the size in bytes to human-friendly representation.
     * @param bytesSize {number} Size in bytes to convert to metric system.
     * @returns {string} Textual representation of the size, either in kB or MB, trimmed to hundredths.
     */
    function convertSizeToLabel(bytesSize) {
        const divider = 1000;
        let metric = 'kB';
        let size = bytesSize / divider;

        if (size > divider) {
            size /= divider;
            metric = 'MB';
        }

        return `${size.toFixed(2)}${metric}`;
    }

    /**
     * Logs error to the console with additional message and constant prefix.
     * @param message Additional message to be logged before actual error.
     * @param error Caught error to be logged.
     */
    function logError(message, error) {
        console.error(`${errorMessagePrefix} -`, message, error);
    }

    /**
     * Logs the storage error with the information about used space and the data size attempted to be stored in the case of write.
     * @param message - Additional message to be logged before actual error.
     * @param error - Caught error to be logged.
     * @param {string} [value] Serialized (JSON string) data attempted to be set in the storage.
     */
    function logStorageError(message, error, value) {
        const storageUsedSize = convertSizeToLabel(getStorageSize());
        let fullMessage;

        if (value !== undefined && typeof value === 'string') {
            // Tried to set value in the storage
            const valueSize = convertSizeToLabel(value.length);
            fullMessage = `${message} size: ${valueSize} (used space: ${storageUsedSize})`;
        } else {
            fullMessage = `${message} (used space: ${storageUsedSize})`;
        }

        logError(fullMessage, error);
    }

    function serialize(value) {
        try {
            return JSON.stringify(value);
        } catch (error) {
            logError('serialize', error);
        }
    }

    function deserialize(value) {
        try {
            return JSON.parse(value);
        } catch (error) {
            logError('deserialize', error);
        }
    }

    try {
        storage = window['localStorage'];
        storage.setItem('__test__', '__test__');
        storage.removeItem('__test__');
    } catch (error) {
        logError('localStorage is not available', error);
    }

    const store = {
        remove: function (key) {
            try {
                storage.removeItem(key);
            } catch (error) {
                logError(`store.remove(${key})`, error);
            }
        },
        set: function (key, value, onErrorCallback) {
            if (value === undefined) {
                return store.remove(key);
            }
            let serializedValue;

            try {
                serializedValue = serialize(value);
                storage.setItem(key, serializedValue);
            } catch (error) {
                logStorageError(`store.set(${key})`, error, serializedValue);
                if (onErrorCallback && typeof onErrorCallback === 'function') {
                    onErrorCallback();
                }
            }
        },
        get: function (key) {
            try {
                return deserialize(storage.getItem(key));
            } catch (error) {
                logStorageError(`store.get(${key})`, error);
            }
        },
        clear: function () {
            try {
                storage.clear();
            } catch (error) {
                logError('store.clear()', error);
            }
        }
    };

    // Expose the store wrapper as the global variable
    this.store = store;
})();