/**
 * Provides overrides that add a custom HTTP header to XHR and fetch requests
 * to make MAU User Activity event tracking easier by identifying requests
 * from inactive tabs/windows
 *
 * @module atlassian/analytics/user-activity-xhr-header
 *
 * @see module:atlassian/analytics/page-visibility
 */
define('atlassian/analytics/user-activity-xhr-header', ['atlassian/analytics/page-visibility'], function (pageVisibility) {

    var ACTIVITY_IGNORE_HEADER = 'x-atlassian-mau-ignore';

    // Store native xhr and fetch methods
    var oldSend = XMLHttpRequest.prototype.send;
    var oldFetch = window.fetch;

    var _installed = false;

    return /** @alias module:atlassian/analytics/user-activity-xhr-header */ {
        /**
         * Install global overrides to XHR send and fetch methods to add a
         * custom HTTP header that makes MAU User Activity event tracking
         * easier by identifying requests from inactive tabs/windows
         */
        install: function () {
            // Only install if the override is not already installed
            // and if the browser supports the Page Visibility API
            if (!_installed && pageVisibility.supported) {

                // Override the native send()
                XMLHttpRequest.prototype.send = function () {
                    if (pageVisibility.isHidden()) {
                        this.setRequestHeader(ACTIVITY_IGNORE_HEADER, pageVisibility.isHidden());
                    }
                    oldSend.apply(this, arguments);
                };

                // Override the native fetch() if it exists
                if (oldFetch) {
                    window.fetch = function (input, init) {
                        var augmentedInit = init || {};
                        var headers = augmentedInit.headers;

                        if (pageVisibility.isHidden()) {
                            augmentedInit.headers = (headers) ? new Headers(headers) : new Headers();
                            augmentedInit.headers.set(ACTIVITY_IGNORE_HEADER, pageVisibility.isHidden());
                        }

                        return oldFetch.call(this, input, augmentedInit);
                    };
                }

                _installed = true;
            }
        },

        /**
         * Restore native XHR send and fetch methods
         */
        uninstall: function () {
            if (_installed) {
                XMLHttpRequest.prototype.send = oldSend;
                if (oldFetch) {
                    window.fetch = oldFetch;
                }
            }
            _installed = false;
        }
    };

});

require('atlassian/analytics/user-activity-xhr-header').install();