/**
 * @module confluence-drag-and-drop/default-drop-handler
 *
 */
define('confluence-drag-and-drop/default-drop-handler', [
    'ajs',
    'document',
    'confluence/meta',
    'jquery',
    'confluence-drag-and-drop/uploader',
    'confluence-drag-and-drop/upload-utils',
    'confluence-drag-and-drop/drag-and-drop-utils',
], function (AJS, document, Meta, $, fileUploader, uploadUtils, dragAndDropUtils) {
    'use strict';

    let DragAndDrop = {};
    let initialised = false;

    DragAndDrop.initialise = function () {
        // Prevent this initialisation from running more than once
        if (initialised) {
            return;
        }
        initialised = true;
        fileUploader.init();

        // we want to clear the default uploaded and create our own.
        AJS.DragAndDrop.defaultDropHandler = null;

        /**
         * Check if any popup dialog with transparent blanket is displayed on the current view.
         * This does not count the Drag and Drop progress dialog, as we still allow drag and drop when the
         *   progress dialog appears.
         * @returns {boolean}
         * @private
         */
        function isPopUpDialogVisible() {
            let $blanket = $('.aui-blanket:visible');
            return (
                $blanket.length &&
                $blanket.css('visibility') !== 'hidden' &&
                !$('#drag-and-drop-progress-dialog:visible').length
            );
        }

        dragAndDropUtils.bindDragOver(document.body, function (event){
            event.preventDefault();
        });

        dragAndDropUtils.bindDrop(document.body, function (event) {

            event.preventDefault();

            if (isPopUpDialogVisible()) {
                // cancel uploading all files
                return;
            }

            // CONFDEV-14351 - Prevent main page event handler from fire if the editor is present.
            // In that case specific editor upload handler should be used
            // If the upload dialog is visible, a subsequent drag and drop will trigger this handler as it will be
            // caught by ".aui-blanket". Pass the trigger to the editor's uploader to let it handle the queue etc.

            // This is useful for comments editor uploads. When a file/files are dragged over the application window
            // towards the comment, the viewcontent-init.js code is triggered, and thus the handler in default-drop-handler.
            // We need to specifically trigger the drop handler from editor-drop-handler.js by checking the exact location
            // in the window and finding out if the editor is present, or we can't drop files into the comment editor.
            let parentClasses;
            if (AJS.Editor && AJS.Editor.isVisible()) {
                parentClasses = AJS.Editor.getCurrentForm().parentElement.classList;
                if (
                    fileUploader &&
                    (parentClasses.contains('ic-body') || parentClasses.contains('comment-body'))
                ) {
                    let dropHandler = fileUploader.getEditorHtmlNode().ondrop;
                    if (typeof dropHandler === 'function') {
                        dropHandler.call(fileUploader.getEditorHtmlNode(), event);
                    }
                }

                // Since the editor-drop-handler.js is called for handling, we can simply return here. No need
                // to upload again below.
                return;
            }

            var files = uploadUtils.getFilesFromDropEvent(event);

            fileUploader.upload(files);
        });
    };

    return DragAndDrop;
});
