/**
 * @tainted Confluence
 * @tainted tinymce
 */
define('confluence-drag-and-drop/editor-drop-handler', [
    'ajs',
    'jquery',
    'confluence-drag-and-drop/analytics/files-upload-analytics',
    'confluence-drag-and-drop/drag-and-drop-overlay',
    'document',
    'window',
    'confluence/meta',
    'confluence-drag-and-drop/uploader',
    'confluence-drag-and-drop/upload-utils',
    'confluence-drag-and-drop/drag-and-drop-utils',
], function (
    AJS,
    $,
    filesUploadAnalytics,
    dragAndDropOverlay,
    document,
    window,
    Meta,
    fileUploader,
    uploadUtils,
    dragAndDropUtils
) {
    'use strict';

    let DragAndDrop = {};

    // I'm sure there is a better way to check this... but it's all I could think of at the moment.
    function isFullscreenMode() {
        return $(document.body).hasClass('contenteditor');
    }

    function isCQ() {
        return Meta.get('content-type') === 'com.atlassian.confluence.plugins.confluence-questions:question';
    }

    DragAndDrop.initialise = function () {
        /**
         * Why are we switching between nodes when in fullscreen or not?
         *
         * when the editor is fullscreen,
         * A) the "drop files here" overlay element gets in the way of the drop target
         * B) to cater to anyone who drops their files on the toolbar or app chrome rather than in the editor
         *
         * otherwise, in places like comments,
         * C) The "default" drop handler is likely loaded, which would result in double-handling of the drop event
         *
         * When you also consider the `if (AJS.Editor)` conditions in the "default" drop handler,
         * this awkward logic should probably be refactored. Maybe in to a strategy pattern, and use a stack
         * of handlers so the editor's can take precedence when needed? Or, perhaps the "default" and "editor" handlers
         * could be merged in to a single unified handler!
         */
        let isUsesEditor = !(isFullscreenMode() || isCQ());
        let editorHtmlNode;
        if (isUsesEditor && !AJS.Rte.getEditor()) {
            // the editor has been deactivated - no point to continue
            return;
        }
        editorHtmlNode = !isUsesEditor ? document.body : AJS.Rte.getEditor().getBody().parentNode;

        fileUploader.init();

        // Store editorHtmlNode in uploader.js
        fileUploader.setEditorHtmlNode(editorHtmlNode);

        // Need to set this. This uploader is used in confluence-editor plugin.
        // Specifically in this path
        // confluence-editor/p2-plugin/frontend/confluence-editor/src/editor-plugins/autoconvert/clipboard-access.ts
        AJS.Confluence.Uploader = fileUploader;

        dragAndDropUtils.bindDragOver(editorHtmlNode, function (event){
            event.preventDefault();
        });

        function handleEditorDrop(event) {
            event.preventDefault();
            event.stopPropagation(); // Stop propagation of the drop event
            let files = uploadUtils.getFilesFromDropEvent(event);
            filesUploadAnalytics.triggerEvent('confluence.editor.files.drag-and-drop', files);
            fileUploader.upload(files);
        }

       // dragAndDropUtils.bindDrop(editorHtmlNode, handleEditorDrop);

        // This is necessary to handle comment editor uploads. This code is used in default-drop-handler.js
        // bindDrop function. Check comment there for more details. So we can't use dragAndDropUtils.bindDrop
        // here, so we have to take care of the propagation here.
        editorHtmlNode.ondrop = handleEditorDrop;
    };

    DragAndDrop.initialiseOverlay = function () {
        let editor = AJS.Rte.getEditor();
        let editorHtmlNode;
        // the editor might have been deactivated at this point
        if (!editor) {
            return;
        }
        editorHtmlNode = editor.getBody().parentNode;

        // in lighter theme, editor title element #editor-precursor overlaps RTE, so need to bind overlay on that too
        dragAndDropOverlay.bindFileDragOverlay({
            $dragZone: $(editorHtmlNode).add('#toolbar ~ #editor-precursor'),
            $overlayZone: $('#rte'),
            isTransparent: true,
        });
    };

    return DragAndDrop;
});
