/**
 * @module confluence-drag-and-drop/image-dialog-client
 *
 */
define('confluence-drag-and-drop/image-dialog-client', [
    'jquery',
    'confluence/legacy',
    'ajs',
    'confluence/meta',
    'underscore',
    'confluence/message-controller',
    'confluence/api/logger',
    'confluence-drag-and-drop/upload-utils',
], function ($, Confluence, AJS, Meta, _, MessageController, logger, uploadUtils) {
    'use strict';

    /** ***************************************************************** */
    /* Because z-index of Insert File Dialog is 3004 so 3005 is used for drag overlay in Insert File Dialog */
    let DRAG_AND_DROP_Z_INDEX = 3005;

    let BatchProgress = function () {
        this._workIdToBytesUploaded = {};
        this._totalBytes = 0;
    };

    BatchProgress.prototype = {
        update: function (workId, uploaded, fileSize) {
            if (!(workId in this._workIdToBytesUploaded)) {
                this._totalBytes += fileSize;
            }
            this._workIdToBytesUploaded[workId] = uploaded;
        },
        percentComplete: function () {
            var totalBytesUploaded = 0;
            $.each(this._workIdToBytesUploaded, function (key, value) {
                totalBytesUploaded += value;
            });
            return Math.round((totalBytesUploaded * 100) / this._totalBytes);
        },
    };

    /** ***************************************************************** */
    /*
     * Support upload images (and other file types in near future) in client side
     * It replies on Plupload plugin
     * 'attachmentPanelComponent' is an instance of Confluence.Editor.FileDialog.AttachmentsPanel
     * In this class, we should not reply on DOM manipulation. Let 'attachmentPanelComponent' do these DOM jobs
     *
     * */
    let UploadClient = function (options) {
        this.urlHandleUpload = '/plugins/drag-and-drop/upload.action';
        this.pageId = parseInt(Meta.get('page-id'), 10);
        this.draftId = parseInt(Meta.get('draft-id'), 10);
        this.dragAndDropEntityId = Meta.get('drag-and-drop-entity-id');
        this.spaceKey = Meta.get('space-key') || '';
        this.atlToken = Meta.get('atl-token');
        this.base = /^\w+:\/\/[^/?#]+/.exec(location.href); // get base url from current url
        this.metaMaxFileSize = Meta.get('global-settings-attachment-max-size');
        this.errors = [];
        this.batchProgress = new BatchProgress();
        this.attachmentPanelComponent = options.attachmentPanelComponent;
        this.dropZone = options.dropZone;
        this.browserButtonId = options.browserButtonId;
        this.uploader = null;
        this.isUploadUsingDragAndDrop = false; // this variable is used to distinguish between uploading file using dragAndDop or upload button
    };

    UploadClient.prototype.createUploader = function () {
        var that = this;
        var uploader = {};

        // refresh everything when start
        that.errors = [];
        that.attachmentPanelComponent.clearErrors();
        that.batchProgress = new BatchProgress();
        that.metaMaxFileSize = Meta.get('global-settings-attachment-max-size');

        uploader.filterFilesBeforeUpload = function(files) {
            let filteredFiles = [];
            if (files.length > 0) {
                that.attachmentPanelComponent.getNoFileContainer().hide();
                filteredFiles = AJS.UploadUtils.filterFiles(files, that.metaMaxFileSize);
                for (var i = 0; i < filteredFiles.length; i++) {
                    if (!filteredFiles[i].hasError) {
                        if (that.metaMaxFileSize > filteredFiles[i].file.size) {
                            // there's a dependency here with editor code in confluence. It appears that
                            // code was written with plupload in mind. Adding required details for now. Will be removed
                            // once the changes are made there as well
                            filteredFiles[i].file.nativeFile = filteredFiles[i].file;
                            filteredFiles[i].file.id = filteredFiles[i].workId;
                            that.attachmentPanelComponent.addPreviewImage(filteredFiles[i].file);
                        }
                    }
                }
            }

            return filteredFiles;
        };

        uploader.beforeUpload = function(filteredFile) {
            var upload_url;
            var url = that.base + AJS.contextPath() + that.urlHandleUpload;
            // it was doing some odd things with coercion hence the explict check
            var params = {};
            var extension = filteredFile.file.name.substr(filteredFile.file.name.lastIndexOf('.') + 1);
            if (that.pageId) {
                params.pageId = that.pageId;
            } else {
                params.draftId = that.draftId;
            }

            if (that.dragAndDropEntityId) {
                params.dragAndDropEntityId = that.dragAndDropEntityId;
            }

            params.filename = filteredFile.file.name;
            params.size = filteredFile.file.size;
            // if we are in the image dialog flag all attachments as hidden
            params.minorEdit = true;
            params.spaceKey = that.spaceKey;
            params.mimeType = AJS.UploadUtils.mimeTypes[extension.toLowerCase()] || AJS.DragAndDropUtils.defaultMimeType;
            params.atl_token = that.atlToken;
            params.name = filteredFile.file.name;
            upload_url = AJS.UploadUtils.buildUrl(url, params);

            Confluence.Editor.FileDialog.eventListener.on('uploadingfile.cancelled', function (filteredFile) {
                filteredFile.xhrConnection.abort();
                that.attachmentPanelComponent.attachmentUploadingCancelled(filteredFile.workId);
            });

            that.batchProgress = new BatchProgress();
            return upload_url;
        };

        uploader.uploadFile = function(filteredFile) {
            if (!filteredFile || !filteredFile.file) {
                logger.log("Invalid filteredFile object passed to image client dialog.");
                return;
            }

            var upload_url = uploader.beforeUpload(filteredFile);
            if (upload_url) {
                return new Promise(function (resolve, reject) {
                    filteredFile.xhrConnection = new XMLHttpRequest();
                    filteredFile.xhrConnection.open("POST", upload_url, true);

                    // Set up progress event listener
                    filteredFile.xhrConnection.upload.onprogress = function (event) {
                        if (event.lengthComputable) {
                            var bytesUploaded = event.loaded;
                            try {
                                that.batchProgress.update(filteredFile.file.workId, bytesUploaded, filteredFile.file.size);
                                that.attachmentPanelComponent.setUploadInProgress(that.batchProgress.percentComplete() / 100, filteredFile.file.workId);
                            } catch (error) {
                                logger.log("Error updating progress:", error);
                                filteredFile.errorCode = AJS.UploadUtils.ErrorCode.GENERIC_ERROR;
                                filteredFile.errorMessage = error.errorMessage;
                                filteredFile.hasError = true;
                                reject(filteredFile);
                            }
                        }
                    };

                    // Set up load event listener for completion or HTTP errors
                    filteredFile.xhrConnection.onload = function () {
                        try {
                            if (filteredFile.xhrConnection.status>= 200 && filteredFile.xhrConnection.status < 300) {
                                var data = JSON.parse(filteredFile.xhrConnection.response);
                                var fileServerId = data.data.id;
                                that.attachmentPanelComponent.attachmentUploaded(filteredFile.workId, fileServerId);
                                resolve(filteredFile.xhrConnection.response);
                            } else {
                                filteredFile.errorCode = AJS.UploadUtils.ErrorCode.HTTP_ERROR;
                                filteredFile.errorMessage = filteredFile.xhrConnection.response.responseText;
                                filteredFile.hasError = true;
                                reject(filteredFile);
                            }
                        } catch (error) {
                            logger.log("Error completing progress:", error);
                            filteredFile.errorCode = AJS.UploadUtils.ErrorCode.GENERIC_ERROR;
                            filteredFile.errorMessage = error.errorMessage;
                            filteredFile.hasError = true;
                            reject(filteredFile);
                        }
                    };

                    // Set up error event listener
                    filteredFile.xhrConnection.onerror = function () {
                        filteredFile.errorCode = AJS.UploadUtils.ErrorCode.HTTP_ERROR;
                        filteredFile.errorMessage = filteredFile.xhrConnection.response.responseText;
                        filteredFile.hasError = true;
                        reject(filteredFile);
                    };

                    // Prepare and send the file
                    filteredFile.xhrConnection.setRequestHeader('Content-Type', 'application/octet-stream');
                    filteredFile.xhrConnection.send(filteredFile.file);
                });
            }
        };

        uploader.handleErrorScenarios = function(errorFilteredFile) {
            if (!errorFilteredFile.file || !errorFilteredFile.file.name) {
                console.error('Invalid file object passed to handleErrorScenarios:', errorFilteredFile);
                return;
            }

            var result;
            var message;
            var response = null;
            if (errorFilteredFile && errorFilteredFile.xhrConnection) {
                response = errorFilteredFile.xhrConnection.response;
            }

            if (response) {
                try {
                    // only http errors
                    result = response ? JSON.parse(response) : null;
                    message = result.actionErrors[0];
                } catch (e) {
                    message = MessageController.parseError(errorFilteredFile.xhrConnection);
                }
                errorFilteredFile.errorMessage = message;
                AJS.trigger('analyticsEvent', { name: 'confluence.image-dialog.upload.error.server-unknown' });
            } else {
                message = errorFilteredFile.errorMessage;
                var fileName = errorFilteredFile.file.name;
                if (errorFilteredFile.errorCode === AJS.UploadUtils.ErrorCode.FILE_SIZE_ERROR) {
                    var fileSizeLimit = AJS.DragAndDropUtils.niceSize(that.metaMaxFileSize).toString();
                    message = AJS.I18n.getText('dnd.validation.file.too.large.withname', fileName, fileSizeLimit);
                    AJS.trigger('analyticsEvent', { name: 'confluence.image-dialog.upload.error.file-size' });
                } else if (errorFilteredFile.errorCode === AJS.UploadUtils.ErrorCode.FILE_IS_A_FOLDER_ERROR ||
                    errorFilteredFile.errorCode === AJS.UploadUtils.ErrorCode.FILE_EXTENSION_ERROR) {
                    message = AJS.I18n.getText('dnd.validation.file.type.not.supported.withname', fileName);
                    AJS.trigger('analyticsEvent', { name: 'confluence.image-dialog.upload.error.file-type' });
                } else if (errorFilteredFile.errorCode === AJS.UploadUtils.ErrorCode.TEMPLATE_NOT_SUPPORTED) {
                    message = AJS.I18n.getText('dnd.templates.not.supported.message');
                    AJS.trigger('analyticsEvent', { name: 'confluence.image-dialog.upload.error.file-type' });
                }
                else {
                    message = MessageController.parseError(errorFilteredFile.xhrConnection);
                    AJS.trigger('analyticsEvent', { name: 'confluence.image-dialog.upload.error.server-unknown' });
                }
                errorFilteredFile.errorMessage = message;
            }

            that.attachmentPanelComponent.displayErrors([message]);
            that.errors.push(message);
        };

        uploader.startUpload = async function(files) {

            // trigger file upload event
            require(['confluence-drag-and-drop/analytics/files-upload-analytics'], function (filesUploadAnalytics) {
                filesUploadAnalytics.triggerEvent(
                    that.isUploadUsingDragAndDrop ? 'confluence.insert-files-dialog.drag-and-drop'
                        : 'confluence.insert-files-dialog.upload',
                    files
                );

                that.isUploadUsingDragAndDrop = false;
            });

            var filteredFiles = uploader.filterFilesBeforeUpload(files, this.metaMaxFileSize);
            var result;

            for (var fileIndex = 0; fileIndex < filteredFiles.length; fileIndex++) {
                try {
                    if (!filteredFiles[fileIndex].hasError) {
                        result = await uploader.uploadFile(filteredFiles[fileIndex]);
                        logger.log(result);
                    } else {
                        // Send the filteredFile obj to handleErrorScenarios function directly.
                        uploader.handleErrorScenarios(filteredFiles[fileIndex]);
                    }
                } catch (errorFilteredFileObj) {
                    uploader.handleErrorScenarios(errorFilteredFileObj);
                }
            }
        };

        require(['confluence-drag-and-drop/drag-and-drop-overlay'], function (dragAndDropOverlay) {
            dragAndDropOverlay.bindFileDragOverlay({ $dragZone: $(that.dropZone), zIndex: DRAG_AND_DROP_Z_INDEX });
        });

        return uploader;
    };

    UploadClient.prototype.init = function () {
        this.uploader = this.createUploader();

        // Handle drag and drop scenarios.
        AJS.DragAndDropUtils.bindDragOver(this.dropZone, function (event){
            event.preventDefault();
        });

        // bind(this) is required because within the addEventListener code, the context for 'this'
        // changes to the fileInput, while outside it, this refers to the uploadClient, and we want
        // that to access the uploader code.
        AJS.DragAndDropUtils.bindDrop(this.dropZone, function (event) {
            event.preventDefault();
            this.isUploadUsingDragAndDrop = true;

            let files = uploadUtils.getFilesFromDropEvent(event);

            this.uploader.startUpload(files);
        }.bind(this));

        // handle files being selected from browse button.
        if (this.browserButtonId) {
            var browserButton = document.getElementById(this.browserButtonId);
            if (browserButton) {

                // create a file input element
                var fileInput = document.createElement('input');
                fileInput.type = 'file';
                fileInput.multiple = true; // Allow multiple file selection
                // Position and style the file input to cover the browser button
                // this class extra is in image-dialog-client-browser-button.css file.
                fileInput.className = 'file-library-input-file file-library-input-file-extra';

                browserButton.style.position = 'relative';
                browserButton.appendChild(fileInput);

                // bind(this) is required because within the addEventListener code, the context for 'this'
                // changes to the fileInput, while outside it, this refers to the uploadClient, and we want
                // that to access the uploader code.
                fileInput.addEventListener('change', function (event) {
                    // Get files from the input element
                    var files = Array.from(event.target.files);
                    if (files.length > 0) {
                        this.uploader.startUpload(files); // Pass files to startUpload
                    }
                    event.target.value = '';
                }.bind(this));
            }
        }
    };

    /** ***************************************************************** */

    /**
     * In modern browsers, we will use a new method of upload
     */
    if (Confluence.Editor && Confluence.Editor.FileDialog) {
        Confluence.Editor.FileDialog.eventListener.on('AttachmentsPanelView.render', function (viewObj) {
            viewObj.getUploaderController = function () {
                return new Confluence.DragAndDrop.UploadClientController(viewObj.context);
            };
        });
    }
    /** ***************************************************************** */

    AJS.toInit(function () {
        if (!Confluence.Editor || !Confluence.Editor.ImageDialog) {
            logger.debug(
                'Either editor or image dialog missing',
                Confluence.Editor,
                Confluence.Editor && Confluence.Editor.ImageDialog
            );
            return;
        }

        /**
         * Ensure we register listeners outside of toInit(). This is to ensure they get registered before they are invoked
         * in application logic that does reside inside toInit().
         */
        var bsl = Confluence.Editor.ImageDialog.beforeShowListeners;

        bsl &&
            bsl.push(function () {
                var attachmentPanelComponent;
                var $attachedImages;
                var uploadClient;

                attachmentPanelComponent = Confluence.Editor.ImageDialog.findPanelComponentById('attachments');
                if (!attachmentPanelComponent) {
                    logger.debug(
                        'Do not support Attachment Panel and Drag-Drop in Insert Image Dialog (ex: in creating Template)'
                    );
                    return;
                }

                $attachedImages = attachmentPanelComponent.getPanelElement();
                if (!$attachedImages.length) {
                    logger.debug('no attach image dialog so no need to support uploading');
                    return;
                }

                if (Meta.getBoolean('can-attach-files')) {
                    uploadClient = new UploadClient({
                        dropZone: $attachedImages[0],
                        browserButtonId: 'upload-files-button',
                        attachmentPanelComponent: attachmentPanelComponent,
                    });
                    uploadClient.init();
                }
            });
    });

    return UploadClient;
});

require('confluence/module-exporter').exportModuleAsGlobal(
    'confluence-drag-and-drop/image-dialog-client',
    'Confluence.DragAndDrop.UploadClient'
);
