/**
 * @module confluence-drag-and-drop/upload-utils
 */
define('confluence-drag-and-drop/upload-utils', ['ajs'], function (AJS) {
    'use strict';

    /**
     * 4096 bytes is the smallest allocation unit for filesystems.
     * Then OS reserved the size of directory less than or equals to 4096 bytes
     */
    let LIMITED_SIZE_OF_FOLDER = 4096; // bytes

    let ErrorCode = {
        FILE_SIZE_ERROR: 'FILE_SIZE_ERROR',
        FILE_EXTENSION_ERROR: 'FILE_EXTENSION_ERROR',
        FILE_IS_A_FOLDER_ERROR: 'FILE_IS_A_FOLDER_ERROR',
        GENERIC_ERROR: 'GENERIC_ERROR',
        HTTP_ERROR: 'HTTP_ERROR',
        INVALID_FILE_TYPE_ERROR: 'INVALID_FILE_TYPE_ERROR',
        TEMPLATE_NOT_SUPPORTED: 'TEMPLATE_NOT_SUPPORTED',
    };

    let mimeTypes = {
        doc: 'application/msword',
        dot: 'application/msword',
        pdf: 'application/pdf',
        pgp: 'application/pgp-signature',
        ps: 'application/postscript',
        ai: 'application/postscript',
        eps: 'application/postscript',
        rtf: 'application/rtf',
        xls: 'application/vnd.ms-excel',
        xlb: 'application/vnd.ms-excel',
        ppt: 'application/vnd.ms-powerpoint',
        pps: 'application/vnd.ms-powerpoint',
        pot: 'application/vnd.ms-powerpoint',
        zip: 'application/zip',
        jar: 'application/java-archive',
        war: 'application/java-archive',
        ear: 'application/java-archive',
        swf: 'application/x-shockwave-flash',
        swfl: 'application/x-shockwave-flash',
        xlsx: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        xltx: 'application/vnd.openxmlformats-officedocument.spreadsheetml.template',
        pptx: 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
        ppsx: 'application/vnd.openxmlformats-officedocument.presentationml.slideshow',
        potx: 'application/vnd.openxmlformats-officedocument.presentationml.template',
        docx: 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        dotx: 'application/vnd.openxmlformats-officedocument.wordprocessingml.template',
        mpga: 'audio/mpeg',
        mpega: 'audio/mpeg',
        mp2: 'audio/mpeg',
        mp3: 'audio/mpeg',
        wav: 'audio/x-wav',
        m4a: 'audio/mp4',
        bmp: 'image/bmp',
        gif: 'image/gif',
        jpeg: 'image/jpeg',
        jpg: 'image/jpeg',
        jpe: 'image/jpeg',
        psd: 'image/photoshop',
        png: 'image/png',
        svg: 'image/svg+xml',
        svgz: 'image/svg+xml',
        tiff: 'image/tiff',
        tif: 'image/tiff',
        htm: 'text/html',
        html: 'text/html',
        xhtml: 'text/html',
        xml: 'text/xml',
        js: 'text/javascript',
        css: 'text/css',
        java: 'text/x-java-source',
        mpeg: 'video/mpeg',
        mpg: 'video/mpeg',
        mpe: 'video/mpeg',
        qt: 'video/quicktime',
        mov: 'video/quicktime',
        mp4: 'video/mp4',
        m4v: 'video/x-m4v',
        flv: 'video/x-flv',
        wmv: 'video/x-ms-wmv',
        avi: 'video/avi',
        webm: 'video/webm',
        rv: 'video/vnd.rn-realvideo',
        csv: 'text/csv',
        asc: 'text/plain',
        txt: 'text/plain',
        text: 'text/plain',
        diff: 'text/plain',
        log: 'text/plain',
        exe: 'application/octet-stream',
    };

    function generateUUID() {
        return Math.random().toString(36).toUpperCase().substr(2, 9);
    }

    /**
     * This method is used for constructing the query included upload url.
     *
     * @param baseUrl the base url of the confluence server
     * @param params the list of params to be added as query parameters
     */
    function buildUrl (baseUrl, params) {
        let query = [];
        for (let key in params) {
            if (params.hasOwnProperty(key)) {
                query.push(encodeURIComponent(key) + "=" + encodeURIComponent(params[key]));
            }
        }
        query = query.join("&");
        return query ? baseUrl + (baseUrl.indexOf("?") !== -1 ? "&" : "?") + query : baseUrl;
    }

    /**
     * This method is used almost as a class for the filtered file object, that contains upload corresponding properties of the file
     * @param file
     * @param hasError
     * @param errorMessage
     * @param errorCode
     * @returns {{file, xhrConnection: null, errorMessage, progressHandler: null, errorCode, hasError, workId: string, uploadProgressDialog: null}}
     */
    function filteredFile(file, hasError, errorMessage, errorCode) {
        const fileObject = {
            file: file,
            workId: generateUUID(),
            hasError: hasError,
            errorMessage: errorMessage,
            errorCode: errorCode,
            progressHandler: null,
            uploadProgressDialog: null,
            xhrConnection: null,
        };

        return fileObject;
    }

    /**
     * This method is used to filter the array of files, and call the callback function with an array of filtered file functions that has
     * information about the properties of the file.
     * If it's a folder, then it will fire an 'Error' event with the ErrorCode is FILE_IS_A_FOLDER_ERROR.
     *
     * @param files Array of files need to be verified
     * @param max_file_size_for_upload The maximum allowed size for file upload. Set by confluence admin
     */
    function filterFiles (files, max_file_size_for_upload) {
        const filteredFiles = [];

        for (let i = 0; i < files.length; i++) {
            if (files[i].isDirectory) {
                filteredFiles.push(filteredFile(files[i], true, 'File is a folder', ErrorCode.FILE_IS_A_FOLDER_ERROR));
            } else {
                // file size above max allowed size set by confluence admin
                if (files[i].size > max_file_size_for_upload) {
                    filteredFiles.push(filteredFile(files[i], true, 'File size error.', ErrorCode.FILE_SIZE_ERROR));
                }
                // We should be checking for whether file extensions are allowed or not,
                // but the plugin didn't check when it was using plupload.
                // So we're maintaining the same behaviour.
                else {
                    filteredFiles.push(filteredFile(files[i], false, null, null));
                }
            }

        }
        return filteredFiles;
    }

    /**
     * This function handles the upload progress for a filtered file object.
     *
     * @param filteredFile The filtered file object containing the file and upload progress dialog along with other properties
     */
    function UploadProgressHandler (filteredFile) {
        if (!filteredFile || !filteredFile.file) {
            console.error("Invalid filteredFile object passed to UploadProgressHandler.");
            return null;
        }

        function updateProgress(bytesUploaded) {
            filteredFile.uploadProgressDialog.renderUpdateToBytesUploaded(filteredFile.workId, bytesUploaded, filteredFile.file.size);
        }

        function completeProgress() {
            console.log("Completing progress for:", filteredFile.file.name);
            filteredFile.uploadProgressDialog.renderUpdateToBytesUploaded(filteredFile.workId, filteredFile.file.size, filteredFile.file.size);
            filteredFile.uploadProgressDialog.renderComplete(filteredFile.workId);
        }

        function cancelUpload() {
            filteredFile.xhrConnection.abort();
            if (!filteredFile.uploadProgressDialog.isVisible()) {
                filteredFile.uploadProgressDialog.show();
            }
            filteredFile.uploadProgressDialog.renderInfo(
                filteredFile.workId,
                AJS.I18n.getText('dnd.upload.file.removed.from.queue')
            );
            filteredFile.uploadProgressDialog.showCloseButton();
        }

        return {
            updateProgress: function (bytesUploaded) {
                try {
                    updateProgress(bytesUploaded);
                } catch (error) {
                    console.error("Error in updateProgress:", error);
                }
            },
            completeProgress: function () {
                try {
                    completeProgress();
                } catch (error) {
                    console.error("Error in completeProgress:", error);
                }
            },
            cancelUpload: function () {
                try {
                    cancelUpload();
                } catch (error) {
                    console.error("Error in cancelUpload:", error);
                }
            }
        };
    }

    function getFilesFromDropEvent(event) {
        let items = event.dataTransfer?.items || [];
        let files = [];

        if (items) {
            for (let i = 0; i < items.length; i++) {
                let item = items[i].webkitGetAsEntry();
                if (item && item.isDirectory) {
                    files.push(item);
                } else {
                    files.push(items[i].getAsFile());
                }
            }
        } else {
            files = Array.from(event.dataTransfer.files);
        }

        return files;
    }

    return {
        ErrorCode: ErrorCode,

        // Define the MIME types and their corresponding extensions
        mimeTypes: mimeTypes,

        buildUrl: buildUrl,

        filteredFile: filteredFile,

        filterFiles: filterFiles,

        UploadProgressHandler: UploadProgressHandler,

        getFilesFromDropEvent: getFilesFromDropEvent
    };
});

require('confluence/module-exporter').exportModuleAsGlobal('confluence-drag-and-drop/upload-utils', 'AJS.UploadUtils');