/**
 * @module confluence-drag-and-drop/uploader
 */
define('confluence-drag-and-drop/uploader', [
    'ajs',
    'document',
    'confluence/meta',
    'jquery',
    'confluence/legacy',
    'window',
    'confluence/api/constants',
    'confluence/api/event',
    'confluence/api/logger',
    'confluence/message-controller',
    'confluence-drag-and-drop/upload-utils',
    'confluence-drag-and-drop/drag-and-drop-utils',
], function (AJS, document, Meta, $, Confluence, window, constants, event, logger,
             MessageController, uploadUtils, dragAndDropUtils) {
    'use strict';

    // Store a reference for the progress dialog
    let uploadProgressDialog;
    let initialised = false;

    const max_file_size_for_upload = +Meta.get('global-settings-attachment-max-size');
    let isUploading = false;
    let buffer = [];
    let editorHtmlNode = null;
    let dropLocation = false;

    /**
     * Includes protocol, host, port (excludes context path and trailing slash)
     */
    const base = /^\w+:\/\/[^/?#]+/.exec(window.location.href);

    // If the user has already been alerted that templates are not supported, don't alert them again.
    let alertedTemplatesNotSupported = false;

    const dndCustomEvents = {
        FILE_ATTACHED: 'FILE_ATTACHED',
        ALL_FILES_ATTACHED: 'ALL_FILES_ATTACHED'
    };

    function isEditorVisible() {
        return AJS.Editor && AJS.Editor.isVisible();
    }

    function getAnalyticsText() {
        return isEditorVisible() ? 'confluence.editor.upload' : 'confluence.default-drop.upload';
    }

    function alertTemplatesNotSupportedOnce() {
        let popup;
        if (alertedTemplatesNotSupported) {
            return;
        }
        popup = new AJS.Dialog(450, 180);
        popup.addHeader(AJS.I18n.getText('dnd.templates.not.supported.heading'));
        popup.addSubmit(AJS.I18n.getText('dnd.templates.not.supported.ok'), function () {
            popup.remove();
        });
        popup.addPanel('Panel 1', 'panel1');
        popup.getCurrentPanel().html(AJS.I18n.getText('dnd.templates.not.supported.message'));
        popup.show();
        alertedTemplatesNotSupported = true;
    }

    /**
     * This function filters the original set of files dropped for upload, sets the
     * corresponding metadata related to the file in terms of upload. Check upload-utils.js
     * @param files the set of files being uploaded via drag and drop
     * @returns {[]} the array of fitleredFiles objects.
     */
    function filterFilesBeforeUpload (files) {
        let filteredFiles = uploadUtils.filterFiles(files, max_file_size_for_upload);

        for (let i = 0; i < filteredFiles.length; i++) {
            let currentFile = filteredFiles[i];
            uploadProgressDialog.render(currentFile);
            // this is a bit of a hack, but we need to pass the uploadProgressDialog to the uploadProgress method
            // So we pass the reference to the uploadProgressDialog to each of the filteredFile instances.
            currentFile.uploadProgressDialog = uploadProgressDialog;
        }

        // Return the filtered files
        return filteredFiles;
    }

    /**
     * This function returns the upload url, after appending the required query parameters to the base url.
     * @param filteredFile object that contains the actual file and corresponding metadata. Created in UploadUtils class..
     * @returns upload url.
     */
    function beforeUpload (filteredFile) {
        let upload_url = null;
        let url;
        if (isEditorVisible()) {
            url = base;
        } else {
            url = dragAndDropUtils.base;
        }
        url = url + constants.CONTEXT_PATH + '/plugins/drag-and-drop/upload.action';

        let dragAndDropEntityId;

        let pageId = parseInt(Meta.get('page-id'));
        let params = pageId !== 0 ? { pageId: pageId } : { draftId: parseInt(Meta.get('draft-id')) };
        let currentFile = filteredFile.file;
        if (isEditorVisible()){
            dragAndDropEntityId = Meta.get('drag-and-drop-entity-id');
            if (dragAndDropEntityId) {
                params.dragAndDropEntityId = dragAndDropEntityId;
            }
        }

        if (currentFile) {
            let extension = currentFile.name.substr(currentFile.name.lastIndexOf('.') + 1);
            params.filename = currentFile.name;
            params.size = currentFile.size;
            // if we dont have the mime type just send a default
            params.mimeType = uploadUtils.mimeTypes[extension.toLowerCase()] || dragAndDropUtils.defaultMimeType;
            params.spaceKey = Meta.get('space-key') || '';
            params.atl_token = Meta.get('atl-token');
            if (isEditorVisible()) {
                // if we are in the editor flag all attachments as hidden
                params.minorEdit = true;
                params.contentType = Meta.get('content-type');
                params.isVFMSupported = !!AJS.MacroBrowser.getMacroMetadata('view-file');
            }
            params.name = currentFile.name;
            upload_url = uploadUtils.buildUrl(url, params);

            uploadProgressDialog.cancelListeners.push(function (e, filteredFile) {
                filteredFile.progressHandler.cancelUpload();
            });
            uploadProgressDialog.show();
        }

        return upload_url;
    }

    /**
     * Uploads a file with progress tracking.
     * @param filteredFile object that contains the actual file and corresponding metadata. Created in UploadUtils class.
     */
    function uploadFileWithProgress (filteredFile, upload_url) {
        if (!filteredFile || !filteredFile.file) {
            logger.log("Invalid filteredFile object passed to uploadFileWithProgress.");
            return;
        }

        if (upload_url) {
            return new Promise(function (resolve, reject) {
                filteredFile.xhrConnection = new XMLHttpRequest();
                filteredFile.progressHandler = uploadUtils.UploadProgressHandler(filteredFile);

                filteredFile.xhrConnection.open("POST", upload_url, true);

                // Set up progress event listener
                filteredFile.xhrConnection.upload.onprogress = function (event) {
                    if (event.lengthComputable) {
                        let bytesUploaded = event.loaded;
                        try {
                            filteredFile.progressHandler.updateProgress(bytesUploaded);
                        } catch (error) {
                            logger.log("Error updating progress:", error);
                            filteredFile.errorCode = uploadUtils.ErrorCode.GENERIC_ERROR;
                            filteredFile.errorMessage = error.errorMessage;
                            filteredFile.hasError = true;
                            reject(filteredFile);
                        }
                    }
                };

                // Set up load event listener for completion or HTTP errors
                filteredFile.xhrConnection.onload = function () {
                    if (!filteredFile.xhrConnection.status) {
                        // this happens if the server suddenly became unavailable and there is no response
                        uploadProgressDialog.renderError(filteredFile.workId, AJS.I18n.getText('dnd.error.server.not.responding'));
                    } else if (filteredFile.xhrConnection.status >= 200 && filteredFile.xhrConnection.status < 300) {
                        try {
                            if (isEditorVisible() && filteredFile.xhrConnection.response && filteredFile.file && filteredFile.file.encoding !== 'base64') {
                                // only http errors
                                let result = filteredFile.xhrConnection.response ? JSON.parse(filteredFile.xhrConnection.response) : null;
                                if (
                                    dropLocation &&
                                    $(dropLocation.startElement.parentNode).closest('[contenteditable="true"]', editorHtmlNode)
                                        .length
                                ) {
                                    // there is a drop location and it's within content editable
                                    tinymce.activeEditor.selection.setCursorLocation(
                                        dropLocation.startElement,
                                        dropLocation.startOffset
                                    );
                                }
                                if (result.htmlForEditor.substr(0, 4) === '<img') {
                                    tinymce.confluence.ImageUtils.insertImagePlaceholder(result.htmlForEditor);
                                } else {
                                    AJS.Rte.getEditor().execCommand('mceInsertContent', false, result.htmlForEditor, {
                                        skip_focus: true,
                                    });
                                }

                            }

                            filteredFile.progressHandler.completeProgress();

                            if (isEditorVisible()) {
                                const result = filteredFile.xhrConnection.response ?
                                    filteredFile.xhrConnection
                                    : { response: '' }; // Default to an empty response if not available

                                const fileAttachedEvent = new CustomEvent(dndCustomEvents.FILE_ATTACHED, {
                                    detail: result
                                });

                                editorHtmlNode.dispatchEvent(fileAttachedEvent);
                            }

                            resolve(filteredFile.xhrConnection.response);
                        } catch (error) {
                            logger.log("Error completing progress:", error);
                            filteredFile.errorCode = uploadUtils.ErrorCode.GENERIC_ERROR;
                            filteredFile.errorMessage = error.errorMessage;
                            filteredFile.hasError = true;
                            reject(filteredFile);
                        }
                    } else {
                        filteredFile.errorCode = uploadUtils.ErrorCode.HTTP_ERROR;
                        filteredFile.errorMessage = filteredFile.xhrConnection.response.responseText;
                        filteredFile.hasError = true;
                        reject(filteredFile);
                    }
                };

                // Set up error event listener
                filteredFile.xhrConnection.onerror = function () {
                    filteredFile.errorCode = uploadUtils.ErrorCode.HTTP_ERROR;
                    filteredFile.errorMessage = filteredFile.xhrConnection.response.responseText;
                    filteredFile.hasError = true;
                    reject(filteredFile);
                };

                // Prepare and send the file
                /*
                *
                * For the code below, normally we need to send filteredFile.file, which is the file object.
                * This is the case for all the drag and drop/upload button scenarios. But this code is also
                * used for copy/paste feature from the clipboard, where the files are also uploaded.
                *
                * Here, the file objects sent are not of type 'File', but custom objects that have a getData method
                * that returns the file data. So for this reason, we're checking to see if getData is present,
                * and if it is, we use that to obtain the information. This also means the file is from
                * clipboard-access.ts, if not, we use the file directly.
                *
                * */
                filteredFile.xhrConnection.setRequestHeader('Content-Type', 'application/octet-stream');
                if (filteredFile.file.encoding) {
                    filteredFile.xhrConnection.setRequestHeader('Content-Encoding', filteredFile.file.encoding);
                }
                const dataToSend = filteredFile.file.getData ? filteredFile.file.getData() : filteredFile.file;
                filteredFile.xhrConnection.send(dataToSend);
            });
        }
    }

    /**
     * Handles any error scenarios. Both from before making the xhr network call and while making the network call, including http errors.
     * @param errorFilteredFile obj containing information about the error. Created specifically in the UploadFileWithProgress function.
     */
    function handleErrorScenarios (errorFilteredFile) {
        let result;
        let message;
        let response = null;
        if (errorFilteredFile && errorFilteredFile.xhrConnection) {
            response = errorFilteredFile.xhrConnection.response;
        }

        if (response) {
            try {
                // only http errors
                result = response ? JSON.parse(response) : null;
                message = result.actionErrors[0];
            } catch (e) {
                message = AJS.I18n.getText('dnd.error.invalid.response.from.server');
            }
            errorFilteredFile.errorMessage = message;
            uploadProgressDialog.renderError(errorFilteredFile.workId, message);
            event.trigger('analyticsEvent', { name: getAnalyticsText() + '.error.server-unknown' });
        } else {
            message = errorFilteredFile.errorMessage;
            if (errorFilteredFile.errorCode === uploadUtils.ErrorCode.FILE_SIZE_ERROR) {
                message = AJS.I18n.getText(
                    'dnd.validation.file.too.large',
                    dragAndDropUtils.niceSize(Meta.get('global-settings-attachment-max-size'))
                );
                event.trigger('analyticsEvent', { name: getAnalyticsText() + '.error.file-size' });
            } else if (errorFilteredFile.errorCode === uploadUtils.ErrorCode.FILE_IS_A_FOLDER_ERROR ||
                errorFilteredFile.errorCode === uploadUtils.ErrorCode.FILE_EXTENSION_ERROR) {
                message = AJS.I18n.getText('dnd.validation.file.type.not.supported');
                event.trigger('analyticsEvent', { name: getAnalyticsText() + '.error.file-type' });
            } else if (errorFilteredFile.errorCode === uploadUtils.ErrorCode.TEMPLATE_NOT_SUPPORTED) {
                uploadProgressDialog.renderError(errorFilteredFile.workId, errorFilteredFile.errorMessage);
                event.trigger('analyticsEvent', { name: getAnalyticsText() + '.error.server-unknown' });
            }
            else {
                try {
                    message = MessageController.parseError(errorFilteredFile.xhrConnection);
                } catch (error) {
                    message = errorFilteredFile.errorMessage;
                }

                uploadProgressDialog.renderError(errorFilteredFile.workId, message);
                event.trigger('analyticsEvent', { name: getAnalyticsText() + '.error.server-unknown' });
            }

            if (!uploadProgressDialog) {
                createUploadDialog();
            }
            // to handle cases where the response is null but the error isn't related to file sizes
            errorFilteredFile.errorMessage = message;
            uploadProgressDialog.render(errorFilteredFile);
            if (!uploadProgressDialog.isVisible()) {
                uploadProgressDialog.show();
                uploadProgressDialog.showCloseButton();
            }
        }
    }

    /**
     * Returns the editor HTML node that is used for drag and drop uploads for comment editors
     * @returns {editorHtmlNode}
     */
    function getEditorHtmlNode () {
        return editorHtmlNode;
    }

    /**
     * Sets the editor HTML node that is used for drag and drop uploads for comment editors.
     * @param node
     */
    function setEditorHtmlNode (node) {
        editorHtmlNode = node;

        // dropLocation used by mozilla on file upload to place the file correctly
        if (navigator.userAgent.indexOf('Firefox') > -1) {
            dragAndDropUtils.bindDragEnter(editorHtmlNode, function (e) {
                if (e.rangeParent && e.rangeOffset !== undefined) {
                    dropLocation = { startElement: e.rangeParent, startOffset: e.rangeOffset };
                } else {
                    dropLocation = false;
                }
            });
        }
    }

    /**
     * The main function that performs the file uploading
     * @param files the files to be uploaded
     * @returns {Promise<void>}
     */
    async function upload (files) {
        let filteredFiles = filterFilesBeforeUpload(files);
        let result;

        Array.prototype.push.apply(buffer, filteredFiles);
        if (!isUploading)
        {
            isUploading = true;

            /*
            * The while loop below is the main logic for uploading files. It's being done sequentially (via await) due to a
            * limitation in the backend. It appears, as the files are uploaded, the backend is updating the database
            * with the uploaded files (creating rows) and uses a lock for atomicity reasons. Trying to upload in parallel is
            * causing lock related exceptions.
            *
            * We could make these uploads happen in parallel by storing the Promise returned from the uploadwithProgress function in an array
            * and then settle them using Promise.all or Promise.allSettled. Each call would also have it's own .then()
            * block.
            *
            * Until the backend is updated though, we will upload the files sequentially, which was the behaviour when plupload
            * was used for uploading files.
            *
            * This is the same case in editor-drop-handler.js as well.
            *
            * Regarding the isUploading bool, we are using a buffer (essentially a queue) to store
            * the incoming files to be uploaded. The idea is to store all files in the buffer and work off it, while freeing
            * up the upload function to accept more files to be uploaded. (Check clipboard-access.ts for this use case).
            *
            * Two use cases, here, where the files can be uploaded sequentially or all at once. The bool check
            * ensures the uploading 'while loop' isn't executed multiple times. As long as the buffer has files to be uploaded, the
            * first upload call will take care of the uploading. If the buffer is empty, that iteration of uploading will be completed
            * and the upload progress dialog will be closed.
            *
            * */

            try {
                while (buffer.length > 0)
                {
                    let thisfilteredFile = buffer.shift();
                    try {
                        if (isEditorVisible() && Meta.get('content-type') === 'template') {
                            // Don't support drag and drop for page templates
                            thisfilteredFile.hasError = true;
                            thisfilteredFile.errorCode = uploadUtils.ErrorCode.TEMPLATE_NOT_SUPPORTED;
                            thisfilteredFile.errorMessage = AJS.I18n.getText('dnd.templates.not.supported.message');
                            alertTemplatesNotSupportedOnce();
                        }

                        if (!thisfilteredFile.hasError) {
                            let upload_url = beforeUpload(thisfilteredFile);
                            result = await uploadFileWithProgress(thisfilteredFile, upload_url);
                            logger.log(result);
                        } else {
                            // Send the filteredFile obj to handleErrorScenarios function directly.
                            handleErrorScenarios(thisfilteredFile);
                        }
                    } catch (errorFilteredFileObj) {
                        handleErrorScenarios(errorFilteredFileObj);
                    }
                }
            } finally {
                isUploading = false;
            }


            // Reload the page to show the attachments
            if (uploadProgressDialog) {
                uploadProgressDialog.showCloseButton();
                if (!uploadProgressDialog.hasErrors()) {
                    setTimeout(function () {
                        uploadProgressDialog.hide();
                        uploadProgressDialog.clearRenderOutput();
                        if (isEditorVisible()) {
                            // restore the focus to the editor (but not just that, we want to do better and restore the cursor as well)
                            AJS.Rte.BookmarkManager.restoreBookmark();
                        } else {
                            window.location.reload();
                        }
                    }, 1000);
                }
            }

            if (isEditorVisible()) {
                editorHtmlNode.dispatchEvent(new CustomEvent(dndCustomEvents.ALL_FILES_ATTACHED));
            }
        }
    }

    return {
        /**
         * initializes the fileUploader.
         */
        init: function() {

            // Prevent this initialisation from running more than once
            if (initialised) {
                return;
            }
            initialised = true;

            if (!uploadProgressDialog) {
                uploadProgressDialog = new AJS.DragAndDropProgressDialog();
            }
        },

        upload: upload,

        filterFilesBeforeUpload: filterFilesBeforeUpload,

        beforeUpload: beforeUpload,

        uploadFileWithProgress: uploadFileWithProgress,

        handleErrorScenarios: handleErrorScenarios,

        getEditorHtmlNode: getEditorHtmlNode,

        setEditorHtmlNode: setEditorHtmlNode,

        // This is used in clipboard-access.ts file in confluence editor
        dndCustomEvents: dndCustomEvents
    };
});