/**
 * @module confluence-macro-browser/macro-browser-UI
 */
define('confluence-macro-browser/macro-browser-UI', [
    'jquery',
    'ajs',
    'confluence/legacy',
    'confluence/api/constants'
], function(
    $,
    AJS,
    Confluence,
    CONSTANTS
) {
    'use strict';

    return function(MacroBrowser) {
        var BUTTON_SAVE_SELECTOR = '#macro-browser-dialog .dialog-button-panel .ok';

        function toggleSpinner($elm, show) {
            if (show) {
                $elm.spin('large');
            } else {
                $elm.spinStop();
            }
        }

        function showBrowserSpinner(show) {
            toggleSpinner($('#macro-browser-dialog'), show);
        }

        function showPreviewSpinner(show) {
            toggleSpinner($('#macro-browser-preview'), show);
        }

        function enableSaveButton(enable) {
            var okButton = $(BUTTON_SAVE_SELECTOR);
            okButton.prop('disabled', !enable);
        }

        function updateButtonText(mode) {
            var okButton = $(BUTTON_SAVE_SELECTOR);
            if (mode == 'edit') {
                okButton.text(AJS.I18n.getText('save.name'));
            } else {
                okButton.text(AJS.I18n.getText('insert.name'));
            }
        }

        // private functions
        /**
         * Returns the template for macro summary list
         * @param {int} id Category
         * @returns {*|jQuery|HTMLElement}
         */
        function makeCategoryList(id) {
            return $(Confluence.Templates.MacroBrowser.macroSummaryList({ category: id }));
        }

        /**
         * Creates the DOM element for the macro summary panel and returns it.
         * @param metadata
         * @returns {*|jQuery|HTMLElement}
         */
        function makeMacroSummary(metadata, onclick) {
            var macroDiv = $(Confluence.Templates.MacroBrowser.macroSummaryItem());

            // setting icon
            if (metadata.icon) {
                var iconLocation = (metadata.icon.relative ? AJS.params.staticResourceUrlPrefix : '') + metadata.icon.location;
                macroDiv.prepend('<img src=\'' + iconLocation + '\' alt=\'\' '
                    + 'width=\'' + metadata.icon.width + '\' height=\'' + metadata.icon.height + '\' title=\'' + metadata.title + '\'/>');
            } else {
                macroDiv.prepend('<span class=\'macro-icon-holder icon-' + metadata.macroName + '\'></span>');
            }

            $('.macro-title', macroDiv).text(metadata.title);
            $('.macro-desc', macroDiv).prepend(metadata.description);
            if (metadata.macroName == 'gadget') {
                var url = metadata.gadgetUrl;
                if (url) {
                    if (!url.match('^https?://.*')) {
                        url = CONSTANTS.CONTEXT_PATH + '/' + url;
                    }
                    $('.macro-title', macroDiv).after(Confluence.Templates.MacroBrowser.gadgetLink({ url: url }));
                }
            }

            // keyboard support
            macroDiv.on('keyup', function(event) {
                if (event.type === 'click' || event.key === 'Enter' || event.key === ' ') {
                    onclick(event, metadata);
                }
            });

            macroDiv[0].addEventListener('click', function(event) {
                // skip click events that are not triggered by the mouse or QUnit tests
                if (event.isTrusted && !event.screenY) {
                    return;
                }
                onclick(event, metadata);
            });

            return macroDiv;
        }

        /**
         *
         * @returns {{all: (*|jQuery|HTMLElement)}}
         */
        function getMacroSummaryHTMLElements(macros, onclick) {
            var cats = { all: makeCategoryList('all') };

            // Content on the right, setup macro list items
            for (var i = 0, ii = macros.length; i < ii; i++) {
                var macro = macros[i];
                if (macro.hidden) {
                    if (MacroBrowser.isHiddenMacroShown(macro)) {
                        macro.categories.push('hidden-macros');
                    } else {
                        continue;
                    }
                }
                // fill the ALL category
                cats.all.append(makeMacroSummary(macro, onclick).attr('id',
                    macro.id));

                // for each category, we create a different macroDiv.
                // be AWARE: we are duplicating macroDivs for the same macro in different categories!!
                // TODO: event delegation for onclick
                // TODO: unique macro summary DOM instances
                for (var j = 0, jj = macro.categories.length; j < jj; j++) {
                    var catKey = macro.categories[j];
                    cats[catKey] = cats[catKey] || makeCategoryList(catKey);
                    cats[catKey].append(makeMacroSummary(macro, onclick).attr('id', catKey + '-' + macro.id));
                }
            }
            return cats;
        }

        function sortMacros(macros, text) {
            return macros.sort(function(a, b) {
                var MAX_INDEX = 1000;
                var aTitle = $(a).find('h3').text().toLowerCase();
                var bTitle = $(b).find('h3').text().toLowerCase();
                // if search term is not found in the title, set index to MAX_INDEX so that it will be sorted to the end
                var aSearchTermIndex = aTitle.indexOf(text) === -1 ? MAX_INDEX : aTitle.indexOf(text);
                var bSearchTermIndex = bTitle.indexOf(text) === -1 ? MAX_INDEX : bTitle.indexOf(text);
                if (aSearchTermIndex < bSearchTermIndex) {
                    return -1;
                } if (aSearchTermIndex > bSearchTermIndex) {
                    return 1;
                } if (aTitle.length < bTitle.length) {
                    return -1;
                } if (aTitle.length > bTitle.length) {
                    return 1;
                }
                return 0;
            });
        }

        function createDialog(options) {
            var categorySummaryHTMLElements = getMacroSummaryHTMLElements(options.macros,
                options.onClickMacroSummary);

            var dialog = AJS.ConfluenceDialog({
                width: options.width || 865,
                height: options.height || 530,
                id: 'macro-browser-dialog',
                onSubmit: options.onSubmit,
                onCancel: options.onCancel
            });

            // Don't let the dialog catch keydown presses on AUI buttons and attempt something silly,
            // like submitting the dialog.
            dialog.popup.element.on('keydown', '.aui-button', function(e) {
                e.stopPropagation();
            });

            // Set an id on the "Select Macro Page"
            dialog.getPage(0).element.attr('id', 'select-macro-page');

            dialog.addHeader(options.title);

            // menu on the left, setup category panels
            dialog.addPanel(AJS.I18n.getText('macro.browser.category.all'), categorySummaryHTMLElements.all,
                'all', 'category-button-all').getPanel(0).setPadding(0);

            for (var i = 0, ii = options.categories.length; i < ii; i++) {
                var category = options.categories[i];
                dialog.addPanel(category.displayName,
                    categorySummaryHTMLElements[category.name] || makeCategoryList(category.name),
                    category.name, 'category-button-' + category.name)
                    .getPanel(i + 1).setPadding(0); // remove the default dialog padding
            }

            dialog.addCancel(AJS.I18n.getText('cancel.name'), options.onCancel);

            setHelpLink(dialog);

            // todo: make sure this is needed.
            // if it is, add proper comments an tests. Otherwise, delete it.
            dialog.addHelpText('<a href=\'{0}\' id=\'marketplace-link\'><span class=\'aui-icon aui-icon-small aui-iconfont-search\'></span>{1}</a>', [
                CONSTANTS.CONTEXT_PATH + '/plugins/servlet/upm/marketplace/featured?category=Macros&source=macro_browser',
                AJS.I18n.getText('macro.browser.marketplace.tip')
            ]);

            // prepare insert macro page
            var insertMacroBody = $(Confluence.Templates.MacroBrowser.insertMacroPanel());
            insertMacroBody.find('form').submit(function(e) {
                MacroBrowser.complete();
                e.stopPropagation();
                return false;
            });

            $('#macro-browser-preview-link', insertMacroBody).click(function(e) {
                var $link = $(this);
                if (!$link.prop('disabled')) {
                    MacroBrowser.previewMacro(dialog.activeMetadata);
                }
                e.stopPropagation();
                return false;
            });

            dialog.addPage()
                .addPanel('X', insertMacroBody, 'macro-input-panel')
                .addLink(AJS.I18n.getText('macro.browser.title'), function(dialog) {
                    dialog.prevPage();
                    dialog.searcher.focus();
                }, 'dialog-back-link')
                .addButton('insert.name', function() {
                    MacroBrowser.complete();
                }, 'ok aui-button aui-button-primary')
                .addCancel(AJS.I18n.getText('cancel.name'), function() {
                    return options.onCancel();
                })
                .getPanel(0)
                .setPadding(0); // remove the default dialog padding

            // Set an id on the "Insert/Update Macro Page"
            dialog.getPage(1).element.attr('id', 'macro-details-page');

            $('#macro-browser-dialog .dialog-button-panel .ok')
                .removeClass('button-panel-button')
                .before('<span id=\'save-warning-span\' class=\'hidden\'></span>');

            // todo: move search component to AJS.MacroBrowser.Search?? is it worthy?
            // add search box
            dialog.searcher = Confluence.DomFilterField({
                items: '#macro-browser-dialog .dialog-panel-body #category-all .macro-list-item',
                formId: 'macro-browser-search-form',
                inputId: 'macro-browser-search',
                searcher: function($items, text) {
                    var ids = null;
                    if (text != '') {
                        if (dialog.getCurrentPanel() != dialog.getPanel(0)) {
                            dialog.gotoPanel(0);
                        }
                        var filteredSummaries = MacroBrowser.searchSummaries(text);
                        ids = {};
                        $.each(filteredSummaries, function() {
                            ids[this.id] = this;
                        });
                        var filteredItems = $items.filter(function() {
                            return this.id in ids;
                        });
                        // Sort the filtered items by the position of the search term, and then by length of the title
                        // e.g. for the search term "page", "page tree" should come before "contents page"
                        // and "page" should come before "page tree" (because it's shorter)
                        var $SortedItems = sortMacros(filteredItems, text);
                        // Detach and re-append the items to maintain the sorted order
                        $SortedItems.detach().appendTo('#category-all');
                    }
                    $items.each(function() {
                        (!ids || this.id in ids) ? $(this).show() : $(this).hide();
                    });
                },
                submitCallback: function($visibleItems, searchText) {
                    if (searchText != '' && $visibleItems.length === 1) {
                        // Only one macro found with search - select it.
                        $visibleItems.click();
                    }
                }
            });

            dialog.page[0].header.append(dialog.searcher.form);
            dialog.page[0].ontabchange = function(newPanel, oldPanel) {
                if (newPanel != dialog.getPanel(0, 0)) {
                    // Moving away from the "All" macro panel; reset the search value if present
                    dialog.searcher.reset();
                }
            };

            // Create the missing meta-data page
            var missingMetadataBody = $(Confluence.Templates.MacroBrowser.missingUserMacroMetadataPanel({ showAdminMessage: AJS.Meta.getBoolean('is-admin') }));

            dialog.addPage()
                .addPanel(AJS.I18n.getText('macro.browser.missing.metadata.title'), missingMetadataBody, 'missing-metadata-panel')
                .addLink(AJS.I18n.getText('macro.browser.back.button'), function(dialog) {
                    dialog.gotoPage(0);
                    dialog.searcher.focus();
                }, 'dialog-back-link')
                .addCancel(AJS.I18n.getText('cancel.name'), function() {
                    // todo: does it need return value? can we just pass the reference to onCancel?
                    return options.onCancel();
                });

            dialog.gotoPage(2);
            dialog.addHeader(AJS.I18n.getText('macro.browser.missing.metadata.title'));

            dialog.gotoPanel(0, 0);

            dialog.ready = true; // todo: necessary?

            return dialog;
        }

        /**
         * Puts the focus on the first legitimate input field of the details form.
         */
        function focusOnMacroDetailsFirstInput() {
            var $firstInput = $('#macro-insert-container').find('.macro-input-fields :input:visible:not(button,.select2-offscreen)').first();
            if (!$firstInput.length) {
                AJS.log('No input to focus');
                return;
            }

            if ($firstInput.hasClass('select2-input')) {
                // Select2 focusing is tricky. Best to tell it to focus itself, but calling select2('focus') didn't work
                // during testing: the select2 state gets updated but the document.activeElement is never updated.
                var select2Hidden = $firstInput.closest('.select2-container').parent().find('.select2-offscreen');

                if (typeof select2Hidden.select2 !== 'function') {
                    AJS.log('select2Hidden is not a select2 hidden input, skipping focus attempt.');
                    return;
                }
                select2Hidden.select2('open');
                select2Hidden.select2('close'); // the close straight after the open seems to stop the dropdown from ever being visible. But yes, ugh.
            } else {
                $firstInput.focus();
            }

            if (!MacroBrowser.selectedMacroDefinition && $firstInput.val() !== '') {
                // prefilled data in new macro form - select first field for easy overwrite.
                $firstInput.select();
            }
        }

        /**
         * Adds a Help link to the right-hand side of the dialog header bar, for the current page.
         *
         * @param dialog the current MacroBrowser dialog
         * @param props an optional map to pass to the helpLink Soy render
         */
        function setHelpLink(dialog, props) {
            var linkHtml;
            if (!props) {
                // Default Macro Browser help link
                linkHtml = Confluence.Templates.MacroBrowser.helpLink();
            } else {
                linkHtml = Confluence.Templates.Dialog.helpLink(props);
            }

            var currentPage = dialog.getPage(dialog.curpage);

            // CONFDEV-12853: Add help link via prepend() instead of append() to prevent FF display issue
            currentPage.element.find('.dialog-title').prepend(linkHtml);
        }

        return {

            /**
             * Creates the macro browser dialog using AUI components
             * @param {Object} options.macroBrowser MacroBrowser instance
             * @returns {*}
             */

            createDialog: createDialog,

            /**
             * Show/hide macro browser spinner
             * @param show
             */

            showBrowserSpinner: showBrowserSpinner,

            /**
             * Show/hide macro browser spinner
             * @param show
             */

            showPreviewSpinner: showPreviewSpinner,

            /**
             * Update button text according to operation mode (save/insert)
             * @param mode - Defaults to "edit"
             */
            updateButtonText: updateButtonText,

            /**
             * Enable/Disable save/edit button
             * @param {Boolean} enable
             */
            enableSaveButton: enableSaveButton,

            /**
             * Focus on the first input of the details form
             */

            focusOnMacroDetailsFirstInput: focusOnMacroDetailsFirstInput,

            /**
             * Set the help link at the top-right corner of the dialog to have the provided properties.
             */
            setHelpLink: setHelpLink,

            /**
             * Sort the macro list based on the search term
             */
            sortMacros: sortMacros
        };
    };
});

require('confluence/module-exporter').safeRequire('confluence-macro-browser/macro-browser-UI');
