/**
 * @module confluence-macro-browser/macro-browser-editor
 */
define('confluence-macro-browser/macro-browser-editor', [
    'ajs',
    'jquery',
    'confluence/legacy',
    'confluence-macro-browser/macro-browser-cql',
    'confluence/macro-js-overrides',
    'underscore'
], function(
    AJS,
    $,
    Confluence,
    MacroBrowserCQL,
    MacroJsOverrides,
    _
) {
    'use strict';

    return function(MacroBrowser) {
        /**
         * Namespace to organize common methods related to the macro editor
         */
        function loadMacroInBrowser(metadata, mode) {
            if (!metadata || !metadata.formDetails) {
                AJS.trigger('analytics', { name: 'macro-browser.unknown-macro-error' });
                alert(AJS.I18n.getText('macro.browser.unknown.macro.message'));
                return;
            }

            var tinymce = require('tinymce');
            var t = MacroBrowser;
            var detail = metadata.formDetails;
            var macroName = detail.macroName;
            var jsOverride = MacroJsOverrides.get(macroName);
            var selectedMacro = t.settings.selectedMacro;
            var selectedMacroElement = tinymce.confluence.macrobrowser.editedMacroDiv;

            if (jsOverride && typeof jsOverride.opener === 'function') {
                t.close();
                jsOverride.opener({ name: macroName });
                return;
            }

            var params = detail.parameters;
            var cqlParam = _.find(params, function(param) {
                return param.type.name === 'cql';
            });

            // Macro description and documentation link
            $('#macro-insert-container .macro-name').val(macroName);

            var macroParametersForm = $('#macro-insert-container .macro-input-fields form');
            macroParametersForm.empty(); // the one form is reused, so we need to flush it for each macro.

            // Description, including optional documentation link.
            var macroDescription = metadata.extendedDescription || metadata.description;
            if (cqlParam) {
                if (detail.documentationLink) {
                    // CQL doc links go in the title bar of the dialog
                    t.UI.setHelpLink(t.dialog, {
                        href: detail.documentationLink
                    });
                }
            } else if (macroDescription || detail.documentationLink) {
                var macroDesc = $(Confluence.Templates.MacroBrowser.macroDescription({
                    descriptionHtml: macroDescription,
                    documentationLink: detail.documentationLink
                }));
                macroParametersForm.append(macroDesc);
            }

            // Macros with bodies get a body form field before other macro parameters.
            if (detail.body && detail.body.bodyType != 'NONE' && (selectedMacroElement && $(selectedMacroElement).hasClass('editor-inline-macro'))) { // only enable editing of the body in the macro browser for inline body macros
                var pluginKey = metadata.pluginKey;
                if (detail.body.label == MacroBrowser.Utils.makeDefaultKey(pluginKey, macroName, 'body', 'label')) {
                    detail.body.label = '';
                }
                if (detail.body.description == MacroBrowser.Utils.makeDefaultKey(pluginKey, macroName, 'body', 'desc')) {
                    detail.body.description = '';
                }
                var body = makeBodyDiv(detail.body, t.selectedMacroDefinition);
                if (body) {
                    macroParametersForm.append(body);
                }
            }

            /**
             * Add the macro parameter fields to the form. There are 2 main types of params:
             *
             * 1. Simple params
             * 2. CQL params
             *
             * CQL params can represent a number of components in the UI that aggregate to represent a CQL expression.
             * CQL components handle their own rendering and value initialisation.
             */

            var selectedParams = selectedMacro ? $.extend({}, selectedMacro.params) : {}; // make a copy

            // The div that parameter fields will be appended to.
            var parameterDivContainer = macroParametersForm;

            if (cqlParam) {
                // Pick up custom properties, if any, for the cql label picker select2 component
                // eslint-disable-next-line vars-on-top
                var cqlSelect2CustomOptions;
                if (jsOverride && jsOverride.fields) {
                    cqlSelect2CustomOptions = $.extend({}, jsOverride.fields.cqlSelect2CustomOptions);
                }

                // The CQL macro param has special needs for rendering with or without data. Drop it from the default behaviours.
                params = _.without(params, cqlParam);

                var macroBrowserCql = MacroBrowserCQL.build({
                    cqlParam: cqlParam,
                    renderParams: params,
                    cqlValue: selectedParams.cql,
                    container: macroParametersForm,
                    flagRequiredParam: flagRequiredParam,
                    paramChanged: MacroBrowser.paramChanged,
                    cqlSelect2CustomOptions: cqlSelect2CustomOptions
                });

                delete selectedParams.cql; // this avoids cql being treated as an unknown parameter further down

                parameterDivContainer = macroBrowserCql.elements.filter('.cql-render-fields');

                macroBrowserCql.loading.done(function() {
                    MacroBrowser.paramChanged();
                    MacroBrowser.Preview.previewMacro(metadata, selectedMacro);
                    MacroBrowser.UI.focusOnMacroDetailsFirstInput();
                });
            }

            // Parameters may have dependencies so all fields need to be created before values are set.
            $(params).each(function() {
                parameterDivContainer.append(makeParameterDiv(metadata, this, jsOverride));
            });

            // Fully-implemented macros may have JS overrides defined in a Macro object.
            if (jsOverride && typeof jsOverride.beforeParamsSet === 'function') {
                selectedParams = jsOverride.beforeParamsSet(selectedParams, !selectedMacro);
            }

            $(params).each(function() {
                var param = this;
                var value;

                if (param.name == '') {
                    value = t.selectedMacroDefinition ? t.selectedMacroDefinition.defaultParameterValue : param.defaultValue;
                } else {
                    value = selectedParams[param.name];
                }

                if (value != null) {
                    delete selectedParams[param.name];
                } else {
                    // try looking for aliased parameters
                    $(param.aliases).each(function() {
                        if (selectedParams[this]) {
                            value = selectedParams[this];
                            delete selectedParams[this];
                        }
                    });
                }
                if (value == null) {
                    value = param.defaultValue;
                }
                if (value != null) {
                    t.fields[param.name].setValue(value);
                }
            });

            // Any remaining "selectedParameters" are unknown for the current Macro details.
            t.unknownParams = selectedParams;

            // open all links in a new window
            $('a', macroParametersForm).click(function() {
                var windowOpened = window.open(this.href, '_blank');
                windowOpened.focus();
                windowOpened.opener = null;
                return false;
            });

            if (!$('#macro-browser-dialog:visible').length) {
                t.showBrowserDialog();
            }

            var macroData = {};
            $.extend(true, macroData, metadata); // clone the metadata to a new object

            // for safety ensure that we have a 'formDetails', containing a body.
            if (!macroData.formDetails) { macroData.formDetails = {}; }
            if (!macroData.formDetails.body) { macroData.formDetails.body = {}; }

            t.dialog.activeMetadata = macroData; // This is important to getMacroBody.

            // do we have have selected text for the macro body or are we editing an existing macro (in which case any selected text is ignored)
            if (t.settings.selectedMacro) {
                macroData.formDetails.body.content = t.settings.selectedMacro.body;
            } else if (macroData.formDetails.body.bodyType && macroData.formDetails.body.bodyType.toLowerCase() === 'plain_text') {
                macroData.formDetails.body.content = t.settings.selectedText;
            } else {
                macroData.formDetails.body.content = t.settings.selectedHtml;
            }
        }

        function getMacroParametersFromForm(metadata) {
            return getMacroDefinitionFromForm(metadata).params;
        }

        function getMacroDefinitionFromForm(metadata) {
            var macroName = $('#macro-insert-container').find('.macro-name').val(); // todo: why don't we get this from the metadata?
            var defaultParameterValue;
            var t = MacroBrowser;
            var sharedParameters = {};

            var parameters = getMacroParametersFromFormFields(metadata, sharedParameters);

            if (parameters['']) {
                defaultParameterValue = parameters[''];
            }

            $.extend(parameters, t.unknownParams); // apply unknown parameters

            parameters = applyjsOverrides(macroName, parameters, metadata, sharedParameters);

            return {
                name: macroName,
                bodyHtml: t.getMacroBody(),
                params: parameters,
                defaultParameterValue: defaultParameterValue
            };
        }

        function processRequiredParameters() {
            if ($('.macro-input-fields form').is('.loading')) {
                AJS.log('Form still loading - cannot processRequiredParameters');
                return false;
            }

            var blankRequiredInputs = $('#macro-insert-container .macro-param-div.required .macro-param-input')
                .not('.select2-container')// FIXME ugh. select2 label control getting doubled up, so blankRequiredInputs.length always > 0
                .filter(function() {
                    var val = $(this).val();
                    return (val === null || val === '');
                });
            // TODO - picking up required CQL?
            var hasAllRequiredData = (blankRequiredInputs.length === 0);
            var disabled = hasAllRequiredData ? '' : 'disabled';
            var classFn = disabled ? 'addClass' : 'removeClass';

            $('#macro-browser-dialog button.ok').prop('disabled', disabled);
            $('#macro-browser-dialog .macro-preview-header .refresh-link').prop('disabled', disabled)[classFn]('disabled');

            return hasAllRequiredData;
        }

        // private functions

        /**
         * Apply jsOverrides for a macroName
         *
         * @param macroName
         * @param parameters
         * @param macro
         * @param sharedParameters
         * @returns {*} parameters
         */
        function applyjsOverrides(macroName, parameters, macro, sharedParameters) {
            var beforeParamsRetrieved = MacroJsOverrides.getFunction(macroName, 'beforeParamsRetrieved');

            if (beforeParamsRetrieved) {
                parameters = beforeParamsRetrieved(parameters, macro, sharedParameters);
            }
            return parameters;
        }

        /**
         * Get value from the input element according to input type (input text, checkbox, etc)
         * @param $element jQuery input element
         * @returns {*}
         */
        function extractValueFromInputElement($element) {
            var paramVal = $element.val();

            if ($element.attr('type') == 'checkbox') {
                paramVal = '' + $element.prop('checked'); // will return 'true' if checked property is not empty
            }
            return paramVal;
        }

        /**
         * Fill parameters and shared parameters according to form values
         * @param metadata
         * @param sharedParameters
         * @returns {{}}
         */
        function getMacroParametersFromFormFields(metadata, sharedParameters) {
            var parameters = {};
            for (var i = 0, l = metadata.formDetails.parameters.length; i < l; i++) {
                var parameter = metadata.formDetails.parameters[i];
                var paramVal;
                if (parameter.type && parameter.type.name === 'cql') {
                    var cqlComponent = $('.macro-input-fields form').data('cqlComponent');
                    paramVal = cqlComponent.getValue();
                } else {
                    paramVal = extractValueFromInputElement($('#macro-param-' + parameter.name));
                }
                if (paramVal) {
                    // the shared property is used by gadgets in jsOverrides
                    if (parameter.shared) {
                        sharedParameters[parameter.name] = paramVal;
                    }

                    if (parameter.hidden
                        || (!parameter.defaultValue || parameter.defaultValue != paramVal)) {
                        // only add the parameter value if its not the default
                        parameters[parameter.name] = paramVal;
                    }
                }
            }
            return parameters;
        }

        /**
         * Creates a div for a macro body.
         *
         * @param macroBodyFormComponent
         * @param macroDefinition
         * @returns {*|jQuery|HTMLElement}
         */
        function makeBodyDiv(macroBodyFormComponent, macroDefinition) {
            var t = MacroBrowser;
            var bodyDiv = $(Confluence.Templates.MacroBrowser.macroBody());
            var bodyTextareaValue;

            if (macroDefinition) { // we are editing an existing macro
                bodyTextareaValue = macroDefinition.body;
            } else { // check if there is selected text in the editor
                bodyTextareaValue = t.settings.selectedText;
            }

            $('textarea', bodyDiv).val(bodyTextareaValue || '');

            if (macroBodyFormComponent.label) {
                $('label', bodyDiv).text(macroBodyFormComponent.label);
            }
            if (macroBodyFormComponent.description) {
                bodyDiv.append(Confluence.Templates.MacroBrowser.macroParameterDesc({ description: macroBodyFormComponent.description }));
            }
            if (macroBodyFormComponent.hidden) {
                bodyDiv.hide();
            }
            return bodyDiv;
        }

        /**
         * Creates a div for a single macro parameter.
         *
         * @param metadata
         * @param param
         * @param jsOverride
         * @returns {*}
         */
        function makeParameterDiv(metadata, param, jsOverride) {
            param.pluginKey = metadata.pluginKey;
            param.macroName = metadata.macroName;
            var t = MacroBrowser;
            var field;
            var type = param.type.name;
            var paramId = 'macro-param-' + param.name;

            // Plugin point - other JS files can define more specific field-builders based on macro name, param name and
            // type.
            if (jsOverride) {
                var builder = jsOverride.fields && jsOverride.fields[type];
                if (builder && typeof builder !== 'function') {
                    // Types can be overridden for specific parameters - so the "type" object contains a "name" function.
                    builder = builder[param.name];
                }
                if (typeof builder === 'function') {
                    field = builder.call(jsOverride, param);
                }
            }

            // ID for aria-describedby when description exists
            if (param.description !== MacroBrowser.Utils.makeDefaultKey(metadata.pluginKey, metadata.macroName, 'param', param.name, 'desc')) {
                param.descriptionId = paramId + '-desc';
            }

            // If no override specific to the macro, look for general overrides specific to the parameter type.
            if (!field) {
                if (!(type in t.ParameterFields && typeof t.ParameterFields[type] === 'function')) {
                    type = 'string';
                }
                // In some cases id should be added on input creation
                param.selectInputId = paramId;
                field = t.ParameterFields[type](param);
            }

            t.fields[param.name] = field;
            var paramDiv = field.paramDiv;
            var input = field.input;

            paramDiv.attr('id', 'macro-param-div-' + param.name);
            input.addClass('macro-param-input').attr('id', paramId);
            if (param.hidden) {
                paramDiv.hide();
            }

            // Use param label and desc or correct fallback.
            var pluginKey = metadata.pluginKey;
            if (param.displayName == MacroBrowser.Utils.makeDefaultKey(pluginKey, metadata.macroName, 'param', param.name, 'label')) {
                param.displayName = param.name;
            }
            if (param.description == MacroBrowser.Utils.makeDefaultKey(pluginKey, metadata.macroName, 'param', param.name, 'desc')) {
                param.description = '';
            }

            $('label', paramDiv).attr('for', paramId).text(param.displayName);

            if (param.required) {
                flagRequiredParam(paramDiv);
            }

            if (param.description && param.descriptionId) {
                paramDiv.append(Confluence.Templates.MacroBrowser.macroParameterDesc({ id: param.descriptionId, description: param.description }));
            }
            return paramDiv;
        }

        function flagRequiredParam(paramDiv) {
            var label = paramDiv.find('label');
            var labelText = label.text() + ' *';
            var input = paramDiv.find('input');
            label.text(labelText);
            input.attr('aria-required', 'true');
            paramDiv.addClass('required'); // set class against div, not input, to allow styling of label if nec
        }

        return {
            /**
             * Loads the given macro json in the browser's insert macro page.
             *
             * @param metadata
             * @param mode "insert" or "edit"
             */

            loadMacroInBrowser: loadMacroInBrowser,

            /**
             * Returns a Map of all parameter values from the form, including the default parameter value which has a zero
             * length string as a key.
             * @param metadata meta data about each parameter in the macro
             */

            getMacroParametersFromForm: getMacroParametersFromForm,

            /**
             * Constructs the macro markup from the insert macro page
             *
             * @param {Object} metadata detailed metadata about the macro.  Includes form details etc.
             * @returns {{name: (*|jQuery), bodyHtml: *, params: {}, defaultParameterValue: *}}
             */

            getMacroDefinitionFromForm: getMacroDefinitionFromForm,

            /**
             * Checks and returns true if all the required macro parameters have values.
             * It disables the insert/preview buttons if false.
             *
             * @returns {boolean}
             */

            processRequiredParameters: processRequiredParameters
        };
    };
});

require('confluence/module-exporter').safeRequire('confluence-macro-browser/macro-browser-editor');
