/**
 * @module confluence-macro-browser/macro-browser-fields
 */
define('confluence-macro-browser/macro-browser-fields', [
    'ajs',
    'jquery',
    'underscore',
    'confluence/legacy',
    'confluence/meta',
], function(
    AJS,
    $,
    _,
    Confluence,
    Meta
) {
    'use strict';

    function getAttachmentHandler(MacroBrowserParameterFields, MacroBrowserField, updateDependencies) {
        return function attachmentHandler(param, options) {
            if (param.multiple) {
                return MacroBrowserParameterFields.string(param,
                    options);
            }

            var paramDiv = $(Confluence.Templates.MacroBrowser.macroParameterSelect());
            var input = $('select', paramDiv);

            options = options || {};
            options.setValue = options.setValue || function(value) {
                // check if the value being set is in the list of options
                // if not then add it as a new option - with an indication that
                // it is not a valid choice for this select box
                var foundOption = false;
                // Don't use a JQuery filter of "[value=" +value+"]" since value is un-escaped user-data
                input.find('option').each(function() {
                    if (this.value == value) {
                        foundOption = true;
                        return false;
                    }
                });

                if (!foundOption) {
                    input.append($('<option></option>').val(value).text(value + ' (' + AJS.I18n.getText('macro.browser.smartfields.not.found') + ')'));
                    input.tempValue = value;
                } else {
                    delete input.tempValue;
                }

                // CONF-15415 : Spurious error thrown in IE6
                try {
                    input.val(value);
                } catch (err) {
                    AJS.logError(err);
                }

                input.change();
            };

            var field = MacroBrowserField(paramDiv, input, options);
            field.updateDependencies = updateDependencies;
            field.getData = function(req) {
                if (!((req.title && req.spaceKey) || req.pageId || req.draftId)) {
                    AJS.log('Not enough parameters to send attachmentsearch request');
                    return;	// not enough content info to get attachments
                }

                var currentValue = input.tempValue || input.val();

                if (options.fileTypes) {
                    req.fileTypes = options.fileTypes;
                }

                var url = AJS.contextPath() + (req.draftId && !Meta.get('shared-drafts') ? '/json/draftattachmentsearch.action' : '/json/attachmentsearch.action');
                $.getJSON(url, req, function(data) {
                    if (data.error) {
                        return;
                    }

                    $('option', input).remove();
                    var attachments = data.attachments;

                    // if there are no attachments on the page then populate the options with
                    // a message stating this
                    if (!attachments || !attachments.length) {
                        // AJS.log("attachment - No attachments so creating message. tempValue = " + input.tempValue);
                        input.append($('<option></option>').val('').html(AJS.I18n.getText('macro.browser.smartfields.no.attachments')));

                        if (input.tempValue) {
                            options.setValue(input.tempValue);
                        }
                    } else {
                        for (var i = 0, length = attachments.length; i < length; i++) {
                            input.append($('<option></option>').val(attachments[i].name).text(attachments[i].name));
                        }

                        currentValue = currentValue || input.tempValue;
                        options.setValue(currentValue || attachments[0].name);
                    }
                    // in IE9 only the first character of the selected option is shown, force a redraw
                    input.attr('style', '');
                });
            };

            return field;
        };
    }

    return function(MacroBrowser) {
        /**
         * Returns an object wrapper for a parameter-div jQuery object and the input in
         * that div that stores the internal parameter value (as opposed to the display
         * field, although they may be the same).
         */
        var MacroBrowserField = function(paramDiv, input, options) {
            options = options || {};

            var setValue = options.setValue || function(value) {
                input.val(value);
            };

            var getValue = options.getValue || function() {
                return input.val();
            };

            input.change(options.onchange || MacroBrowser.paramChanged);

            return {
                paramDiv: paramDiv,
                input: input,
                setValue: setValue,
                getValue: getValue
            };
        };

        /**
         * Update the dependencies of the identified parameter with the supplied value.
         */
        var updateDependencies = function(paramName, dependencies, value) {
            if (dependencies && dependencies.length) {
                for (var i = 0, length = dependencies.length; i < length; i++) {
                    MacroBrowser.fields[dependencies[i]] && MacroBrowser.fields[dependencies[i]].dependencyUpdated(paramName, value);
                }
            }
        };

        /**
         * ParameterFields defines default "type" logic for fields in the Macro
         * Browser's Insert/Edit Macro form.
         *
         * Each method in this object corresponds to a parameter type as defined in the
         * MacroParameterType enum.
         */
        var MacroBrowserParameterFields = {
            updateDependencies: updateDependencies,

            username: function(param, options) {
                options = options || {};

                var className = param.multiple ? 'autocomplete-multiuser' : 'autocomplete-user';
                var paramDiv = $(Confluence.Templates.MacroBrowser.macroParameter());
                var input = $('input[type=\'text\']', paramDiv);
                if (param.selectInputId && typeof param.selectInputId === 'string') {
                    input.attr('id', param.selectInputId);
                }
                input.addClass(className).attr('data-none-message', AJS.I18n.getText('macro.browser.smartfields.not.found'));

                // CONF-16859 - check if mandatory params are now filled
                if (param.required) {
                    input.keyup(MacroBrowser.processRequiredParameters);
                }

                if (param.multiple) {
                    Confluence.Binder.autocompleteMultiUser(paramDiv);
                    options.setValue = function(value) {
                        input.val(value);

                        var values = value.split(',');
                        var requests = [];
                        var users = [];

                        // unfortunately there is no rest api to get me a list of users, however since its not a common case
                        // to edit macros with multiple users, going to call the existing api for each user
                        for (var i = 0, l = values.length; i < l; i++) {
                            var request = function(username) {
                                return $.getJSON(AJS.contextPath() + '/rest/prototype/1/user/non-system/' + username)
                                    .done(function(user) {
                                        if (user && user.name) {
                                            users.push({ id: user.name, text: user.displayName, imgSrc: user.avatarUrl });
                                        }
                                    }).fail(function(response) {
                                        AJS.logError('Cannot parse user data for macro browser field with: ' + response);
                                        users.push({ id: username, text: AJS.I18n.getText('unknown.name', username), imgSrc: AJS.contextPath() + '/images/icons/profilepics/default.png' });
                                    });
                            };

                            requests.push(request(values[i]));
                        }

                        // wait for all requests to finish, whether they passed/failed and then populate the field value
                        $.when.apply($, _.map(requests, function(aRequest) {
                            var monitor = $.Deferred();
                            aRequest.always(function() {
                                monitor.resolve();
                            });
                            return monitor.promise();
                        })).done(function() {
                            if (users.length) {
                                input.auiSelect2('data', users);
                            }
                        });
                    };
                } else {
                    input.bind('selected.autocomplete-content', function(e, data) {
                        if (options.onselect) {
                            options.onselect(data.selection);
                        } else if (options.setValue) {
                            options.setValue(data.content.username);
                        } else {
                            updateDependencies(param.name, options.dependencies, input.val());
                            (typeof options.onchange === 'function') && options.onchange.apply(input);
                        }
                    });
                    AJS.Confluence.Binder.autocompleteUserOrGroup(paramDiv);
                }

                if (param.descriptionId) {
                    // Use select input as the other input field isn't visible
                    var select = paramDiv.find('.select2-container input[type="text"]');
                    if (select.length) select.attr('aria-describedby', param.descriptionId);
                }
                return MacroBrowserField(paramDiv, input, options);
            },

            spacekey: function(param, options) {
                // for multple space keys just use a String field at the moment
                if (param.multiple) {
                    return MacroBrowserParameterFields.string(param, options);
                }

                options = options || {};

                var paramDiv = $(Confluence.Templates.MacroBrowser.macroParameter());
                var input = $('input[type=\'text\']', paramDiv);
                input.addClass('autocomplete-space')
                    .attr('data-value-key', 'key')
                    .attr('data-none-message', AJS.I18n.getText('macro.browser.smartfields.not.found'));

                // CONF-16859 - check if mandatory params are now filled
                if (param.required) {
                    input.keyup(MacroBrowser.processRequiredParameters);
                }

                if (param.descriptionId) {
                    input.attr('aria-describedby', param.descriptionId);
                }

                input.bind('selected.autocomplete-content', function(e, data) {
                    if (options.onselect) {
                        options.onselect(data.selection);
                    } else if (options.setValue) {
                        options.setValue(data.content.key);
                    } else {
                        updateDependencies(param.name, options.dependencies, input.val());
                        (typeof options.onchange === 'function') && options.onchange.apply(input);
                    }
                });
                AJS.Confluence.Binder.autocompleteSpace(paramDiv);

                return MacroBrowserField(paramDiv, input);
            },

            attachment: getAttachmentHandler(MacroBrowserParameterFields, MacroBrowserField, updateDependencies),

            full_attachment: getAttachmentHandler(MacroBrowserParameterFields, MacroBrowserField, updateDependencies),

            'confluence-content': function(param, options) {
                // If multiple confluence-content field then only return a String at the moment
                if (param.multiple) {
                    return MacroBrowserParameterFields.string(param, options);
                }

                options = options || {};
                param.options = param.options || {};

                var paramDiv = $(Confluence.Templates.MacroBrowser.macroParameter());
                var input = $('input[type=\'text\']', paramDiv)
                    .attr('data-none-message', AJS.I18n.getText('macro.browser.smartfields.not.found'))
                    .attr('data-value-key', ''); // no template as some logic is required to build the value

                // CONF-16859 - check if mandatory params are filled on keypresses in this field.
                if (param.required) {
                    input.keyup(MacroBrowser.processRequiredParameters);
                }

                // CONF-15438 - update any dependencies of the field when it is changed
                options.onchange = options.onchange || function(e) {
                    var val = input.val();
                    updateDependencies(param.name, options.dependencies, val);
                };
                options.setValue = options.setValue || function(value) {
                    input.val(value);
                    (typeof options.onchange === 'function') && options.onchange.apply(input);
                };

                input.bind('selected.autocomplete-content', function(e, data) {
                    var datePathPrefix = '';
                    if (param.options.includeDatePath == 'true' && data.content.type === 'blogpost') {
                        var splitDate = data.content.createdDate.date.split('-');
                        datePathPrefix = '/' + splitDate[0] + '/' + splitDate[1] + '/' + splitDate[2].substring(0, 2) + '/';
                    }
                    var spaceKey = data.content.space && data.content.space.key;
                    var isSpaceKeyNeeded = (spaceKey && spaceKey != AJS.Meta.get('space-key'));
                    if (data.content.title.indexOf(':') !== -1) {
                        isSpaceKeyNeeded = true;
                    }
                    var markup = (isSpaceKeyNeeded ? (spaceKey + ':') : '') + datePathPrefix + data.content.title;

                    input.val(markup);

                    if (options.onselect) {
                        options.onselect(data.selection);
                    } else {
                        options.setValue(markup, input);
                    }
                });

                if (param.options.spaceKey) {
                    if (param.options.spaceKey.toLowerCase() === '@self') {
                        param.options.spaceKey = AJS.Meta.get('space-key');
                    }
                    input.attr('data-spacekey', param.options.spaceKey);
                }

                var type = param.options.type;
                if (typeof type === 'string') {
                    if (type == 'page') {
                        input.addClass('autocomplete-page');
                        AJS.Confluence.Binder.autocompletePage(paramDiv);
                    } else if (type == 'blogpost') {
                        input.addClass('autocomplete-blogpost');
                        AJS.Confluence.Binder.autocompleteBlogpost(paramDiv);
                    } else if (type === 'attachment' || type === 'full_attachment') {
                        input.addClass('autocomplete-attachment');
                        AJS.Confluence.Binder.autocompleteAttachment(paramDiv);
                    }
                } else { // default to pages and blogs
                    input.addClass('autocomplete-confluence-content');
                    AJS.Confluence.Binder.autocompleteConfluenceContent(paramDiv);
                }

                return MacroBrowserField(paramDiv, input, options);
            },

            /**
             * Default field for all unknown types.
             */
            string: function(param, options) {
                var paramDiv = $(Confluence.Templates.MacroBrowser.macroParameter());
                var input = $('input', paramDiv);

                if (param.required) {
                    input.keyup(MacroBrowser.processRequiredParameters);
                }

                if (param.descriptionId) {
                    input.attr('aria-describedby', param.descriptionId);
                }

                return MacroBrowserField(paramDiv, input, options);
            },

            /**
             * A checkbox - assumes not true means false, not null.
             */
            boolean: function(param, options) {
                var paramDiv = $(Confluence.Templates.MacroBrowser.macroParameterBoolean());
                var input = $('input', paramDiv);

                options = options || {};
                options.setValue = options.setValue || function(value) {
                    if (/true/i.test(value)
                            || (/true/i.test(param.defaultValue) && !(/false/i).test(value))) {
                        input.prop('checked', true);
                    }
                };
                if (param.descriptionId) {
                    input.attr('aria-describedby', param.descriptionId);
                }

                return MacroBrowserField(paramDiv, input, options);
            },

            enum: function(param, options) {
                if (param.multiple) {
                    return MacroBrowserParameterFields.string(param, options);
                }

                var paramDiv = $(Confluence.Templates.MacroBrowser.macroParameterSelect());
                var input = $('select', paramDiv);
                if (!(param.required || param.defaultValue)) {
                    input.append($('<option></option>').val(''));
                }
                if (param.descriptionId) {
                    input.attr('aria-describedby', param.descriptionId);
                }
                $(param.enumValues).each(function(index, enumValue) {
                    var valueDescriptionValue = enumValue;
                    // No i18n (i.e. enumToI18nKeyMapping) available for custom macros, like user macros.
                    // enumToI18nKeyMapping defined in com.atlassian.confluence.macro.browser.beans.MacroParameter
                    if (param.enumToI18nKeyMapping) {
                        var i18nKey = [param.pluginKey, param.macroName, 'param', param.name, this, 'desc'].join('.');
                        if (param.enumToI18nKeyMapping[enumValue] !== i18nKey) {
                            valueDescriptionValue = param.enumToI18nKeyMapping[enumValue];
                        }
                    }
                    input.append($('<option></option>').val(this).html('' + valueDescriptionValue));
                });

                return MacroBrowserField(paramDiv, input, options);
            },

            /**
             * Like a "string" field but hidden.
             */
            _hidden: function(param, options) {
                var paramDiv = $(Confluence.Templates.MacroBrowser.macroParameterHidden()).hide();
                var input = $('input', paramDiv);

                return MacroBrowserField(paramDiv, input, options);
            }

        };

        return {
            Field: MacroBrowserField,
            ParameterFields: MacroBrowserParameterFields
        };
    };
});

require('confluence/module-exporter').safeRequire('confluence-macro-browser/macro-browser-fields');
