/**
 * @module confluence-macro-browser/macro-browser-model
 */
define('confluence-macro-browser/macro-browser-model', [
    'ajs'
], function(
    AJS
) {
    'use strict';

    return function(MacroBrowser) {
        /**
         * Metadata model operations
         *
         */
        function transformMetaDataDefault(metadata) {
            // if we get an internationalization key back for the title, then pull the macro name out.
            if (metadata.title == MacroBrowser.Utils.makeDefaultKey(metadata.pluginKey, metadata.macroName, 'label')) {
                metadata.title = metadata.macroName.charAt(0).toUpperCase() + metadata.macroName.substring(1).replace(/-/g, ' ');
            }

            if (metadata.description == MacroBrowser.Utils.makeDefaultKey(metadata.pluginKey, metadata.macroName, 'desc')) {
                metadata.description = '';
            }
            metadata.id = 'macro-' + (metadata.alternateId || metadata.macroName);

            // Store two copies of keywords - for Macro Browser and Macro Autocomplete searching
            var keywordsArr = [metadata.macroName, metadata.title].concat(metadata.aliases || []);
            metadata.keywordsNoDesc = keywordsArr.join(',');

            var desc = (metadata.description && metadata.description.replace(/,/g, ' ')) || '';
            keywordsArr.push(desc);
            metadata.keywords = keywordsArr.join(',');
            return metadata;
        }

        function loadMacros(macros) {
            if (!macros) {
                AJS.log('AJS.MacroBrowser.loadModel - no macro data, aborting');
                return;
            }

            var t = MacroBrowser;
            t.metadataList = [];
            t.aliasMap = {};

            // Clean up unset titles and descriptions before sorting
            for (var i = 0, ii = macros.length; i < ii; i++) {
                var metadata = macros[i];

                metadata = transformMetaDataDefault(metadata);

                if (metadata.aliases) {
                    for (var j = 0, jl = metadata.aliases.length; j < jl; j++) {
                        metadata.aliases[j] = metadata.aliases[j].toLowerCase();
                        t.aliasMap[metadata.aliases[j]] = metadata.macroName.toLowerCase();
                    }
                }

                t.metadataList.push(metadata);
            }
            t.metadataList.sort(function(one, two) { return (one.title.toLowerCase() > two.title.toLowerCase() ? 1 : -1); });
        }

        return {
            /**
             * Called when the macro data is returned from the back-end, loads the macro metadata into a JS model for the
             * browser (and autocomplete) to use.
             *
             * @param macros the macros array from the browse-macros.action JSON response
             */
            loadMacros: loadMacros,

            /**
             * Applies the necessary transformations when macro metadata is retrieved form the server
             * @param metadata
             * @returns transformed metadata
             */
            transformMetaDataDefault: transformMetaDataDefault
        };
    };
});

require('confluence/module-exporter').safeRequire('confluence-macro-browser/macro-browser-model');
