/**
 * @module confluence-macro-browser/macro-browser-preview
 */
define('confluence-macro-browser/macro-browser-preview', [
    'jquery',
    'ajs',
    'confluence/legacy',
    'confluence/macro-js-overrides',
], function(
    $,
    AJS,
    Confluence,
    MacroJsOverrides
) {
    'use strict';

    return function(MacroBrowser) {
        /**
         * Macro preview rendering
         * @type {Function}
         */
        var refreshButton = {
            disable: function() {
                $('#macro-browser-preview-link').prop('disabled', true).addClass('disabled');
            },
            enable: function() {
                $('#macro-browser-preview-link').prop('disabled', false).removeClass('disabled');
            }
        };

        function getPreviewContainerDiv() {
            return $('#macro-insert-container .macro-preview');
        }

        function removePreviewContainer() {
            // removes the iFrame.
            getPreviewContainerDiv().html('');
        }

        function renderPreview(html) {
            // create the iframe and the insert everything into it before inserting into the DOM
            var previewIframe = document.createElement('iframe');
            var doc;
            var errorSpan;

            previewIframe.setAttribute('name', 'macro-browser-preview-frame');
            previewIframe.setAttribute('id', 'macro-preview-iframe');
            previewIframe.srcdoc = html;
            getPreviewContainerDiv().append(previewIframe);

            doc = previewIframe.contentDocument;
            $(doc).ready(function() {
                // This event is used with AJS.trace to allow WD tests to wait until the preview is fully loaded.
                AJS.trigger('macro-browser.preview-ready');
            });

            errorSpan = $('div.error span.error', doc);
            if (errorSpan.length) {
                AJS.logError('Error rendering macro definition : ');
            }
        }

        function loadingInProgressUI(loading) {
            if (loading) {
                MacroBrowser.UI.showPreviewSpinner(true);
                refreshButton.disable();
            } else {
                MacroBrowser.UI.showPreviewSpinner(false);
                refreshButton.enable();
            }
        }

        function previewMacro(macro, macroDefinition) {
            var t = MacroBrowser;
            var params;
            var updateMacroParametersForPreview;
            var options;

            removePreviewContainer();

            if (!MacroBrowser.Editor.processRequiredParameters()) {
                AJS.trigger('analytics', { name: 'macro-browser.missing-params-error' });
                AJS.log('previewMacro: missing required params');
                return false;
            }

            // A macroDefinition may already exist if we are previewing an existing macro that has been reloaded in
            // the macro browser.
            if (typeof macroDefinition !== 'object') {
                macroDefinition = MacroBrowser.Editor.getMacroDefinitionFromForm(macro);
            }

            params = macroDefinition.params;

            updateMacroParametersForPreview = MacroJsOverrides.getFunction(macro.macroName, 'updateMacroParametersForPreview');

            if (updateMacroParametersForPreview) {
                params = updateMacroParametersForPreview(params);
            }

            if (params['']) {
                delete params['']; // this is sent separately as the defaultParameterValue
            }

            options = {
                contentId: Confluence.getContentId(),
                macroName: macro.macroName,
                params: params,
                body: t.getMacroBody(),
                defaultParameterValue: macroDefinition.defaultParameterValue,

                successCallback: function(html) {
                    renderPreview(html);
                    // todo: the spinner is being hidden when the iframe gets loaded
                    // so this timeout won't affect it anyway. It will disable the preview button though
                    setTimeout(function() {
                        loadingInProgressUI(false);
                    }, 1000);
                },

                errorCallback: function(e) {
                    // todo: better err handling?
                    AJS.logError(e);
                    loadingInProgressUI(false);
                }
            };

            loadingInProgressUI(true);

            MacroBrowser.Rest.fetchMacroPreviewData(options);
        }

        return {

            /**
             * Makes an ajax request to render the macro markup and updates the preview.
             *
             * @param macro
             */
            previewMacro: previewMacro,

            /**
             * Removes the preview container.
             *
             */
            removePreviewContainer: removePreviewContainer
        };
    };
});

require('confluence/module-exporter').safeRequire('confluence-macro-browser/macro-browser-preview');
