/**
 * @module confluence-macro-browser/macro-browser-rest
 */
define('confluence-macro-browser/macro-browser-rest', [
    'jquery',
    'ajs',
    'wrm',
    'confluence/api/constants'
], function(
    $,
    AJS,
    WRM,
    CONSTANTS
) {
    'use strict';

    /**
     * Encapsulates service rest calls
     */
    var MACRO_META_DATA_ENDPOINT = '/plugins/macrobrowser/browse-macros.action';
    var MACRO_META_DATA_DETAILS_ENDPOINT = '/plugins/macrobrowser/browse-macros-details.action';
    var MACRO_PREVIEW_ENDPOINT = '/rest/tinymce/1/macro/preview';
    var macroMetadataClientCacheKey = WRM.data.claim('confluence.macro.metadata.provider:macro-browser-metadata-resources.macro-meta-client-cache-key');

    var TIMEOUT = 30000; // mitigate CONF-29810

    function defaultErrorCallback(e) {
        AJS.logError(e);
    }

    function loadMacroMetadata(options) {
        options.data = options.data || {};
        // this key will be changed if macro metadata is changed in server
        options.data.macroMetadataClientCacheKey = macroMetadataClientCacheKey;

        return $.ajax({
            type: 'GET',
            cache: true,
            timeout: options.timeout || TIMEOUT,
            dataType: 'json',
            url: (options.context || CONSTANTS.CONTEXT_PATH) + MACRO_META_DATA_ENDPOINT,
            data: options.data,
            success: options.successCallback,
            error: options.errorCallback || defaultErrorCallback
        }).promise();
    }

    function fetchMacroMetadataDetails(options) {
        var data = { id: options.id };
        if (options.alternateId) {
            data.alternateId = options.alternateId;
        }

        $.ajax({
            type: 'GET',
            dataType: 'json',
            url: (options.context || CONSTANTS.CONTEXT_PATH) + MACRO_META_DATA_DETAILS_ENDPOINT,
            data: data,
            success: options.successCallback,
            error: options.errorCallback || defaultErrorCallback
        });
    }

    function fetchMacroPreviewData(options) {
        function createDataParameters(options) {
            var data = {
                contentId: options.contentId,
                macro: {
                    name: options.macroName,
                    body: options.body,
                    params: options.params,
                    defaultParameterValue: options.defaultParameterValue
                }
            };

            return $.toJSON(data);
        }

        if (!options.contentId || !options.macroName) {
            return (options.errorCallback || defaultErrorCallback)('invalid parameters');
        }

        $.ajax({
            type: 'POST',
            contentType: 'application/json; charset=utf-8',
            url: (options.context || CONSTANTS.CONTEXT_PATH) + MACRO_PREVIEW_ENDPOINT,
            data: createDataParameters(options),
            dataType: 'html',
            success: options.successCallback,
            error: options.errorCallback || defaultErrorCallback
        });
    }

    return {
        /**
         * Fetches metadata for registered macros/widgets
         *
         * @param {Object} options.data
         * @param {Object} options.timeout [XHR timeout = TIMEOUT]
         * @param {Object} options.context [Confluence base url = AJS.params.contextPath]
         * @param {Function} options.successCallback
         * @param {Function} [options.errorCallback = AJS.logError] - error callback or logging to console.
         */

        loadMacroMetadata: loadMacroMetadata,

        /**
         * Fetches more details for a particular widget on the server endpoint.
         *
         * @param {Object} options.id - widget id
         * @param {Object} options.context [Confluence base url = AJS.params.contextPath]
         * @param {Function} options.successCallback
         * @param {Function} [options.errorCallback = AJS.logError] - error callback or logging to console.
         */

        fetchMacroMetadataDetails: fetchMacroMetadataDetails,

        /**
         * Fetch macro preview
         * @param {Object} options.macroName Macro name
         * @param {Object} options.body Macro body
         * @param {Object} options.params Macro body
         * @param {Object} options.defaultParameterValue defaultParameterValue
         * @param {Object} options.contentId contentId
         * @param {Object} options.context [Confluence base url = AJS.params.contextPath]
         * @param {Function} options.successCallback
         * @param {Function} [options.errorCallback = AJS.logError] - error callback or logging to console.
         */

        fetchMacroPreviewData: fetchMacroPreviewData
    };
});

require('confluence/module-exporter').safeRequire('confluence-macro-browser/macro-browser-rest');
