/**
 * @module confluence-macro-browser/macro-browser
 */
define('confluence-macro-browser/macro-browser', [
    'window',
    'jquery',
    'ajs',
    'confluence-macro-browser/macro-browser-utils',
    'confluence-macro-browser/macro-browser-editor',
    'confluence-macro-browser/macro-browser-fields',
    'confluence-macro-browser/macro-browser-model',
    'confluence-macro-browser/macro-browser-preview',
    'confluence-macro-browser/macro-browser-rest',
    'confluence-macro-browser/macro-browser-UI',
    'confluence-macro-browser/macro-browser-smartfields',
    'confluence-macro-browser/macro-parameter-serializer',
    'confluence/macro-js-overrides',
    'underscore'
], function(
    window,
    $,
    AJS,
    MacroBrowserUtils,
    MacroBrowserEditor,
    MacroBrowserFields,
    MacroBrowserModel,
    MacroBrowserPreview,
    MacroBrowserRest,
    MacroBrowserUI,
    MacroBrowserSmartFields,
    MacroParameterSerializer,
    MacroJsOverrides,
    _
) {
    'use strict';

    var MacroBrowser = {};

    var macroBrowserFields = MacroBrowserFields(MacroBrowser);

    /**
     * @since 7.0.0
     * This function was originally in AUI 7
     * AUI 8 remove it, so we move it from AUI 7.9.7 to here
     * Because nowhere else is using this function
     *
     * Filters a list of entries by a passed search term.
     *
     * Options:
     * - `keywordsField` name of entry field containing keywords, default "keywords".
     * - `ignoreForCamelCase` ignore search case for camel case, e.g. CB matches Code Block *and* Code block.
     * - `matchBoundary` match words only at boundary, e.g. link matches "linking" but not "hyperlinks".
     * - `splitRegex` regex to split search words, instead of on whitespace.
     *
     * @param {Array} entries An array of objects with a "keywords" property.
     * @param {String} search One or more words to search on, which may include camel-casing.
     * @param {Object} options Specifiy to override default behaviour.
     *
     * @returns {Array}
     */
    function filterBySearch(entries, search, options) {
        var keywordsField;
        var camelCaseFlags;
        var boundaryFlag;
        var splitRegex;
        var filterWords;
        var filters;
        var subfilters;
        var camelRegexStr;
        var result;
        var i;
        var j;
        var somethingMatches;

        // search for nothing, get nothing - up to calling code to handle.
        if (!search) {
            return [];
        }

        keywordsField = (options && options.keywordsField) || 'keywords';
        camelCaseFlags = options && options.ignoreForCamelCase ? 'i' : '';
        boundaryFlag = options && options.matchBoundary ? '\\b' : '';
        splitRegex = (options && options.splitRegex) || /\s+/;

        // each word in the input is considered a distinct filter that has to match a keyword in the record
        filterWords = search.split(splitRegex);
        filters = [];

        filterWords.forEach(function(word) {
            // anchor on word boundaries
            subfilters = [new RegExp(boundaryFlag + word, 'i')];

            // split camel-case into separate words
            if (/^([A-Z][a-z]*) {2,}$/.test(this)) {
                camelRegexStr = this.replace(/([A-Z][a-z]*)/g, '\\b$1[^,]*');

                subfilters.push(new RegExp(camelRegexStr, camelCaseFlags));
            }

            filters.push(subfilters);
        });

        result = [];

        entries.forEach(function(entry) {
            for (i = 0; i < filters.length; i++) {
                somethingMatches = false;

                for (j = 0; j < filters[i].length; j++) {
                    if (filters[i][j].test(entry[keywordsField])) {
                        somethingMatches = true;
                        break;
                    }
                }

                if (!somethingMatches) {
                    return;
                }
            }

            result.push(entry);
        });

        return result;
    }

    MacroBrowser.ParameterFields = macroBrowserFields.ParameterFields;
    MacroBrowser.Field = macroBrowserFields.Field;

    // NOTE: This function call mutates MacroBrowser.
    MacroBrowserSmartFields(MacroBrowser, MacroBrowser.ParameterFields);

    MacroBrowser.Utils = MacroBrowserUtils;
    MacroBrowser.Editor = MacroBrowserEditor(MacroBrowser);
    MacroBrowser.Preview = MacroBrowserPreview(MacroBrowser);
    MacroBrowser.Model = MacroBrowserModel(MacroBrowser);
    MacroBrowser.Rest = MacroBrowserRest;
    MacroBrowser.UI = MacroBrowserUI(MacroBrowser);

    var loadMacroMetadataPromise;

    function onGoingPreloadRequest() {
        return loadMacroMetadataPromise && loadMacroMetadataPromise.state() === 'pending';
    }

    /**
     *
     * @param metadata
     * @param mode
     */
    var fetchMetaDataAndLoadMacro = function(metadata, mode) {
        var options = {
            id: metadata.macroName,
            successCallback: function(data) {
                if (data && data.details) {
                    data.details = MacroBrowser.Model.transformMetaDataDefault(data.details);
                }
                MacroBrowser.Editor.loadMacroInBrowser(data.details, mode);

                fetchMetaDataAndLoadMacro.displayDetails();
                MacroBrowser.Preview.previewMacro(data.details);
            },
            errorCallback: function(err) {
                AJS.trigger('analytics', { name: 'macro-browser.fetch-metadata-error' });
                window.alert(AJS.I18n.getText('macro.browser.load.error.message'));
                fetchMetaDataAndLoadMacro.displayDetails();
            }
        };

        if (metadata.alternateId) {
            options.alternateId = metadata.alternateId;
        }

        fetchMetaDataAndLoadMacro.setUI(mode, metadata.title);

        MacroBrowser.Rest.fetchMacroMetadataDetails(options);
    };

    fetchMetaDataAndLoadMacro.setUI = function(mode, title) {
        var placeHolderTitle = mode == 'edit' ? MacroBrowser.editTitle : MacroBrowser.insertTitle;

        $('#macro-insert-container').hide();
        MacroBrowser.UI.updateButtonText(mode);
        MacroBrowser.UI.enableSaveButton(false);
        MacroBrowser.dialog.gotoPage(1).addHeader(placeHolderTitle.replace(/\{0\}/, title));
        MacroBrowser.dialog.show();
        MacroBrowser.UI.showBrowserSpinner(true);
    };

    fetchMetaDataAndLoadMacro.displayDetails = function() {
        MacroBrowser.UI.showBrowserSpinner(false);
        MacroBrowser.UI.enableSaveButton(false);
        $('#macro-insert-container').show();
        MacroBrowser.UI.focusOnMacroDetailsFirstInput();
    };

    // this is absolutely horrible.
    // we should go with instance objects instead of singleton
    MacroBrowser.reset = function() {
        loadMacroMetadataPromise && loadMacroMetadataPromise.resolve && loadMacroMetadataPromise.resolve();
        loadMacroMetadataPromise = null;
        MacroBrowser.initMacroBrowserAfterRequest = null;
        MacroBrowser.initData = null;
        MacroBrowser.hasInit = false;
        MacroBrowser.metadataList = [];
        MacroBrowser.aliasMap = {};
        MacroBrowser.fields = {};
        MacroJsOverrides.reset();

        MacroBrowser.Macros = MacroJsOverrides.elements();
    };

    // This section has only been added so that the next Connect release and Confluence release aren't tightly coupled.
    // These should be removed soon. TODO: CRA-1219
    /**
     * @Deprecated Should be removed as soon as usages in connect have been removed.
     */
    MacroBrowser.getMacroJsOverride = MacroJsOverrides.get;

    /**
     * @Deprecated Should be removed as soon as usages in connect have been removed.
     */
    MacroBrowser.setMacroJsOverride = MacroJsOverrides.put;
    // End section

    /**
     * @deprecated Since 3.3. Macros is an ambiguous name, use confluence/macro-js-overrides.
     */
    MacroBrowser.Macros = MacroJsOverrides.elements();

    MacroBrowser.hasInit = false;
    MacroBrowser.metadataList = [];
    MacroBrowser.aliasMap = {}; // maps each alias to the corresponding macro name
    MacroBrowser.fields = {}; // stores fields for a given macro form.

    /**
     * Checks and returns true if all the required macro parameters have values.
     * It disables the insert/preview buttons if false.
     *
     * @returns {*}
     */
    MacroBrowser.processRequiredParameters = function() {
        return MacroBrowser.Editor.processRequiredParameters();
    };

    /**
     * Called when a parameter field value changes.
     */
    MacroBrowser.paramChanged = function() {
        // TODO - Could be used to preview?
        MacroBrowser.Editor.processRequiredParameters();
    };

    // Loads the given macro json in the browser's insert macro page.
    // Exposed to plugins (don'MacroBrowser remove)
    MacroBrowser.loadMacroInBrowser = function(metadata, mode) {
        MacroBrowser.Editor.loadMacroInBrowser(metadata, mode);
    };

    // Constructs the macro markup from the insert macro page
    MacroBrowser.getMacroDefinitionFromForm = function(metadata) {
        MacroBrowser.Editor.getMacroDefinitionFromForm(metadata);
    };

    /**
     * Returns a Map of all parameter values from the form, including the default parameter value which has a zero
     * length string as a key.
     * @param macroParamDetails meta data about each parameter in the macro
     */
    MacroBrowser.getMacroParametersFromForm = function(macro) {
        MacroBrowser.Editor.getMacroParametersFromForm(macro);
    };

    // Makes an ajax request to render the macro markup and updates the preview
    // Moved to different namespace. This is just a facade.
    MacroBrowser.previewMacro = function(macro) {
        MacroBrowser.Preview.previewMacro(macro);
    };

    // This gets called on the preview window's onload to re-adjust the height of the frame
    MacroBrowser.previewOnload = function(body) {
        var selectedMacroName = MacroBrowser.dialog.activeMetadata.macroName;
        var postPreview = MacroJsOverrides.getFunction(selectedMacroName, 'postPreview');
        if (postPreview) {
            postPreview($('#macro-preview-iframe')[0], MacroBrowser.dialog.activeMetadata);
        }
        AJS.Editor.disableFrame(body);

        // open all links in a new window
        $(body).click(function(e) {
            var windowOpened;
            if (e.target.tagName.toLowerCase() === 'a') {
                var a = e.target;
                var link = $(e.target).attr('href');
                if (link && link.indexOf('#') != 0 && link.indexOf(window.location) == -1) {
                    windowOpened = window.open(link, '_blank');
                    windowOpened.focus();
                    windowOpened.opener = null;
                }
                return false;
            }
        });
    };

    /**
     * Returns the macro metadata object for a given macro name.
     *
     * Call jsOverride.getMacroDetailsFromSelectedMacro instead of this method for macros such as "gadget" that map
     * multiple macros to a single name.
     *
     * @param macroName macro name to search metadata for
     */
    MacroBrowser.getMacroMetadata = function(macroName) {
        for (var i = 0, len = MacroBrowser.metadataList.length; i < len; i++) {
            var metadata = MacroBrowser.metadataList[i];
            if (metadata.macroName == macroName) {
                return metadata;
            }
        }
        return null;
    };

    /**
     * @since 5.6
     */
    MacroBrowser.getMetadataPromise = function() {
        return loadMacroMetadataPromise;
    };

    /**
     * Called when the user either clicks the Macro Browser button or clicks Edit in a
     * macro placeholder in the RTE.
     *
     * Note that the macro browser is not initialsed/loaded until opened for the first time.
     *
     * @param settings macro browser settings include:
     *      presetMacroMetadata : the metadata for a preset macro to load into the macro browser
     *      selectedHtml : string of selected HTML from RTE when no macro selected
     *      selectedText: the text contents of the selected HTML from the RTE when no macro selected
     *      onComplete : function to call when Macro Browser's "Insert" button is pressed
     *      onCancel : function to call when Macro Browser is closed when incomplete
     *      searchText : text to filter on if opening to the "Select Macro" page, if omitted no filter is done
     *      selectedCategory: the category name that should be selected by default
     */
    MacroBrowser.open = function(settings) {
        if (!settings) {
            settings = {};
            AJS.log('No settings to open the macro browser.');
        }
        var t = MacroBrowser;

        // if there is a custom editor for this macro, launch that instead
        var macro = settings.selectedMacro;

        if (!macro && settings.presetMacroMetadata) {
            macro = {
                name: settings.presetMacroMetadata.macroName
            };
        }

        if (macro && macro.name) {
            var opener = MacroJsOverrides.getFunction(macro.name, 'opener');
            if (opener) {
                opener(macro);
                return;
            }
        }

        if (!t.hasInit) { // init the macro browser for the first time
            AJS.debug('init macro browser');
            MacroBrowser.UI.showBrowserSpinner(true);

            if ((t.initData !== null) && $.isEmptyObject(t.initData)) {
                // the only case where we set initData={} is when we got an error on preloading from the rest endpoint
                // try preloading again..
                AJS.trigger('analytics', { name: 'macro-browser.init-reattempt' });
                AJS.logError('Macro browser preload failed. Trying again...');
                t.initMacroBrowserAfterRequest = settings;
                t.preLoadMacro();
                return;
            }
            if (t.initData) { // preloading completed
                t.initBrowser();
            } else { // ajax request not returned yet; set a flag to init the browser later
                AJS.trigger('analytics', { name: 'macro-browser.init-overlap' });
                AJS.debug('Waiting for macro browser preloading...');
                t.initMacroBrowserAfterRequest = settings;
                return;
            }
        }
        t.openMacroBrowser(settings);
    };

    /**
     * Open the Macro Browser Dialog, either to a specific macro or the selection page depending on
     * the settings parameter.
     * This method must be called after the dialog has been initialised
     *
     * @param {Object} settings.presetMacroName
     * @param {Object} settings.selectedCategory
     */
    MacroBrowser.openMacroBrowser = function(settings) {
        var t = MacroBrowser;
        t.settings = settings;
        t.selectedMacroDefinition = settings.selectedMacro;

        var selectedMacroName = (t.selectedMacroDefinition && t.selectedMacroDefinition.name)
            || settings.presetMacroName;

        // Preset macro overrides everything, just use it if present
        if (settings.presetMacroName) {
            settings.presetMacroMetadata = t.getMacroMetadata(settings.presetMacroName);
        }

        var metadata = settings.presetMacroMetadata;
        if (!metadata) {
            var selectedMacro = settings.selectedMacro;
            if (selectedMacro) {
                // Editing an existing macro - find metadata for it
                selectedMacroName = selectedMacro.name.toLowerCase();
                selectedMacroName = t.aliasMap[selectedMacroName] || selectedMacroName;

                var updateSelectedMacro = MacroJsOverrides.getFunction(selectedMacroName, 'updateSelectedMacro');
                var getMacroDetailsFromSelectedMacro = MacroJsOverrides.getFunction(selectedMacroName, 'getMacroDetailsFromSelectedMacro');

                if (updateSelectedMacro) {
                    updateSelectedMacro(selectedMacro);
                }

                if (getMacroDetailsFromSelectedMacro) {
                    metadata = getMacroDetailsFromSelectedMacro(t.metadataList, selectedMacro);
                }

                if (!metadata) {
                    metadata = t.getMacroMetadata(selectedMacroName);
                }
            }
        }

        // todo: get a reference to the back button, or create functions to show/hide the button
        // those functions should go on UI?
        var backButton = $('#macro-browser-dialog').find('button.back');

        if (metadata) {
            AJS.debug('Open macro browser to edit macro: ' + metadata.macroName);

            backButton.hide();

            // todo: should we remove this replicateSelectMacro method and use fetchMetaDataAndLoadMacro?
            t.replicateSelectMacro(metadata, settings.mode || 'edit');
        }
        // todo: using selectedMacroName var, which is declared above inside an if??
        else if (selectedMacroName) {
            // the user chose to edit a macro but no metadata was available
            backButton.show();

            // todo:  move to UI namespace??

            t.dialog.overrideLastTab();
            t.dialog.gotoPage(2);
            t.showBrowserDialog();
        } else {
            backButton.show();
            if (settings.selectedCategory) {
                // todo: extract category selection to function
                var categoryIndex = $('#select-macro-page .dialog-page-menu button').index($('#category-button-' + settings.selectedCategory));
                if (categoryIndex < 0) {
                    categoryIndex = 0;
                }
                t.dialog.overrideLastTab();
                t.dialog.gotoPanel(0, categoryIndex);
            } else {
                t.dialog.gotoPage(0);
            }
            t.showBrowserDialog();

            // If non-blank searchText has been passed in, filter on it
            t.dialog.searcher.focusAndSearch(settings.searchText);
        }
    };

    MacroBrowser.showBrowserDialog = function() {
        MacroBrowser.dialog.show();
        MacroBrowser.UI.showBrowserSpinner(false);
    };

    // Called when dialog is closed by Inserting/Saving.
    MacroBrowser.complete = function(dialog) {
        if (!$('#macro-browser-dialog .dialog-button-panel .ok').is(':visible:not(:disabled)')) {
            // If triggered by enter key but not ready to complete, ignore.
            return;
        }
        var t = MacroBrowser;
        var metadata = t.dialog.activeMetadata;

        var manipulateMarkup = MacroJsOverrides.getFunction(metadata.macroName, 'manipulateMarkup');

        if (manipulateMarkup) {
            manipulateMarkup(metadata);
        }

        var macroDefinition = MacroBrowser.Editor.getMacroDefinitionFromForm(metadata);

        t.close();
        if (t.settings.onComplete) {
            t.settings.onComplete(macroDefinition);
        }
    };

    /**
     * Called when dialog is closed by various cancel buttons or via Esc key.
     * @returns {boolean}
     */
    MacroBrowser.cancel = function() {
        var t = MacroBrowser;
        t.close();
        if (typeof t.settings.onCancel === 'function') {
            t.settings.onCancel();
        }
        return false;
    };

    MacroBrowser.close = function() {
        var t = this;
        t.unknownParams = {};
        t.fields = {};
        MacroBrowser.Preview.removePreviewContainer(); // CONFDEV-26807 makes sure the iFrame is removed
        t.dialog.hide();
    };

    /**
     * Replicates the user behaviour of selecting a macro and displaying the insert macro page
     * @param metadata
     * @param mode
     */
    MacroBrowser.replicateSelectMacro = function(metadata, mode) {
        fetchMetaDataAndLoadMacro(metadata, mode);
    };

    /**
     * Loads the categories and macros into the dialog
     * @returns {boolean}
     */
    MacroBrowser.initBrowser = function() {
        var t = MacroBrowser;
        var data = t.initData;

        // 1 . validation
        if (!data.categories || !MacroBrowser.metadataList.length) {
            AJS.trigger('analytics', { name: 'macro-browser.init-browser-error' });
            window.alert(AJS.I18n.getText('macro.browser.load.error.message'));
            MacroBrowser.UI.showBrowserSpinner(false);
            return false;
        }

        // 2. Save state
        t.editTitle = data.editTitle;
        t.insertTitle = data.insertTitle;

        // 3. Prepare data
        //--
        // sort the categories and macros
        // Skip the hidden category unless option is set, in which case append it to end of array.
        var hiddenCat;
        data.categories = $.map(data.categories, function(cat) {
            if (cat.name == 'hidden-macros') {
                hiddenCat = cat;
                return null;
            }
            return cat;
        });
        data.categories.sort(function(one, two) {
            return (one.displayName.toLowerCase() > two.displayName.toLowerCase() ? 1 : -1);
        });
        if (hiddenCat && AJS.params.showHiddenUserMacros) {
            data.categories.push(hiddenCat);
        }
        //--

        // 4. Build UI
        t.dialog = MacroBrowser.UI.createDialog({

            title: data.title,
            categories: data.categories,
            macros: t.metadataList,
            onClickMacroSummary: function(e, metadata) {
                e.preventDefault();
                fetchMetaDataAndLoadMacro(metadata, 'insert');
            },
            onSubmit: t.complete,
            onCancel: t.cancel

        });

        t.hasInit = true;

        return true;
    };

    /**
     * Search macro name, title and description for the given text.
     * @param text (required) text to search on
     * @param options options to pass to the AJS.filterBySearch method
     * @return array of macro summaries matching the search text.
     */
    MacroBrowser.searchSummaries = function(text, options) {
        options = $.extend({
            splitRegex: /[\s-]+/
        }, options);
        return filterBySearch(MacroBrowser.metadataList, text, options);
    };

    /**
     * Convenience method to return the macro body. This will be one of -
     * - the text in the body textarea on the macro browser dialog
     * - currently selected text in the editor
     * - the current body if editing an existing macro (disregarding any editor selection)
     * - empty String if neither previous case applies.
     */
    MacroBrowser.getMacroBody = function() {
        var t = MacroBrowser;
        var body = '';
        if ($('#macro-insert-container .macro-body-div textarea').length) {
            body = $('#macro-insert-container .macro-body-div textarea').val();
        } else if (t.selectedMacroDefinition) {
            if (t.selectedMacroDefinition.body) {
                body = t.selectedMacroDefinition.body;
            }
        } else if (t.dialog.activeMetadata) {
            body = t.dialog.activeMetadata.formDetails.body.content; // this is encoded as appropriate for the current macro
        }
        return body;
    };

    /**
     * Convenience method to return a macro's parameters from a macro's node.
     * @param macroNode
     * @returns {*} Object containing a property for each parameter
     */
    MacroBrowser.getMacroParams = function(macroNode) {
        return MacroParameterSerializer.deserialize(macroNode.getAttribute('data-macro-parameters'));
    };

    function getMacroNameFromDom(macroNode) {
        if (!macroNode.hasAttribute('data-macro-name')) {
            return undefined;
        }
        return macroNode.getAttribute('data-macro-name');
    }

    /**
     * Convenience method to return a macro's name from a macro's node.
     * @param {HTMLElement} macroNode
     * @returns {string|undefined} The macro's name.
     */
    MacroBrowser.getMacroName = getMacroNameFromDom;

    /**
     * Checks if a normally-hidden macro should be displayed in the UI or not.
     * @param macro - a macro summary
     */
    MacroBrowser.isHiddenMacroShown = function(macro) {
        return AJS.params.showHiddenUserMacros && macro.pluginKey == '_-user-macro-_';
    };

    /**
     * Returns true if the metadata indicates that there are required parameters for this macro
     */
    MacroBrowser.hasRequiredParameters = function(macroMetadata) {
        return macroMetadata.anyParameterRequired;
    };

    /**
     * Provide a CSV of macroNames to include only these macros. Has to be called before preLoadMacro() runs.
     *
     * E.g. "code, cheese"
     *
     * @param whitelist
     */
    MacroBrowser.setWhitelist = function(whitelist) {
        MacroBrowser.whitelist = whitelist;
    };

    /**
     * Preloads macro metadata
     * @return promise
     */
    MacroBrowser.preLoadMacro = function() {
        var t = MacroBrowser;

        t.initData = null; // flag async request started

        if (onGoingPreloadRequest()) {
            return loadMacroMetadataPromise;
        }

        loadMacroMetadataPromise = MacroBrowser.Rest.loadMacroMetadata({

            // todo: currently you can either have a whitelist or drop the details.  Both is to-be-implemented
            data: t.whitelist ? { whitelist: t.whitelist } : { detailed: false },

            successCallback: function(data) {
                t.initData = data;
                MacroBrowser.Model.loadMacros(data.macros);
                if (t.initMacroBrowserAfterRequest) { // we have an existing "open browser" command in the queue waiting for the preloading operation to complete
                    t.initBrowser();
                    t.openMacroBrowser(t.initMacroBrowserAfterRequest);
                }
            },
            errorCallback: function(e) {
                AJS.trigger('analytics', { name: 'macro-browser.preload-error' });
                AJS.logError('Error requesting macro browser metadata:');
                AJS.logError(e);
                t.initData = {}; // empty initData is used as an error flag in the "open" method
            }
        });
    };

    return MacroBrowser;
});

require('confluence/module-exporter')
    .exportModuleAsGlobal('confluence-macro-browser/macro-browser', 'AJS.MacroBrowser');
