// This module depends on the "com.atlassian.auiplugin:aui-select2" but this web-resource doesn't provide the AMD module
define('jira-integration-plugin/fields', [
    'jquery',
    'jira-integration-plugin/lodash',
    'jira-integration-plugin/label-picker',
], function ($, _, labelPicker) {
    var templates = window.jiraIntegration.templates;

    var datePickerFormat = 'YYYY-MM-DD';
    var stringHandler = {
        template: templates.fields.stringField,
        getContext: getStringContext,
        getValue: getStringValue,
    };
    var urlHandler = {
        template: templates.fields.stringField,
        getContext: getUrlContext,
        getValue: getStringValue,
    };
    var assigneeHandler = {
        template: templates.fields.stringField,
        getContext: getUserContext,
        getInternalValue: getSelectValueInternal,
        getValue: getAssigneeValue,
        behavior: addAssigneeAutocompleteBehaviour,
    };
    var userHandler = {
        template: templates.fields.stringField,
        getContext: getUserContext,
        getValue: getUserValue,
        behavior: addUserAutoCompleteBehavior,
    };
    var labelTagHandler = {
        template: templates.fields.arrayField,
        getContext: getArrayContext,
        getValue: getArrayValue,
        behavior: addLabelAutoCompleteBehavior,
    };
    var textareaHandler = {
        template: templates.fields.textareaField,
        getContext: getStringContext,
        getValue: getStringValue,
    };
    var numberHandler = {
        template: templates.fields.numberField,
        getContext: getNumberContext,
        getValue: getNumberValue,
    };
    var arrayHandler = {
        template: templates.fields.arrayField,
        getContext: getArrayContext,
        getValue: getArrayValue,
    };
    var allowedValuesHandler = {
        template: templates.fields.allowedValuesField,
        getContext: getAllowedValuesContext,
        getValue: _.bind(getAllowedValuesBase, null, 'id'),
        behavior: addSelect2Behavior,
    };
    var multipleChoicesHandler = {
        template: templates.fields.allowedValuesField,
        getContext: getAllowedValuesContext,
        getValue: _.bind(getAllowedValuesBase, null, 'id'),
    };
    var timeTrackingHandler = {
        template: templates.fields.timeTrackingField,
        getContext: getTimeTrackingContext,
        getValue: getTimeTrackingValue,
    };
    var dateHandler = {
        template: templates.fields.dateField,
        getContext: getDateContext,
        getValue: getDateStringValue,
        behavior: datePickerBehavior,
    };
    var select2WithIconHandler = {
        template: templates.fields.select2WithIconField,
        getContext: getSelect2WithIconContext,
        getValue: _.bind(getAllowedValuesBase, null, 'id'),
        behavior: addSelect2WithIconBehavior,
    };
    var unsupportedHandler = {
        template: templates.fields.unrenderableTypeField,
        getContext: getUnsupportedContext,
    };
    var sprintHandler = {
        template: templates.fields.stringField,
        getContext: getStringContext,
        getInternalValue: getSelectValueInternal,
        getValue: getNumberValue,
        behavior: addSprintBehavior,
    };
    var epicHandler = {
        template: templates.fields.stringField,
        getContext: getStringContext,
        getInternalValue: getSelectValueInternal,
        getValue: getEpicValue,
        behavior: addEpicBehavior,
    };
    var checkboxHandler = {
        template: templates.fields.checkboxField,
        getContext: getCheckedFieldContext,
        getValue: getMultipleCheckedFieldsValue,
    };
    var radioButtonsHandler = {
        template: templates.fields.radioField,
        getContext: getCheckedFieldContext,
        getValue: getCheckedFieldValue,
    };

    // WARNING: if you add (or remove) from this list you should also update the analytics whitelist
    // in plugin/src/main/resources/analytics/jira-integration-analytics.json - this only applies to
    // complex types like "com.pyxis.greenhopper.jira:gh-epic-label", simple values are already in the
    // allowed dictionary
    var restTypes = {
        'com.pyxis.greenhopper.jira:gh-epic-label': stringHandler,
        'string': stringHandler,
        'summary': stringHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:textfield': stringHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:url': urlHandler,
        'environment': textareaHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:textarea': textareaHandler,
        'description': textareaHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:float': numberHandler,
        'array': arrayHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:labels': arrayHandler,
        'labels': labelTagHandler,
        'priority': select2WithIconHandler,
        'resolution': allowedValuesHandler,
        'fixVersions': allowedValuesHandler,
        'versions': allowedValuesHandler,
        'components': allowedValuesHandler,
        'security': allowedValuesHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:version': allowedValuesHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:multiversion': allowedValuesHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:project': allowedValuesHandler,
        'assignee': assigneeHandler,
        'reporter': userHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:userpicker': userHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:multiuserpicker': userHandler,
        'timetracking': timeTrackingHandler,
        'duedate': dateHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:datepicker': dateHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:multiselect': multipleChoicesHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:select': multipleChoicesHandler,
        'com.pyxis.greenhopper.jira:gh-sprint': sprintHandler,
        'com.pyxis.greenhopper.jira:gh-epic-link': epicHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:multicheckboxes': checkboxHandler,
        'com.atlassian.jira.plugin.system.customfieldtypes:radiobuttons': radioButtonsHandler,
    };

    var defaultRenderOptions = {
        // Filter out fields that has default value ~ to make the form shorter
        ignoreFieldsWithDefaultValue: false,
    };

    function getContext (issue, restField, values, errors) {
        var name = restField.schema.system || 'customfield_' + restField.schema.customId;
        var restTypeId = restField.schema.system || restField.schema.custom || restField.schema.customId;
        var restType = restTypes[restTypeId];
        var isRenderable = Boolean(restType && (!restType.canRender || restType.canRender(restField)));
        var hasOperations = Boolean(restField.operations && restField.operations.length);
        var handler = isRenderable && hasOperations ? restType : unsupportedHandler;
        var extraAttributes = restField.required && {'data-aui-validation-field': '', 'required': true};

        var baseContext = {
            labelText: restField.name,
            name: name,
            isRequired: restField.required,
            value: values[name],
            errorTexts: errors[name],
            jiraType: restTypeId,
            isRenderable: isRenderable,
            hasOperations: hasOperations,
            handler: handler,
            extraAttributes: extraAttributes,
        };

        return handler.getContext(baseContext, restField, issue, /* deprecated, use context.value */ values);
    }

    function getUnsupportedContext (context, restField, issue) {
        if (!context.isRenderable) {
            context.reasonContent = issue ?
                                    AJS.I18n.getText('fields.unrenderable', '<a href="' + issue.url + '">', '</a>') :
                                    AJS.escapeHtml(AJS.I18n.getText('fields.create.unrenderable'));
        } else if (!context.hasOperations) {
            context.reasonContent = issue ?
                                    AJS.I18n.getText('fields.no.permission', '<a href="' + issue.url + '">', '</a>') :
                                    AJS.escapeHtml(AJS.I18n.getText('fields.create.no.permission'));
        } else {
            context.reasonContent = null;
        }
        return context;
    }

    function getStringContext (context, restField, issue) {
        var name = context.name;
        context.value = ($.isPlainObject(context.value) ? context.value.name : context.value) ||
            (issue && issue.fields[name]) || (restField && restField.defaultValue) || '';
        return context;
    }

    function getStringValue ($fieldInput) {
        return $fieldInput.val();
    }

    function getUrlContext (context, restField, issue) {
        var urlValidation = {
            'data-aui-validation-field': '',
            'pattern': '([a-zA-Z][a-zA-Z0-9\-+\.]*:\/\/.+)|^$',
            'data-aui-validation-pattern-msg': AJS.I18n.getText('field.url.format.validation.error'),
        };
        return $.extend(true, {}, getStringContext(context, restField, issue), {extraAttributes: urlValidation});

    }

    function getDateContext (context, restField, issue) {
        var dateValidation = {
            'data-aui-validation-field': '',
            // AUI dateformat validator does not allow for empty dates so we need to use regex validation
            'pattern': '^([0-9]{2,4}-(0?[1-9]|10|11|12)-([0-2]?[1-9]|10|20|30|31))$|^$',
            'data-aui-validation-pattern-msg': AJS.I18n.getText('field.date.format.validation.error'),
        };
        return $.extend(true, {}, getStringContext(context, restField, issue), {extraAttributes: dateValidation});
    }

    function getDateStringValue ($fieldInput) {
        var dateString = $fieldInput.val();
        if (dateString === '') {
            return null;
        } else {
            return dateString;
        }
    }

    function getNumberContext (context, restField, issue) {
        var numberValidation = {
            'data-aui-validation-field': '',
            'pattern': '(([0-9]*[.])?[0-9]+)|^$',
            'data-aui-validation-pattern-msg': AJS.I18n.getText('field.number.format.validation.error'),
        };
        return $.extend(true, {}, getStringContext(context, restField, issue), {extraAttributes: numberValidation});
    }

    function getNumberValue ($fieldInput) {
        var valString = $fieldInput.val();
        if (isNumeric(valString)) {
            return Number(valString);
        }
        return valString || null;
    }

    function datePickerBehavior ($fieldInput) {
        var $input = $fieldInput.find('input');

        // AUI datepicker has many problems on IE (issue: AUI-2071), so if user using IE, we will render it as textfield
        // This issue was fixed by Issac Gerges and this fix will be effect from AUI 5.3.5
        if (!!navigator.userAgent.match(/Trident/) && AJS.version < '5.3.5') {
            var isPlaceholderSupported = 'placeholder' in document.createElement('input');
            $input.attr('placeholder', datePickerFormat);

            // Support placeholder on IE 8-9
            if (!isPlaceholderSupported) {
                $input.on('focus', function () {
                    if ($input.val() === $input.attr('placeholder')) {
                        $input.val('');
                    }
                }).on('blur', function () {
                    if ($input.val() === '') {
                        $input.val($input.attr('placeholder'));
                    }
                }).blur();
            }
        } else {
            WRM.require('wr!com.atlassian.auiplugin:aui-date-picker').done(function () {
                $input.datePicker({
                    'overrideBrowserDefault': true,
                });
            });
        }
    }

    function getArrayContext (context, restField, issue) {
        var name = context.name;
        context.value = (context.value && context.value.join(',')) ||
            (issue && issue.fields[name] && issue.fields[name].join(',')) ||
            (restField && restField.defaultValue && restField.defaultValue.join(' '));
        return context;
    }

    function getArrayValue ($fieldInput) {
        return _.map($fieldInput.val().split(','), $.trim);
    }

    function getAllowedValuesContext (context, restField, issue) {
        var name = context.name;
        var userInputValue = context.value;
        var issueValue = issue && issue.fields[name];
        var defaultFieldValue = restField && restField.defaultValue;
        var selectedValue;

        function extractSelectedValue (inputValue) {
            if (Array.isArray(inputValue)) {
                return inputValue.map(function (val) {
                    return val.name || val.id;
                });
            } else {
                return [inputValue.name || inputValue.id];
            }
        }

        if (userInputValue) {
            selectedValue = extractSelectedValue(userInputValue);
        } else if (issueValue) {
            selectedValue = extractSelectedValue(issueValue);
        } else if (defaultFieldValue) {
            selectedValue = extractSelectedValue(defaultFieldValue);
        } else {
            selectedValue = [];
        }
        context.options = _.map(restField.allowedValues, function (val) {
            return {
                value: val.id,
                text: val.name || val.value,
                selected: selectedValue.includes(val.name || val.id),
            };
        });

        // if this is an "option" field then add a "None" option if it's not required
        if (!restField.required && restField.schema.type === 'option') {
            context.options.unshift({
                value: -1,
                text: AJS.I18n.getText('fields.allowedvalues.none'),
            });
        }

        context.isMultiple = restField.operations.includes('add');
        delete context.value;
        return context;
    }

    function getAllowedValuesBase (propertyName, $fieldInput) {
        var val = $fieldInput.val();
        var multiple = $fieldInput.prop('multiple');
        var getValue = function (val) {
            var valueObject = {};
            valueObject[propertyName] = val;
            return valueObject;
        };
        if (multiple) {
            return Array.isArray(val) ?
                   _.map(val, getValue) :
                [getValue(val)];
        }
        return getValue(val);
    }

    function addSelect2Behavior ($inputField) {
        // Select2-fy multi selects
        $inputField.find('select[multiple]').auiSelect2();
    }

    function getUserContext (context, restField, issue) {
        var name = context.name;
        context.value = (context.value && context.value.name) ||
            (issue && issue.fields[name] && issue.fields[name].name) ||
            // For the Assignee field the "automatic" default value is special
            (getFieldTypeFromRestField(restField) === 'assignee' ? -1 : '') ||
            // For the userpicker field
            (restField && restField.defaultValue && restField.defaultValue.name);
        context.extraClasses = restField.schema.type === 'array' ? 'multi-user-picker' : 'single-user-picker';
        return context;
    }

    function getUserValue ($fieldInput) {
        var getValue = function (val) {
            return {
                name: val,
            };
        };
        var val = $fieldInput.val();
        var multiple = $fieldInput.closest('.jira-field').is('.multi-user-picker');

        if (!val) {
            return multiple ? [] : null;
        }

        return multiple ?
               _.map(val.split(','), getValue) :
               getValue(val);
    }

    function getAssigneeValue ($fieldInput) {
        var value = $fieldInput.val();

        if (value === '-1') {
            // Automatic
            return undefined;
        }

        // for unassigned value will be blank, but that's OK
        return {
            name: value,
        };
    }

    function getCheckedFieldContext (context, restField, issue) {
        var checked = context.value || (issue && issue.fields[context.name]) ||
            (restField && restField.defaultValue);
        context.fields = _.map(restField.allowedValues, function (option) {
            return {
                id: option.id,
                value: option.id,
                labelText: option.name || option.value,
                isChecked: Array.isArray(checked) ?
                           checked.some(function (item) {
                               return item.id === option.id;
                           }) :
                           checked && checked.id === option.id,
            };
        });

        delete context.value;
        return context;
    }

    function getMultipleCheckedFieldsValue ($fieldset) {
        return $fieldset.find('input:checked').toArray().map(function (field) {
            var $field = $(field);
            return {
                id: $field.prop('value'),
                value: $field.parent().find('label').text(),
            };
        });
    }

    function getCheckedFieldValue ($fieldset) {
        var checkedValue = getMultipleCheckedFieldsValue($fieldset);
        return checkedValue.length === 1 ? checkedValue[0] : null;
    }

    /**
     * Add auto complete behavior to text field using auiSelect2
     */
    function addAutoCompleteBehavior ($inputField, context, restType, issue, autoCompleteConfig) {
        var $input = $inputField.find('input');
        var fieldName = $inputField.attr('name');
        $input.removeClass('text').removeClass('long-field').addClass('medium-long-field'); // to avoid layout problem

        var defaultConfig = {
            minimumInputLength: 1,
            id: fieldName,
            name: fieldName,
            query: function (query) {
                function onsuccess (datas) {
                    query.callback({results: datas});
                }

                getAutoCompleteData(context, restType, query.term, issue).done(onsuccess);
            },
        };

        $input.auiSelect2($.extend(defaultConfig, autoCompleteConfig));

        // passing the data to Select2 doesn't actually update the value, so this needs to be done separately for the
        // result to be shown in the select bar
        var data = $input.auiSelect2('data');
        if (data && data.id !== -1) {
            $input.auiSelect2('val', data.id);
        }

        $inputField.find('div.aui-select2-container').addClass('jira-select2-drop-box');
    }

    /**
     * Add user auto complete behavior to text field using auiSelect2
     */
    function addUserAutoCompleteBehavior ($inputField, context, restType, issue) {
        var defaultValue = ($inputField.find('.long-field').prop('value')) || '';
        var userConfig = {
            formatInputTooShort: function () {
                return AJS.I18n.getText('field.user.search.prompt');
            },
            initSelection: _createInitSelection(context, {
                id: -1,
                text: defaultValue,
                isSystemOption: false,
            }),
        };

        addUsersAutoCompleteBehavior($inputField, context, restType, issue, userConfig);
    }


    function addUsersAutoCompleteBehavior ($inputField, context, restType, issue, config) {
        var usersAutoCompleteConfig = $.extend({},
            config,
            {
                multiple: restType === 'com.atlassian.jira.plugin.system.customfieldtypes:multiuserpicker',
                formatResult: function (result) {
                    return templates.fields.userOptionSelect({
                        name: result.id,
                        displayName: result.text,
                        isSystemOption: !!result.isSystemOption,
                    });
                },
            }
        );

        addAutoCompleteBehavior($inputField, context, restType, issue, usersAutoCompleteConfig);
    }

    /**
     * Adds Assignee specific autocomplete behaviour - this differs from a regular user picker in that
     * it allows for "Automatic" and "Unassigned" as options, in addition to a user search.
     */
    function addAssigneeAutocompleteBehaviour ($inputField, context, restType, issue) {
        var assigneeAutoCompleteConfig = {
            minimumInputLength: 0, // we check this in `query` to allow us to display the system options
            initSelection: _createInitSelection(context, {
                id: -1,
                text: AJS.I18n.getText('dialog.assignee.automatic'),
                isSystemOption: true,
            }),
            query: function (query) {
                function onsuccess (datas) {
                    // only include system props on the first page
                    if (query.page === 1) {
                        // if the field is required then "Unassigned" isn't allowed
                        if (!query.element.prop('required')) {
                            datas.unshift({
                                id: '',
                                text: AJS.I18n.getText('dialog.assignee.none'),
                                isSystemOption: true,
                            });
                        }
                        datas.unshift({
                            id: -1,
                            text: AJS.I18n.getText('dialog.assignee.automatic'),
                            isSystemOption: true,
                        });
                    }
                    query.callback({results: datas});
                }

                // only do a real search if we have at least 1 character, otherwise just go straight to
                // onsuccess which will show the system options.
                if (query.term.length > 0) {
                    getAutoCompleteData(context, restType, query.term, issue).done(onsuccess);
                } else {
                    onsuccess([]);
                }
            },
        };

        addUsersAutoCompleteBehavior($inputField, context, restType, issue, assigneeAutoCompleteConfig);
    }

    function addLabelAutoCompleteBehavior ($inputField, context, restType, issue) {
        var $input = $inputField.find('input');

        labelPicker.build($input,
            function (term) {
                return getAutoCompleteData(context, restType, term, issue);
            }
        );
    }

    /**
     * Generic method to get auto complete data from REST resource
     */
    function getAutoCompleteData (context, restType, term, issue) {
        var postData = $.extend({restType: restType, issueKey: (issue && issue.key) || '', term: term}, context);
        return $.ajax({
            type: 'POST',
            timeout: 0,
            contentType: 'application/json',
            dataType: 'json',
            url: AJS.contextPath() + '/rest/jira-integration/latest/fields/autocomplete',
            data: JSON.stringify(postData),
        });
    }

    function getTimeTrackingContext (context, restField, issue) {
        var timeTrackingValidation = {
            'data-aui-validation-field': '',
            'pattern': '(([0-9]+w|[0-9]+d|[0-9]+m|[0-9]+h|[0-9]+)\\s*)*', // interpreted from TimeTrackingHelp in Jira
            'data-aui-validation-pattern-msg': AJS.I18n.getText('field.timetracking.format.validation.error'),
        };
        context.value = (context.value && context.value.remainingEstimate) ||
            (issue && issue.fields[name] && issue.fields[name].remainingEstimate) || '';
        return $.extend(true, {}, context, {extraAttributes: timeTrackingValidation});
    }

    function getTimeTrackingValue ($fieldInput) {
        return {
            remainingEstimate: $fieldInput.val(),
        };
    }

    /**
     * Add select with icon behavior to select field using auiSelect2
     */
    function getSelect2WithIconContext (context, restField, issue) {
        var name = context.name;
        var selectedValueId = (context.value && context.value.id) ||
            (restField && restField.defaultValue && restField.defaultValue.id) || '';
        var selectedValueName = (context.value && context.value.name) ||
            (issue && issue.fields && issue.fields[name] && issue.fields[name].name) ||
            (restField && restField.defaultValue && restField.defaultValue.id) || '';
        delete context.value;

        context.options = _.map(restField.allowedValues, function (val) {
            return {
                value: val.id,
                text: val.name,
                selected: selectedValueName === val.name || selectedValueId === val.id,
                iconUrl: val.iconUrl, //add iconUrl attribute for option element to display icon
            };
        });
        return context;
    }

    function select2WithIconFieldFormat (state) {
        var iconUrl;
        if (state.id) {
            iconUrl = $(state.element).attr('data-icon-url');
        }

        return templates.fields.select2WithIconOption({
            'optionValue': state.text,
            'iconUrl': iconUrl,
        });
    }

    function addSelect2WithIconBehavior ($field, context, restType, issue) {
        if (!$.fn.auiSelect2) {
            AJS.log('AUI version 5.2 or greater is required as this plugin needs the .auiSelect2() jQuery plugin.');
            return;
        }

        var $select = $field.find('select');
        $select.addClass('jira-select2-drop-box');
        $select.auiSelect2({
            hasAvatar: true,
            minimumResultsForSearch: -1,
            formatSelection: select2WithIconFieldFormat,
            formatResult: select2WithIconFieldFormat,
        });
    }

    function addSprintBehavior ($inputField, context, restType, issue) {
        var config = {
            minimumInputLength: 0,
            formatResult: function (result) {
                return templates.fields.sprintSelect({
                    name: result.text,
                    state: result.state,
                    board: result.board,
                });
            },
            query: function (query) {
                getAutoCompleteData(context, restType, query.term, issue).done(function (data) {
                    var grouped = _.groupBy(data, function (item) {
                        return item.suggestion ? 'suggestions' : 'all';
                    });

                    var results = ['suggestions', 'all'] // rather than Object.keys to keep order
                        .filter(function (key) {
                            return grouped[key] && grouped[key].length > 0;
                        })
                        .map(function (key) {
                            return {
                                text: key === 'suggestions' ?
                                      AJS.I18n.getText('field.sprint.suggestions') :
                                      AJS.I18n.getText('field.sprint.all'),
                                children: grouped[key],
                            };
                        });

                    query.callback({results: _.compact(results)});
                });
            },
            initSelection: _createInitSelection(context, {
                id: -1,
                text: AJS.I18n.getText('dialog.field.sprint.placeholder'),
            }),
        };
        return addAutoCompleteBehavior($inputField, context, restType, issue, config);
    }

    function addEpicBehavior ($inputField, context, restType, issue) {
        var config = {
            minimumInputLength: 0,
            formatResult: function (result) {
                return templates.fields.epicSelect({
                    name: result.text,
                    key: result.id,
                });
            },
            query: function (query) {
                getAutoCompleteData(context, restType, query.term, issue).done(function (data) {
                    var groupedData = _.uniq(data.map(function (item) {
                        return item.list;
                    })).map(function (list) {
                        return {
                            text: list,
                            children: data.filter(function (item) {
                                return item.list === list;
                            }),
                        };
                    });

                    query.callback({results: groupedData});
                });
            },
            initSelection: _createInitSelection(context, {
                id: -1,
                text: AJS.I18n.getText('dialog.field.epic.placeholder'),
            }),
        };
        return addAutoCompleteBehavior($inputField, context, restType, issue, config);
    }

    function getEpicValue ($fieldInput) {
        var value = getStringValue($fieldInput);
        return value && value !== '' ? value : undefined;
    }

    function getSelectValueInternal ($fieldInput) {
        var value = $fieldInput.auiSelect2('data');
        return value && value.id === -1 ? undefined : value;
    }

    /**
     * @param type {string} The Jira string to expect listed as 'schema.system' or 'schema.customId' in the REST response
     * @param handler {{
     *         template : function(Object) : string,
     *         getContext : function(baseContext, restField, issue) :Object,
     *         getValue : function($renderedInput) : Object
     *         canRender : ?function(restField) : boolean
     *     }}
     *     getContext produces an object that the template can read
     *     template produces a string of HTML that contains an <input>, <textarea> or <select> whose name matches baseContext.name
     *     getValue takes in a jQuery object that represents the <input>, <textarea> or <select> form the template.
     *     canRender takes in a restField and returns true if it can be rendered, or false if it can't. This is optional and defaults to always returning true.
     */
    function addFieldHandler (type, handler) {
        if (_.has(restTypes, type) && console && console.warn) {
            console.warn('Redefining handler for type ' + type + '.');
        }
        restTypes[type] = handler;
    }

    function getFieldTypeFromRestField (restField) {
        return restField.schema ? (restField.schema.system || restField.schema.custom || restField.schema.customId) : restField;
    }

    function getFieldHandler (restField) {
        return restTypes[getFieldTypeFromRestField(restField)];
    }

    function getFieldHandlerFromInputField ($inputField) {
        var typeId = getRestTypeFromInputField($inputField);
        return typeId && getFieldHandler(typeId);
    }

    function getRestTypeFromInputField ($inputField) {
        return $inputField.closest('.jira-field').attr('data-jira-type');
    }

    function getCreateRequiredFieldsMeta (context, options) {
        return $.ajax({
            type: 'GET',
            timeout: 0,
            url: AJS.contextPath() + '/rest/jira-integration/1.0/servers/' + context.serverId
                + '/projects/' + context.projectKey + '/issue-types/' + context.issueType + '/fields-meta',
        }).pipe(function (data) {
            var fieldsMeta = [];
            _.each(data.fields, function (field) {
                var typeId = getFieldTypeFromRestField(field);
                if (field.required && !options.excludedFields.includes(typeId)) { // using key instead to avoid internationalize problem
                    fieldsMeta.push(field);
                }
            });
            return fieldsMeta;
        });
    }

    function isNumeric (stringNumber) {
        return /\d/.test(stringNumber) && /^-?\d*\.?\d*$/.test(stringNumber);
    }

    function attachFieldBehavior ($fieldInput, context, issue) {
        var typeId = getRestTypeFromInputField($fieldInput);
        var handler = typeId && getFieldHandler(typeId);
        var behavior = handler && handler.behavior;
        if (behavior) {
            behavior($fieldInput, context, typeId, issue);
        }
    }

    function getJSON ($fieldInput) {
        var handler = getFieldHandlerFromInputField($fieldInput);
        return handler && handler.getValue && handler.getValue($fieldInput);
    }

    function _createInitSelection (context, defaultValue) {
        return function (elem, callback) {
            var value = context[elem.attr('id')];
            callback(value || defaultValue);
        };
    }

    var JiraIntegrationFields = {
        addFieldHandler: addFieldHandler,
        getFieldHandler: getFieldHandler,
        getFieldType: getFieldTypeFromRestField,
        isKnownRestType: function (restTypeId) {
            return restTypes.hasOwnProperty(restTypeId);
        },
        canRender: function (restField) {
            var restTypeId = getFieldTypeFromRestField(restField);
            var restType = restTypes[restTypeId];

            if (!restType) {
                return false;
            }

            return restField.operations && restField.operations.length && (!restType.canRender || restType.canRender(restField));
        },
        getContext: function (issue, restField, values, errors) {
            return getContext(issue, restField, values || {}, errors || {});
        },
        renderField: function (issue, restField, values, errors) {
            var context = getContext(issue, restField, values || {}, errors || {});
            return context.handler.template(context);
        },
        // in order to render selected value some fields require the data in different format than that
        // added to the request (provided by getJSON)
        getInternalJSON: function ($fieldInput) {
            var handler = getFieldHandlerFromInputField($fieldInput);
            return handler && handler.getInternalValue && handler.getInternalValue($fieldInput) || getJSON($fieldInput);
        },
        getJSON: getJSON,
        attachFieldBehavior: attachFieldBehavior,
        attachFieldBehaviors: function ($container, context, issue) {
            $container.find('.jira-field').each(function (index, fieldInput) {
                var $fieldInput = $(fieldInput);
                attachFieldBehavior($fieldInput, context, issue);
            });
        },
        setFieldError: function ($field, errors) {
            var $existingErrors = $field.find('.error');
            $existingErrors.remove();
            if (!errors) {
                return;
            }
            var errorHtml = templates.fields.errors({
                errorTexts: errors,
            });
            $field.append(errorHtml);
        },
        renderCreateRequiredFields: function ($container, afterElement, context, options, errorCallback) {
            options = _.extend({}, defaultRenderOptions, options);

            function renderFields (fields) {
                if (options.ignoreFieldsWithDefaultValue) {
                    fields = _.filter(fields, function (field) {
                        return !field.hasDefaultValue;
                    });
                }

                var unsupportedFields = _.filter(fields, function (field) {
                    return !JiraIntegrationFields.canRender(field);
                });
                if (unsupportedFields.length) {
                    if (errorCallback) {
                        errorCallback(unsupportedFields);
                    }
                    return;
                }
                $container.html(_.map(fields, function (field) {
                    return JiraIntegrationFields.renderField(null, field, null, null);
                }).join(''));

                JiraIntegrationFields.attachFieldBehaviors($container, {
                    serverId: context.serverId,
                    projectKey: context.projectKey,
                }, null);
            }

            // if requiredFields is provided use it to render, otherwise make request to get requiredFields
            if (options.requiredFields) {
                renderFields(options.requiredFields);
            } else {
                getCreateRequiredFieldsMeta(context, options).done(renderFields);
            }
        },
    };

    return JiraIntegrationFields;
});
