define('jira-integration-plugin/jira-create-issue-form-data', ['jquery'], function ($) {
    var servers = {};
    var requiredFieldsMeta = {};
    var securedIconUrl = AJS.contextPath() + '/plugins/servlet/jira-integration/icons?serverId={0}&iconType={1}&{2}';
    var imagesCache = [];

    var restGet = function (ajax, path, statusCode) {
        return ajax({
            dataType: 'json',
            timeout: 0,
            url: AJS.contextPath() + '/rest/jira-integration/1.0/servers' + (path || ''),
            statusCode,
        });
    };

    function loadServers (ajax) {
        return restGet(ajax, '').done(function (data) {
            data.forEach(function (server) {
                servers[server.id] = server;
            });
        });
    }

    function getProjectIconParams (project) {
        const iconURL = new URL(project.avatarUrls['16x16']);
        // get the existing query params present in the avatar url
        const searchParams = iconURL.searchParams;
        // check if pid and avatarId is present - since these are required fields
        if (!searchParams.get('pid')) searchParams.append('pid', project.id);
        if (!searchParams.get('avatarId')) {
            // if not present as query param - check the url: should be ../avatar/avatarId/
            const paths = iconURL.pathname.split('/');
            const avatarIndex = paths.lastIndexOf('avatar');
            searchParams.append('avatarId', paths[avatarIndex + 1]);
        }
        // finally convert the query param object to string
        const projectIconParams = searchParams.toString();

        return projectIconParams;
    }

    const formatAndCacheIcons = (project, serverId, issuetypes = []) => {
        const issueTypes = issuetypes.filter((item) => {
            // Do not allow to create a sub-task
            return !item.subtask;
        });

        const issueTypeIcons = {};
        issueTypes.forEach((item) => {
            issueTypeIcons[item.id] = item.iconUrl;
        });

        const projectIconParams = getProjectIconParams(project);

        imagesCache[getCacheKey(serverId, project.id)] = {
            iconUrl: AJS.format(securedIconUrl, serverId, 'project', projectIconParams),
            issueTypes: issueTypeIcons,
        };

        return issueTypes;
    };

    function loadProjects (serverId, ajax) {
        
        var projectData = servers[serverId] && servers[serverId].projects;
        if (projectData) {
            return $.Deferred().resolve(projectData);
        }
        return restGet(ajax, '/' + serverId + '/projects')
            .done(function (data) {
                if (data.length) {
                    data.forEach(function (project) {
                        formatAndCacheIcons(project, serverId);
                    });

                    servers[serverId].projects = data;
                }
            });
    }

    const getProject = (serverId, projectId) =>
        getProjects(serverId).filter(function (project) {
            return project.id === projectId;
        })[0];

    const loadIssueTypes = (serverId, projectId, ajax) =>
        restGet(ajax, `/${serverId}/projects/${projectId}/issue-types`, { 404: false })
            .then((data) => {
                const project = getProject(serverId, projectId);

                return data ? formatAndCacheIcons(project, serverId, data) : [];
            });

    function getFieldMetadata (context, ajax) {
        var cacheEntry = getCachedFieldMetadataEntry(context);
        if (cacheEntry.value) {
            return $.Deferred().resolve(cacheEntry.value);
        }
        return restGet(ajax, '/' + context.serverId + '/projects/' + context.projectKey + '/issue-types/' + context.issueTypeId + '/fields-meta')
            .done(function (data) {
                requiredFieldsMeta[cacheEntry.key] = data;
            });
    }

    function getCachedFieldMetadataEntry (context) {
        var key = getCacheKey(context.serverId, context.projectKey, context.issueTypeId);
        return {
            key: key,
            value: requiredFieldsMeta[key],
        };
    }

    function getServerById (serverId) {
        return servers[serverId];
    }

    function isIssueCreationSupported (serverId, ajax) {
        if (!servers[serverId]) {
            throw new Error('Can only be called after server is loaded.');
        }
        if ('issueCreationSupported' in servers[serverId]) {
            return $.Deferred().resolve(servers[serverId].issueCreationSupported);
        }

        return restGet(ajax, '/' + serverId + '/features').then(function (data) {
            if (data.errors) {
                return $.Deferred().reject(data.errors);
            } else if (!Array.isArray(data)) {
                return $.Deferred().reject('Unexpected response from Jira');
            }
            servers[serverId].issueCreationSupported = data.indexOf('CREATE_ISSUE') !== -1;
            return servers[serverId].issueCreationSupported;
        });
    }

    function getProjects (serverId) {
        if (!servers[serverId]) {
            throw new Error('Can only be called after server is loaded.');
        }
        return servers[serverId].projects;
    }

    function getProjectById (serverId, projectId) {
        var projects = getProjects(serverId);
        if (!projects) {
            return null;
        }

        for (var i = 0, l = projects.length; i < l; i++) {
            var project = projects[i];
            if (project.id === projectId) {
                return project;
            }
        }
        return null;
    }

    function getProjectIconUrl (serverId, projectId) {
        var cacheKey = getCacheKey(serverId, projectId);
        return imagesCache[cacheKey] ? imagesCache[cacheKey].iconUrl : '';
    }

    function getIssueTypeIconUrl (serverId, projectId, issueTypeId) {
        var cacheKey = getCacheKey(serverId, projectId);
        var cache = imagesCache[cacheKey];
        return (cache && cache.issueTypes[issueTypeId]) ? cache.issueTypes[issueTypeId] : '';
    }

    function getCacheKey (serverId, projectId, issueTypeId) {
        return serverId + (projectId ? '-' + projectId : '') + (issueTypeId ? '-' + issueTypeId : '');
    }

    return {
        loadServers: loadServers,
        loadProjects: loadProjects,
        loadIssueTypes: loadIssueTypes,
        getFieldMetadata: getFieldMetadata,
        getCachedFieldMetadataEntry: getCachedFieldMetadataEntry,
        getServerById: getServerById,
        getProjectIconUrl: getProjectIconUrl,
        getIssueTypeIconUrl: getIssueTypeIconUrl,
        getProjects: getProjects,
        getProjectById: getProjectById,
        isIssueCreationSupported: isIssueCreationSupported,
    };
});
