define('jira-integration-plugin/jira-issue-dialog-trigger', [
    'jquery',
    'jira-integration-plugin/lodash',
    'jira-integration-plugin/jira-issues-dialog',
], function ($, _, JiraIssuesDialog) {

    function isRightClick (e) {
        return e.which === 3;
    }

    function openInSameTab (e) {
        return (!e.which || e.which === 1) &&
            !(e.metaKey || e.ctrlKey || e.shiftKey);
    }

    var trigger = {
        defaults: {
            selector: '.jira-issues-trigger',
            headerTitleFn: null,
        },
    };

    /**
     * Set up live binding
     * @param triggerOptions
     *  - selector : the selector to live bind trigger functionality to
     *  - headerTitleFn : a function which will be passed the trigger and must return a string. That string will
     * @param dialogOptions Options to pass to the generated JiraIssuesDialog (see that file for details)
     */
    function init (triggerOptions, dialogOptions) {
        triggerOptions = $.extend({}, trigger.defaults, triggerOptions);
        var dialog;

        function getDialog () {
            if (!dialog) {
                if (dialogOptions instanceof JiraIssuesDialog) {
                    dialog = dialogOptions;
                } else {
                    dialog = new JiraIssuesDialog(dialogOptions);
                }
            }
            return dialog;
        }

        function onClick (e, $trigger) {
            var dialog = getDialog();
            // Right click fires a click event in Firefox but not in Chrome
            if (isRightClick(e) || (!openInSameTab(e) && !!$trigger.data('single-issue'))) {
                return;
            }
            e.preventDefault();

            // If trigger has issue keys, use them
            var issueKeysAttr = $trigger.attr('data-issue-keys');
            var issueKeys = issueKeysAttr && issueKeysAttr.split(',');
            var initialIssueKey = $trigger.attr('data-initial-issue-key');
            if (issueKeys) {
                dialog.setIssueKeys(issueKeys, initialIssueKey);
            }

            if (typeof triggerOptions.headerTitleFn === 'function') {
                dialog.setTitle(triggerOptions.headerTitleFn($trigger));
            }

            // If the tooltip is shown hide it
            $trigger.tipsy('hide');

            dialog.show();
        }

        var tipsyOptions = {
            gravity: 'n',
            className: function () {
                // Tipsy sets positions before adding the class. This means the class can't affect the position or else bad things happen.
                // Adding the 'issues-tooltip' class is to add the white-space: nowrap property which will affect the width

                var $tipsy = $('.tipsy');
                var tipsyEl = $tipsy[0];

                var oldWidth = tipsyEl.offsetWidth;
                var tooltipClass = 'multi-issues-tooltip';
                $tipsy.addClass(tooltipClass);
                var newWidth = tipsyEl.offsetWidth;

                // New width should be smaller since keys should wrap
                var widthDiff = oldWidth - newWidth;
                var leftPosPx = parseFloat($tipsy.css('left'), 10);
                $tipsy.css('left', (leftPosPx + widthDiff / 2) + 'px');

                return tooltipClass;
            },
        };

        var clickFn;
        var unbind;

        if (typeof triggerOptions.selector === 'string') {
            clickFn = function (e) {
                // jQuery live events with a descendant filter does not trigger a click event for a middle click
                // So we have to do the filtering ourselves
                var $trigger = $(e.target).closest(triggerOptions.selector);
                if ($trigger.length) {
                    onClick(e, $trigger);
                }
            };
            $(document).on('click', clickFn);

            /* Tipsify tooltips for multi-issue triggers and disabled triggers */
            $(triggerOptions.selector + ".multi-issues").tipsy(
                $.extend({}, tipsyOptions, {
                    live: triggerOptions.selector + ".multi-issues",
                })
            );
            $(triggerOptions.selector + ".disabled").tipsy(
                $.extend({}, tipsyOptions, {
                  live: triggerOptions.selector + ".disabled",
                })
            );

            unbind = function () {
                $(document).off('click', clickFn);
                // turn off tipsy??
            };
        } else {
            clickFn = function (e) {
                onClick(e, $(this));
            };
            var $triggers = $(triggerOptions.selector);
            $triggers.on('click', clickFn);
            var $multiIssueTriggers = $triggers.filter('.multi-issues');
            $multiIssueTriggers.tipsy(tipsyOptions);

            unbind = function () {
                $triggers.off('click', clickFn);
                //turn off tipsy??
            };
        }

        return {
            getDialog: getDialog,
            unbind: unbind,
        };
    }

    trigger.init = init;

    return trigger;
});
