
jQuery(function($) {
    var tasklist = {
        contextPath : $("#confluence-context-path").attr("content"),
        baseUrl : $("input[name='baseUrl']:first").val(),

        registerSortFunction : function() {
            $.fn.applyTaskSort = function() {
                return this.pushStack( $.makeArray( Array.prototype.sort.apply( this, arguments ) ) );
            };
        },

        connectAllLists : function() {
            $("ol.tasklist-container").sortable("option", "connectWith", "ol.tasklist-container");
        },

        avoidBeingFocusedByBodyOnloadFunction : function() {
            if (self["placeFocus"]) {
                var placeFocusFunction = self["placeFocus"];
                self["placeFocus"] = function() {
                    var tasklistForms = $("div.task-list form.quick-add, div.task-list ol.tasklist-container li.task form");

                    tasklistForms.attr("name", "inlinecommentform");
                    placeFocusFunction();
                    tasklistForms.removeAttr("name");
                };
            }
        },

        getQuickAddForm : function(taskListContainer) {
            return $("form.quick-add", taskListContainer.parent());
        },

        getTasklistInput : function(taskListContainer) {
            return $("input.taskname-text", this.getQuickAddForm(taskListContainer));
        },

        setInputElementsEnabled : function(enabled) {
            // Here we disable all the state changing task list controls for every task list instance of a page. Why?
            // Because the reason why they get disabled in the first place is to prevent lost edits, and it happens
            // when two there are more than 1 state-changing task list AJAX request sent (btw, those requests actually
            // change the page content).
            // So we do it for every instance since there could be more than an instance of a task list in one page.
            var elementsToToggle = $(".task-list").find(
                "input.taskname-text" + // Quick-add text field
                ", button.add-button" + // Quick-add button
                ", .task-actions .handle" + // Sort handle
                ", .task-actions button.locker" + // Lock/unlock button
                ", .task-actions button.delete" + // Delete task button
                ", .task-actions button.edit" + // Edit task button
                ", .task-actions button.save" + // Save task button
                ", .task-actions .complete" + // Complete/incomplete checkbox
                ", .editable input" + // Priority radio buttons, assignee text field
                ", button.user-search" + // User search trigger
                ", button.group-search" +  // Group search trigger
                ", .uncheck-all" + // Uncheck all button
                ", .sort-select" + // Sort type drop down
                ", button.sort-order"// Sort direction button
            );

            if (enabled) {
                elementsToToggle.prop("disabled", false);
            } else {
                elementsToToggle.prop("disabled", true);
            }

            $(".tasklist-container").sortable("option", "disabled", !enabled);
        },

        getTaskListFieldSet : function(taskListContainer) {
            return taskListContainer.parent().prev("fieldset");
        },

        getTasklistDetails : function(taskListContainer) {
            var taskListDetails = $.data(taskListContainer, "tasklistDetails");

            if (!taskListDetails) {
                var fieldset = this.getTaskListFieldSet(taskListContainer);
                taskListDetails = {
                    id : taskListContainer.attr("id").substr("tasklist-".length),
                    entityId : fieldset.children("input[id$='_entityId']").val(),
                    readOnly : fieldset.children("input[id$='_readOnly']").val() == "true",
                    enableLocking : fieldset.children("input[id$='_enableLocking']").val() == "true",
                    promptOnDelete : fieldset.children("input[id$='_promptOnDelete']").val() == "true"
                };

                $.data(taskListContainer, "tasklistDetails", taskListDetails);
            }

            taskListDetails.sort = $("option:selected", tasklist.getSortSelect(taskListContainer)).val() || null;
            taskListDetails.sortAscending = taskListDetails.sort && tasklist.getSortDirectionButton(taskListContainer).hasClass("asc");

            return taskListDetails;
        },

        getI18nMessages : function(taskListContainer) {
            var messages = {};
            $("input.plugin_tasklist_i18n", this.getTaskListFieldSet(taskListContainer)).each(function() {
                messages[this.name] = this.value;
            });

            return messages;
        },

        getActionUrl : function() {
            return this.contextPath + "/plugins/servlet/tasklist";
        },

        getTasks : function(tasklist) {
            var tasks = [];
            $("li.task", tasklist).each(function() {
                tasks.push($(this));
            });

            return tasks;
        },

        getRenderedTaskNameParagraph : function(task) {
            return $("p.rendered.taskname", task);
        },

        getTaskDetails : function(task) {
            var taskDetails = $.data(task, "taskDetails");

            if (!taskDetails) {
                taskDetails = {
                    createdDate : $("span.task-createdDate", task).text()
                };

                $.data(task, "taskDetails", taskDetails);
            }

            // ID is dynamic, because it is based on its name and it can be changed by the user.
            taskDetails.id = task.attr("id").substr(0, task.attr("id").length - "_hook".length);
            taskDetails.name = $("span.task-name", task).text();
            taskDetails.renderedName = tasklist.getRenderedTaskNameParagraph(task).html();
            taskDetails.priority = (function() {
                var priority = null;

                $("input.high-priority, input.medium-priority, input.low-priority", task).each(function() {
                    if (this.checked)
                        priority = this.value;
                });

                return priority;
            })();
            taskDetails.completed = $("input.complete", task).is(":checked");
            taskDetails.completedDate = $("span.task-completedDate", task).text();
            taskDetails.renderedCompletedDate = $("dd.task-completed-date", task).text();
            taskDetails.locked = task.hasClass("locked");
            taskDetails.assignee = $("span.task-assginee", task).text();

            return taskDetails;
        },

        getPercentageComplete : function(taskListContainer) {
            var tasks = this.getTasks(taskListContainer);
            if (tasks && tasks.length) {
                var completeTaskCount = 0;

                for (var idx = 0; idx < tasks.length; idx++) {
                    if (tasklist.getTaskDetails(tasks[idx]).completed) {
                        ++completeTaskCount;
                    }
                }

                return (completeTaskCount / tasks.length) * 100;
            }

            return 0;
        },

        getTaskListOfTask : function(task) {
            return task.parent();
        },

        getIndexOfTaskInList : function(task) {
            var taskElement = task.get(0);
            var taskIndex = -1;

            $("li.task", this.getTaskListOfTask(task)).each(function(index) {
                if (this == taskElement) {
                    taskIndex = index;
                    return false;
                }
            });

            return taskIndex;
        },

        isSameTaskList : function(taskListContainerLeft, taskListContainerRight) {
            var leftDetails = this.getTasklistDetails(taskListContainerLeft);
            var rightDetails = this.getTasklistDetails(taskListContainerRight);

            return leftDetails.id == rightDetails.id && leftDetails.entityId == rightDetails.entityId;
        },

        makeTaskListSortable : function(taskListContainer) {
            if (this.getTasklistDetails(taskListContainer).readOnly)
                return;

            taskListContainer.sortable({
                opacity: 0.5,
                update : function(event, ui) {
                    var task = ui.item;
                    var sourceTaskList = ui.sender ? ui.sender : tasklist.getTaskListOfTask(task);
                    var sourceTaskListDetails = tasklist.getTasklistDetails(sourceTaskList);
                    var destinationTaskList = tasklist.getTaskListOfTask(task);
                    var destinationTaskListDetails = tasklist.getTasklistDetails(destinationTaskList);
                    var taskIndex = tasklist.getIndexOfTaskInList(task);
                    var isSameTaskList = tasklist.isSameTaskList(sourceTaskList, destinationTaskList);
                    var isCopyOperation =  (event.ctrlKey || event.metaKey) && !isSameTaskList;

                    tasklist.setInputElementsEnabled(false);

                    AJS.safe.ajax({
                        cache : false,
                        data : {
                            action : isCopyOperation ? "copyTasks" : "reorderTasks",
                            listId : sourceTaskListDetails.id,
                            taskId : tasklist.getTaskDetails(task).id,
                            toListId : destinationTaskListDetails.id,
                            index : taskIndex,
                            entityId : sourceTaskListDetails.entityId,
                            destinationEntityId : destinationTaskListDetails.entityId
                        },
                        dataType : "xml",
                        error : function() {
                            tasklist.setInputElementsEnabled(true);
                        },
                        success : function(xml) {
                            if (isCopyOperation) {
                                var responseXml = $(xml);
                                var insertBeforeTaskId = responseXml.find("idOfTaskAfterSourceTask").text();
                                var sourceTaskRepresentation = responseXml.find("representation").text();

                                task.attr("id", $(xml).find("newTaskId").text() + "_hook");

                                if (insertBeforeTaskId && $.trim(insertBeforeTaskId).length > 0) {
                                    var insertBeforeTaskIdHook = insertBeforeTaskId + "_hook";

                                    $("li.task", sourceTaskList).each(function() {
                                        if (this.id === insertBeforeTaskIdHook) {
                                            var aTask = $(this);
                                            aTask.before(sourceTaskRepresentation);
                                            tasklist.initTask(aTask.prev("li.task"));
                                        }
                                    });
                                } else {
                                    sourceTaskList.append(sourceTaskRepresentation);
                                    tasklist.initTask($("li.task:last", sourceTaskList));
                                }

                                sourceTaskList.sortable("refresh");

                            } else {
                                task.attr("id", $(xml).find("id").text() + "_hook");
                            }

                            tasklist.updateProgressBar(sourceTaskList);
                            tasklist.updateMarkAllIncompleteButton(sourceTaskList);

                            if (!isSameTaskList) {
                                tasklist.updateProgressBar(destinationTaskList);
                                tasklist.updateMarkAllIncompleteButton(destinationTaskList);
                            }

                            $("option:first", tasklist.getSortSelect(sourceTaskList)).prop("selected", true).trigger("change");
                            $("option:first", tasklist.getSortSelect(destinationTaskList)).prop("selected", true).trigger("change");

                            tasklist.setInputElementsEnabled(false);
                        },
                        type : "POST",
                        url : tasklist.getActionUrl()
                    });
                }
            });
        },

        updateProgressBar : function(taskListContainer) {
            $("div.progress div", taskListContainer.parent()).css({
                "width" : tasklist.getPercentageComplete(taskListContainer) + "%"
            });
        },

        addTask : function(taskListContainer, taskName, onSuccessCallback, onErrorCallback) {
            if (taskName && $.trim(taskName).length > 0) {
                var taskListDetails = tasklist.getTasklistDetails(taskListContainer);

                tasklist.setInputElementsEnabled(false);
                AJS.safe.ajax({
                    cache : false,
                    data : {
                        action : "addTask",
                        entityId : taskListDetails.entityId,
                        listId : taskListDetails.id,
                        newTask : taskName
                    },
                    dataType : "xml",
                    error: function(jqXHR, textStatus, errorThrown) {
                        if ($.isFunction(onErrorCallback))
                            onErrorCallback(jqXHR, textStatus, errorThrown);

                        tasklist.setInputElementsEnabled(true);
                    },
                    success : function(xml) {
                        var newTask = $(document.createElement("div")).html($(xml).find("representation").text()).find("li.task");

                        newTask.hide().appendTo(taskListContainer).fadeIn("normal", function() {
                            taskListContainer.sortable("refresh");

                            tasklist.updateProgressBar(taskListContainer);
                            tasklist.getTasklistInput(taskListContainer).val("");
                            tasklist.initTask(newTask);

                            if ($.isFunction(onSuccessCallback))
                                onSuccessCallback();
                        });

                        tasklist.setInputElementsEnabled(true);
                    },
                    type : "POST",
                    url : tasklist.getActionUrl()
                });
            }
        },

        initQuickAddForm : function(taskListContainer) {
            this.getQuickAddForm(taskListContainer).submit(function() {
                var quickAddForm = $(this), isProcessing = quickAddForm.data("isProcessing"), setFormIsNotBeingProcessed = function() {
                    quickAddForm.removeData("isProcessing");
                };

                if (!isProcessing) {
                    quickAddForm.data("isProcessing", true);
                    tasklist.addTask(taskListContainer, tasklist.getTasklistInput(taskListContainer).val(), setFormIsNotBeingProcessed, setFormIsNotBeingProcessed);
                }
                return false;
            });
        },

        updateSortDirectionButton : function(taskListContainer) {
            var taskListDetails = this.getTasklistDetails(taskListContainer);
            var sortBy = taskListDetails.sort;
            var sortDirectionButton = this.getSortDirectionButton(taskListContainer);

            if (!sortBy) {
                sortDirectionButton.removeClass("asc").removeClass("desc");
            } else {
                if (taskListDetails.sortAscending) {
                    sortDirectionButton.addClass("asc").removeClass("desc");
                } else {
                    sortDirectionButton.removeClass("asc").addClass("desc");
                }
            }
        },

        compareString : function(leftString, rightString, sign) {
            return sign * (leftString == rightString ? 0 : (leftString > rightString ? 1 : -1));
        },

        getSortSign : function(taskElement) {
            return this.getTasklistDetails(this.getTaskListOfTask($(taskElement))).sortAscending ? 1 : -1;
        },

        compareTaskName : function(left, right) {
            var leftRenderedTaskNameText = $("p.rendered.taskname", $(left)).text();
            var rightRenderedTaskNameText = $("p.rendered.taskname", $(right)).text();

            return tasklist.compareString(leftRenderedTaskNameText, rightRenderedTaskNameText, tasklist.getSortSign(left));
        },

        compareTaskCreatedDate : function(left, right) {
            var createdDateLeft = tasklist.getTaskDetails($(left)).createdDate;
            var createdDateRight = tasklist.getTaskDetails($(right)).createdDate;

            var result = parseFloat(createdDateLeft) - parseFloat(createdDateRight);
            return 0 == result ? tasklist.compareTaskName(left, right) : tasklist.getSortSign(left) * result;
        },

        compareTaskCompletedDate : function(left, right) {
            var completedDateLeft = tasklist.getTaskDetails($(left)).completedDate;
            var completedDateRight = tasklist.getTaskDetails($(right)).completedDate;

            return tasklist.getSortSign(left) * (completedDateLeft - completedDateRight);
        },

        compareTaskAssignee : function(left, right) {
            var assigneeLeft = tasklist.getTaskDetails($(left)).assignee;
            var assigneeRight = tasklist.getTaskDetails($(right)).assignee;
            var result = tasklist.compareString(assigneeLeft, assigneeRight, tasklist.getSortSign(left));
            return 0 == result ? tasklist.compareTaskName(left, right) : result;
        },

        compareTaskPriority : function(left , right) {
            var getTaskPriorityAsNumber = function(task) {
                var taskDetails = tasklist.getTaskDetails(task);
                return { HIGH: 3, MEDIUM: 2, LOW: 1 }[taskDetails.priority] || -1;
            };

            var result = -1 * (getTaskPriorityAsNumber($(left)) - getTaskPriorityAsNumber($(right)));
            return 0 == result ? tasklist.compareTaskName(left, right) : tasklist.getSortSign(left) * result;
        },

        sortList : function(taskListContainer) {
            var taskListDetails = this.getTasklistDetails(taskListContainer);
            var comparators = {
                priority : tasklist.compareTaskPriority,
                date : tasklist.compareTaskCreatedDate,
                completed : tasklist.compareTaskCompletedDate,
                name : tasklist.compareTaskName,
                assignee: tasklist.compareTaskAssignee
            };
            var selectedComparator = comparators[taskListDetails.sort];
            if (selectedComparator) {
                $("li.task", taskListContainer).applyTaskSort(selectedComparator).appendTo(taskListContainer);
            }
        },

        updateTaskDisplay : function(task) {
            var taskDetails = this.getTaskDetails(task);

            $("input.assignee", task).val(taskDetails.assignee);

            if (taskDetails.completed) {
                $("dd.task-completed-date", task).text(taskDetails.renderedCompletedDate);
                $("dt.task-completed-label, dd.task-completed-date", task).show();
                task.addClass("completed");
            } else {
                $("dt.task-completed-label, dd.task-completed-date", task).hide();
                task.removeClass("completed");
            }
        },

        getMarkAllIncompleteButton : function(taskListContainer) {
            return $("button.uncheck-all", taskListContainer.parent());
        },

        updateMarkAllIncompleteButton : function(taskListContainer) {
            var atLeastOneComplete = false;
            var hasLockedTask = false;
            var tasks = this.getTasks(taskListContainer);

            for (var idx = 0; idx < tasks.length; idx++) {
                var taskDetails = tasklist.getTaskDetails(tasks[idx]);

                if (taskDetails.completed)
                    atLeastOneComplete = true;
                if (!hasLockedTask && taskDetails.locked)
                    hasLockedTask = true;
            }

            if (atLeastOneComplete && (!this.getTasklistDetails(taskListContainer).enableLocking || !hasLockedTask))
                this.getMarkAllIncompleteButton(taskListContainer).show();
            else
                this.getMarkAllIncompleteButton(taskListContainer).hide();
        },

        initTaskInfoToggleButton : function(task) {
            $("button.trigger", task).click(function() {
                var buttonTrigger = $(this);
                if(buttonTrigger.hasClass("icon-section-closed"))
                {
                    buttonTrigger.removeClass("icon-section-closed");
                    buttonTrigger.addClass("icon-section-opened");
                }
                else if(buttonTrigger.hasClass("icon-section-opened"))
                {
                    buttonTrigger.removeClass("icon-section-opened");
                    buttonTrigger.addClass("icon-section-closed");
                }

                if (task.hasClass("closed")) {
                    task.removeClass("closed");
                    task.addClass("opened");
                } else if (task.hasClass("opened")) {
                    task.removeClass("opened");
                    task.addClass("closed");
                }

                var assigneeInput = $("input[name='task-assignee']", task);
                assigneeInput.data("lastSavedAssignee", assigneeInput.val());
            });
        },

        initTaskCompleteCheckbox : function(task) {
            $("input.complete", task).click(function() {
                var taskListContainer = tasklist.getTaskListOfTask(task);
                var taskListDetails = tasklist.getTasklistDetails(taskListContainer);
                var taskDetails = tasklist.getTaskDetails(task);
                var checkBox = this;
                var taskCompleted = checkBox.checked;

                tasklist.setInputElementsEnabled(false);

                AJS.safe.ajax({
                    cache : false,
                    data : {
                        action : "toggleTaskStatus",
                        entityId : taskListDetails.entityId,
                        listId : taskListDetails.id,
                        taskId : taskDetails.id
                    },
                    dataType : "xml",
                    error : function() {
                        tasklist.setInputElementsEnabled(true);
                    },
                    success : function(xml) {
                        var xmlDoc = $(xml);

                        $("span.task-completedDate", task).text(xmlDoc.find("completedDate").text());
                        $("span.task-assginee", task).text(xmlDoc.find("assignee").text());
                        $("dd.task-completed-date", task).text(xmlDoc.find("renderedCompletedDate").text());

                        checkBox.checked = taskCompleted; // Sometimes the checkbox reverts its checked state. So we help it here.
                        tasklist.updateTaskDisplay(task);
                        tasklist.updateProgressBar(taskListContainer);
                        tasklist.updateMarkAllIncompleteButton(taskListContainer);

                        if (taskListDetails.sort == "completed")
                            tasklist.sortList(taskListContainer);

                        tasklist.setInputElementsEnabled(true);
                    },
                    type : "POST",
                    url : tasklist.getActionUrl()
                });
            });
        },

        initTaskPriorityCheckboxes : function(task) {
            $("input.high-priority, input.medium-priority, input.low-priority", task).click(function() {
                var taskListContainer = tasklist.getTaskListOfTask(task);
                var taskListDetails = tasklist.getTasklistDetails(taskListContainer);
                var taskDetails = tasklist.getTaskDetails(task);
                var priorityCheckbox = this;

                if (priorityCheckbox.checked) {
                    tasklist.setInputElementsEnabled(false);

                    AJS.safe.ajax({
                        cache : false,
                        data : {
                            action : "changeTaskPriority",
                            entityId : taskListDetails.entityId,
                            listId : taskListDetails.id,
                            taskId : taskDetails.id,
                            priority : this.value
                        },
                        dataType : "xml",
                        error : function() {
                            tasklist.setInputElementsEnabled(true);
                        },
                        success : function() {
                            task.removeClass("low");
                            task.removeClass("medium");
                            task.removeClass("high");

                            task.addClass(priorityCheckbox.value.toLowerCase());

                            if (taskListDetails.sort == "priority")
                                tasklist.sortList(taskListContainer);

                            tasklist.setInputElementsEnabled(true);
                        },
                        type : "POST",
                        url : tasklist.getActionUrl()
                    });
                }
            });
        },

        initTaskReassignInput : function(task) {
            $("input[name='task-assignee']", task).keydown(function(jsEvent) {
                var assigneeInput = $(this),
                    editableContainer = assigneeInput.closest(".editable");

                if (13 === jsEvent.keyCode) {
                    var taskListContainer = tasklist.getTaskListOfTask(task),
                        taskListDetails = tasklist.getTasklistDetails(taskListContainer),
                        taskDetails = tasklist.getTaskDetails(task),
                        assignee = $.trim(assigneeInput.val()),
                        inlineErrorDisplay = $(".error", task);

                    tasklist.setInputElementsEnabled(false);

                    AJS.safe.ajax({
                        cache : false,
                        data : {
                            action : "reassignTask",
                            entityId : taskListDetails.entityId,
                            listId : taskListDetails.id,
                            taskId : taskDetails.id,
                            assignee : assignee
                        },
                        dataType : "xml",
                        statusCode: {
                            400 : function(jqXhr) {
                                inlineErrorDisplay.text($(jqXhr.responseText).text()).removeClass("hidden");
                                tasklist.setInputElementsEnabled(true);
                            }
                        },
                        success : function() {
                            inlineErrorDisplay.addClass("hidden");
                            $("span.task-assginee", task).text(assignee);

                            assigneeInput.data("lastSavedAssignee", assignee);
                            editableContainer.removeClass("prompt-save");

                            if (taskListDetails.sort == "assignee")
                                tasklist.sortList(taskListContainer);

                            tasklist.setInputElementsEnabled(true);
                        },
                        type : "POST",
                        url : tasklist.getActionUrl()
                    });

                    return false;
                } else {
                    setTimeout(function() {
                        var lastSavedAssignee = assigneeInput.data("lastSavedAssignee"),
                            currentAssignee = assigneeInput.val();

                        if (currentAssignee !== lastSavedAssignee)
                            editableContainer.addClass("prompt-save");
                        else
                            editableContainer.removeClass("prompt-save");

                    }, 300); // 300ms delay to detect value change.
                }
            });
        },

        getTaskEditor : function(task) {
            return $("p.editable.taskname", task);
        },

        getTaskEditorInput : function(task) {
            return this.getTaskEditor(task).children("input");
        },

        closeTaskEditor : function(task) {
            var taskEditor = this.getTaskEditor(task);

            taskEditor.hide();
            taskEditor.data("open", false);

            this.getRenderedTaskNameParagraph(task).show();

            task.removeClass("editing");
        },

        getTaskSaveButton : function(task) {
            return $("button.save", task);
        },

        initTaskSaveButton : function(task) {
            this.getTaskSaveButton(task).click(function() {
                var taskListContainer = tasklist.getTaskListOfTask(task);
                var taskListDetails = tasklist.getTasklistDetails(taskListContainer);
                var taskDetails = tasklist.getTaskDetails(task);
                var newTaskName = $.trim(tasklist.getTaskEditorInput(task).val());

                if (newTaskName.length && tasklist.getTaskDetails(task).name != newTaskName) {
                    tasklist.setInputElementsEnabled(false);

                    AJS.safe.ajax({
                        cache : false,
                        data : {
                            action : "editTask",
                            entityId : taskListDetails.entityId,
                            listId : taskListDetails.id,
                            taskId : taskDetails.id,
                            newTask : newTaskName
                        },
                        dataType : "xml",
                        error : function() {
                            tasklist.setInputElementsEnabled(true);
                        },
                        success : function(xml) {
                            var xmlDoc = $(xml);

                            $("span.task-name", task).text(newTaskName);
                            task.attr("id", xmlDoc.find("id").text() + "_hook");

                            tasklist.getRenderedTaskNameParagraph(task).html(xmlDoc.find("renderedName").text());
                            tasklist.closeTaskEditor(task);

                            if (taskListDetails.sort == "name")
                                tasklist.sortList(taskListContainer);

                            tasklist.setInputElementsEnabled(true);
                        },
                        type : "POST",
                        url : tasklist.getActionUrl()
                    });
                } else {
                    tasklist.closeTaskEditor(task);
                }

                return false;
            });
        },

        openTaskEditor : function(task) {
            this.getRenderedTaskNameParagraph(task).hide();
            this.getTaskEditor(task).show();
            this.getTaskEditorInput(task).focus();

            task.addClass("editing");
        },

        initTaskEditButton : function(task) {
            $("button.edit", task).click(function() {
                var taskEditor = tasklist.getTaskEditor(task);

                if (!taskEditor.data("open")) {
                    tasklist.openTaskEditor(task);
                    taskEditor.data("open", true);
                }

                return false;
            });
        },

        initTaskEditorInput : function(task) {
            this.getTaskEditorInput(task).keydown(function(jsEvent) {

                if (13 === jsEvent.keyCode) {
                    tasklist.getTaskSaveButton(task).trigger("click");
                    tasklist.getTaskEditorInput(task).focus(); // Place the focus back on the editor.
                    return false;
                } else if (27 == jsEvent.keyCode) {
                    tasklist.closeTaskEditor(task);
                    return false;
                }
            });
        },

        initTaskDeleteButton : function(task) {
            $("button.delete", task).click(function() {
                var taskListContainer = tasklist.getTaskListOfTask(task);
                var i18nMessages = tasklist.getI18nMessages(taskListContainer);
                var taskListDetails = tasklist.getTasklistDetails(taskListContainer);
                var taskDetails = tasklist.getTaskDetails(task);

                if (!taskListDetails.promptOnDelete || confirm(AJS.format(i18nMessages["i18n-taskdeletemessage"], taskDetails.name))) {
                    tasklist.setInputElementsEnabled(false);
                    AJS.safe.ajax({
                        cache : false,
                        data : {
                            action : "removeTask",
                            entityId : taskListDetails.entityId,
                            listId : taskListDetails.id,
                            taskId : taskDetails.id
                        },
                        dataType : "xml",
                        error : function() {
                            tasklist.setInputElementsEnabled(true);
                        },
                        success : function() {
                            task.fadeOut("slow", function() {
                                task.remove();
                                tasklist.updateProgressBar(taskListContainer);
                                tasklist.updateMarkAllIncompleteButton(taskListContainer);
                            });

                            tasklist.setInputElementsEnabled(true);
                        },
                        type : "POST",
                        url : tasklist.getActionUrl()
                    });
                }

                return false;
            });
        },

        initTaskLockButton : function(task) {
            $("button.locker", task).click(function() {
                var taskListContainer = tasklist.getTaskListOfTask(task);
                var taskListDetails = tasklist.getTasklistDetails(taskListContainer);
                var taskDetails = tasklist.getTaskDetails(task);

                tasklist.setInputElementsEnabled(false);

                AJS.safe.ajax({
                    cache : false,
                    data : {
                        action : "toggleTaskLock",
                        entityId : taskListDetails.entityId,
                        listId : taskListDetails.id,
                        taskId : taskDetails.id
                    },
                    dataType : "xml",
                    error : function() {
                        tasklist.setInputElementsEnabled(true);
                    },
                    success : function() {
                        if (taskDetails.locked) {
                            task.removeClass("locked");
                            task.addClass("unlocked");
                        } else {
                            task.addClass("locked");
                            task.removeClass("unlocked");
                        }

                        tasklist.updateMarkAllIncompleteButton(taskListContainer);
                        tasklist.setInputElementsEnabled(true);
                    },
                    type : "POST",
                    url : tasklist.getActionUrl()
                });

                return false;
            });
        },

        initTaskActions : function(task) {
            $("div.task-actions", task).hover(
                    function() {
                        if (!tasklist.getTaskDetails(task).locked) {
                            $(this).addClass("hover");
                        }
                    },
                    function() {
                        $(this).removeClass("hover");
                    }
            );
        },

        showUserPicker : function(assignFunctionName, task) {
            var assigneeInput = $("input.assignee", task);
            var currentAssignees = assigneeInput.val();
            var searchAction = "openuserpicker.action";
            var url = this.baseUrl + "/spaces/" + searchAction + "?startIndex=0&existingUsers=" + encodeURIComponent(currentAssignees) + "&onPopupSubmit=" + assignFunctionName;
            var windowName = "UserPicker";
            var windowProperties = "status=yes,resizable=yes,top=100,left=200,width=580,height=550,scrollbars=yes";
            var windowOpened = window.open(url, windowName, windowProperties);
            windowOpened.focus();
            windowOpened.opener = null;
        },

        showGroupPicker : function(assignFunctionName, task) {
            var assigneeInput = $("input.assignee", task);
            var currentAssignees = assigneeInput.val();
            var searchAction = "opengrouppicker.action";
            var actionName = "dosearchgroups.action";
            var url = this.baseUrl + "/spaces/" + searchAction + "?startIndex=0&actionName=" + actionName + "&existingGroups=" + encodeURIComponent(currentAssignees) + "&onPopupSubmit=" + assignFunctionName;
            var windowName = "GroupPicker";
            var windowProperties = "status=yes,resizable=yes,top=100,left=200,width=580,height=550,scrollbars=yes";
            var windowOpened = window.open(url, windowName, windowProperties);
            windowOpened.focus();
            windowOpened.opener = null;
        },

        initUserAndGroupPickers : function(task) {
            var taskId = this.getTaskDetails(task).id;
            var assignFunctionName = "set" + taskId.replace(":", "_") + "Assignee";

            self[assignFunctionName] = function(entities) {
                var assigneeInput = $("input.assignee", task);
                var currentAssignees = assigneeInput.val();

                assigneeInput.val($.trim(currentAssignees).length == 0 ? entities : currentAssignees + ", " + entities).focus().trigger("keydown");
            };

            $("button.user-search", task).click(function() {
                tasklist.showUserPicker(assignFunctionName, task);
            });

            $("button.group-search", task).click(function() {
                tasklist.showGroupPicker(assignFunctionName, task);
            });
        },

        initMarkAllIncompleteButton : function(taskListContainer) {
            this.getMarkAllIncompleteButton(taskListContainer).click(function() {
                var taskListDetails = tasklist.getTasklistDetails(taskListContainer);

                tasklist.setInputElementsEnabled(false);

                AJS.safe.ajax({
                    cache : false,
                    data : {
                        action : "markAllIncomplete",
                        entityId : taskListDetails.entityId,
                        listId : taskListDetails.id
                    },
                    dataType : "xml",
                    error : function() {
                        tasklist.setInputElementsEnabled(true);
                    },
                    success : function() {
                        $("input.complete", taskListContainer).each(function() {
                            this.checked = false;
                        });

                        tasklist.updateMarkAllIncompleteButton(taskListContainer);
                        tasklist.updateProgressBar(taskListContainer);

                        $.each(tasklist.getTasks(taskListContainer), function() {
                            tasklist.updateTaskDisplay(this);
                        });

                        tasklist.setInputElementsEnabled(true);
                    },
                    type : "POST",
                    url : tasklist.getActionUrl()
                });
            });
        },

        getSortDirectionButton : function(taskListContainer) {
            return $("button.sort-order", taskListContainer.parent());
        },

        getSortSelect : function(taskListContainer) {
            return $("select.sort-select", taskListContainer.parent());
        },

        initSortSelect : function(taskListContainer) {
            this.getSortSelect(taskListContainer).change(function() {
                var selectedSort = $("option:selected", $(this)).val();
                var sortName = "sortBy" + ((selectedSort === "custom" || selectedSort === "") ? "None" : selectedSort.substring(0, 1).toUpperCase() + selectedSort.substring(1, selectedSort.length));
                var taskListDetails = tasklist.getTasklistDetails(taskListContainer);

                tasklist.setInputElementsEnabled(false);

                AJS.safe.ajax({
                    cache : false,
                    data : {
                        action : sortName,
                        entityId : taskListDetails.entityId,
                        listId : taskListDetails.id
                    },
                    dataType : "xml",
                    error : function() {
                        tasklist.setInputElementsEnabled(true);
                    },
                    success : function() {
                        tasklist.updateSortDirectionButton(taskListContainer);
                        tasklist.sortList(taskListContainer);
                        tasklist.setInputElementsEnabled(true);
                    },
                    type : "POST",
                    url : tasklist.getActionUrl()
                });
            });
        },

        initSortDirectionButton : function(taskListContainer) {
            this.getSortDirectionButton(taskListContainer).click(function() {
                var sortDirectionButton = $(this);
                if (sortDirectionButton.hasClass("asc"))
                    sortDirectionButton.removeClass("asc").addClass("desc");
                else
                    sortDirectionButton.removeClass("desc").addClass("asc");

                tasklist.sortList(taskListContainer);
            });
        },

        initTask : function(task) {
            this.initTaskInfoToggleButton(task);
            this.initTaskCompleteCheckbox(task);
            this.initTaskPriorityCheckboxes(task);
            this.initTaskReassignInput(task);
            this.initTaskSaveButton(task);
            this.initTaskEditButton(task);
            this.initTaskEditorInput(task);
            this.initTaskDeleteButton(task);
            this.initTaskLockButton(task);
            this.initTaskActions(task);
            this.initUserAndGroupPickers(task);
            this.updateTaskDisplay(task);
        },

        initTasks : function(taskListContainer) {
            $.each(this.getTasks(taskListContainer), function() {
                tasklist.initTask(this);
            });
        },

        initTaskList : function(taskListContainer) {
            this.makeTaskListSortable(taskListContainer);
            this.updateProgressBar(taskListContainer);
            this.initQuickAddForm(taskListContainer);
            this.initMarkAllIncompleteButton(taskListContainer);
            this.initTasks(taskListContainer);
            this.initSortSelect(taskListContainer);
            this.initSortDirectionButton(taskListContainer);
        }
    };


    tasklist.registerSortFunction();

    $("ol.tasklist-container").each(function() {
        tasklist.initTaskList($(this));
        tasklist.sortList($(this));
    });

    tasklist.connectAllLists();
    tasklist.avoidBeingFocusedByBodyOnloadFunction();
});
