Confluence.Blueprints.JiraReport = (function ($) {
    var changeLogForm;
    var callbackObjects;
    var STATIC_TEMPLATE_KEY = "jirareports-changelog-static";
    var DYNAMIC_TEMPLATE_KEY = "jirareports-changelog-dynamic";
    var DEFAULT_OPTION_VALUE = '-1';
    var NOT_SUPPORTED_BUILD_NUMBER = -1;
    var START_JIRA_UNSUPPORTED_BUILD_NUMBER = 6109; //jira version 6.0.8
    var END_JIRA_UNSUPPORTED_BUILD_NUMBER = 6155; //jira version 6.1.1
    var JIRA_UNSUPPORTED_MESSAGE = AJS.I18n.getText('jirareports.statusreport.unsupported');
    var $createButton;

    var getSelectedServer = function (serverId) {
        var servers = Confluence.Blueprints.JiraReport.servers;
        for (var i = 0; i < servers.length; i++) {
            if (servers[i].id === serverId) {
                return servers[i];
            }
        }
        return null;
    };

    var executeCallbackFunction = function (methodName, params) {
        var callbackObject;
        for (var i = 0; i < callbackObjects.length; i++) {
            callbackObject = callbackObjects[i];
            if (typeof callbackObject[methodName] === 'function') {
                if (params) {
                    callbackObject[methodName].apply(null, Array.prototype.slice.call(params));
                } else {
                    callbackObject[methodName]();
                }
            }
        }
    };

    var onAuthenticatedSuccess = function (server) {
        server.authUrl = null;
        changeLogForm.oauthForm.addClass('hidden');
        executeCallbackFunction('onAuthenticatedSuccess');
    };

    var getJQLFromStaticForm = function (project, version) {
        var jqlArray = [];
        if ((project.val() !== DEFAULT_OPTION_VALUE) && project.is(':enabled')) {
            jqlArray.push("project = \'", escapeSingleQuote(project.val()), '\'');

            if (version.is(':enabled')) {
                var selectedVersions = version.find('option[value!="' + DEFAULT_OPTION_VALUE + '"]:selected');
                var versionsLeng = selectedVersions.length;

                if (versionsLeng > 0) {
                    var versions = '';
                    for (var i = 0; i < selectedVersions.length; i++) {
                        versions = versions + getRightVersion(selectedVersions[i]) + ", ";
                    }
                    jqlArray.push(' AND fixVersion in (', versions.substring(0, versions.length - 2), ')');
                }
            }

        }

        return jqlArray.join('');
    };

    var getRightVersion = function (selectedVersion) {
        var versionValue = '';
        var versionText = escapeSingleQuote(selectedVersion.text);
        if (versionText === AJS.I18n.getText('jirareports.changelog.blueprint.form.version.noversion') && selectedVersion.value === 'EMPTY') {
            versionValue = selectedVersion.value;
        } else {
            versionValue = "'" + versionText + "'";
        }
        return versionValue;
    };

    var escapeSingleQuote = function (value) {
        return value.replace(/'/g, '\\\'');
    };

    var isJiraUnSupportedVersion = function (server) {
        var buildNumber = server.buildNumber;
        return buildNumber == NOT_SUPPORTED_BUILD_NUMBER ||
            (buildNumber >= START_JIRA_UNSUPPORTED_BUILD_NUMBER && buildNumber < END_JIRA_UNSUPPORTED_BUILD_NUMBER);
    };

    var checkJiraUnSupportedVersion = function (server) {
        if (changeLogForm.selectedReportType !== Confluence.Blueprints.JiraReport.REPORT_TYPE_STATUS)
            return false;

        var isUnsupported = isJiraUnSupportedVersion(server);
        if (isUnsupported) {
            changeLogForm.errorForm.html(Confluence.Blueprints.JiraReports.Dialog.warning({infor: JIRA_UNSUPPORTED_MESSAGE}));
            executeCallbackFunction("disableForm");
            disableCreateButton();
        } else {
            executeCallbackFunction("enableForm");
            enableCreateButton();
        }
        return isUnsupported;
    };

    var disableCreateButton = function () {
        $createButton && $createButton.prop("disabled", true);
    };

    var enableCreateButton = function () {
        $createButton && $createButton.prop("disabled", false);
    };

    return {
        REPORT_TYPE_CHANGE_LOG: 1,
        REPORT_TYPE_STATUS: 2,

        prepareData: function () {
            AJS.$.ajax({
                dataType: 'json',
                url: Confluence.getContextPath() + '/rest/jiraanywhere/1.0/servers',
                async: false,
                success: function(servers) {
                    Confluence.Blueprints.JiraReport.servers = servers;
                    AJS.$.each(servers, function (index, server) {
                        if (server.selected) {
                            Confluence.Blueprints.JiraReport.primaryServer = server;
                        }
                    });
                }
            });
        },

        init: function (state, objects) {
            callbackObjects = objects;
            changeLogForm = {
                server: state.$container.find('#jira-reports-servers'),
                project: state.$container.find('#jira-reports-project'),
                version: state.$container.find('#jira-reports-fix-version'),
                title: state.$container.find('#jira-reports-page-title'),
                spaceKey: state.wizardData.spaceKey,
                textSearch: state.$container.find('#jira-query'),
                oauthForm: state.$container.find('#oauth-form'),
                errorForm: state.$container.find('#error-form'),
                switchSearchType: state.$container.find("#switch-search-type")
            };

            $createButton = AJS.$('#create-dialog .create-dialog-create-button:visible');

            // bind event click
            changeLogForm.switchSearchType.click(function (e) {
                var toggle = changeLogForm.switchSearchType.hasClass("simple");
                if (toggle) {
                    changeLogForm.textSearch.val(getJQLFromStaticForm(changeLogForm.project, changeLogForm.version));
                    //placeholded class just have in IE8,9 because IE8,9 not support placeholder
                    //AUI apply placeholder library to make it. So need to remove placeholded class when have value.
                    changeLogForm.textSearch.removeClass('placeholded');
                    changeLogForm.switchSearchType.removeClass("simple")
                        .addClass("advanced")
                        .text(AJS.I18n.getText('jirareports.changelog.blueprint.form.switch.search.type.simple'));
                } else {
                    changeLogForm.switchSearchType.removeClass("advanced")
                        .addClass("simple")
                        .text(AJS.I18n.getText('jirareports.changelog.blueprint.form.switch.search.type.advance'));
                }
                state.$container.find("[id='advanced-jira-search']").toggleClass('hidden', !toggle);
                state.$container.find("#simple-jira-search").toggleClass('hidden', toggle);
                state.$container.find("#inforWarning").empty();
            });

            executeCallbackFunction('init', [changeLogForm, state.$container]);

            changeLogForm.server.change(function () {
                var selectedServer = getSelectedServer(changeLogForm.server.val());
                if (checkJiraUnSupportedVersion(selectedServer)) return;

                Confluence.Blueprints.JiraReport.checkOauthForm(selectedServer);
                executeCallbackFunction('onServerChange');
            });
        },

        loadServers: function () {
            var servers = Confluence.Blueprints.JiraReport.servers;
            if (servers.length === 1) {
                changeLogForm.server.parent().addClass('hidden');
            }

            changeLogForm.server.append(Confluence.Blueprints.JiraReports.Dialog.serverOptions({servers: servers}));

            if (checkJiraUnSupportedVersion(Confluence.Blueprints.JiraReport.primaryServer)) return;

            Confluence.Blueprints.JiraReport.checkOauthForm(Confluence.Blueprints.JiraReport.primaryServer);

            changeLogForm.project.trigger('servers-load-completed');
        },

        validateForm: function (state) {
            if ($createButton && $createButton.is(":disabled")) {
                return false;
            }

            if (state.$container.find('#simple-jira-search').hasClass('hidden')) {
                return Confluence.Blueprints.JiraReport.DynamicDialog.validateForm(state);
            }
            return Confluence.Blueprints.JiraReport.StaticDialog.validateForm(state);
        },

        checkOauthForm: function (server) {
            if (server && server.authUrl) {
                changeLogForm.oauthForm.removeClass('hidden');
                changeLogForm.oauthForm.click(function (e) {
                    AppLinks.authenticateRemoteCredentials(server.authUrl, function () {
                        onAuthenticatedSuccess(server);
                    });
                    e.preventDefault();
                });
            } else {
                changeLogForm.oauthForm.addClass('hidden');
            }
        },

        setupChangeLogSubmitData: function (state) {
            if (changeLogForm.switchSearchType.hasClass("simple")) {
                state.pageData.contentTemplateKey = STATIC_TEMPLATE_KEY;
                state.pageData.multiVersion = Confluence.Blueprints.JiraReport.StaticDialog.getMultiVersion();
            } else {
                state.pageData.contentTemplateKey = DYNAMIC_TEMPLATE_KEY;
            }
        },

        setupStatusReportSubmitData: function (state) {
            if (changeLogForm.switchSearchType.hasClass("simple")) {
                state.pageData['dialogMode'] = "simple";
                state.pageData['jira-query'] = Confluence.Blueprints.JiraReport.StaticDialog.getStaticJiraQuery(state);
            }
        },

        setSelectedReportType: function (reportType) {
            changeLogForm.selectedReportType = reportType;
        }
    };
})(AJS.$);

Confluence.Blueprints.JiraReport.StaticDialog = (function ($) {

    var form;
    var JIRA_REPORT_REST_URL = Confluence.getContextPath() + '/rest/jirareports/1.0/';
    var UNAUTHORIZED_STATUS = 401;
    var DEFAULT_OPTION_VALUE = '-1';
    var MAX_TIMES_CHECK_TITLE = 5;
    var SUGGESTION_SUFFIX_LENGTH = 4;
    var TITLE_MAX_LENGTH = 255;
    var HELLIP_CHARACTER = "\u2026";
    var projectsCache = {};

    var clearAndAddDefaultOption = function (selectBox) {
        selectBox.empty();
        selectBox.append(Confluence.Blueprints.JiraReports.Dialog.defaultOption({defaultValue: DEFAULT_OPTION_VALUE}));
        selectBox.auiSelect2('val', DEFAULT_OPTION_VALUE);
        removeError(selectBox);
    };

    var clearOption = function (selectBox) {
        selectBox.empty();
        selectBox.auiSelect2('val', DEFAULT_OPTION_VALUE);
    };

    var cleanOptionAndDisableVersion = function (selectBox) {
        clearOption(selectBox);
        selectBox.auiSelect2("enable", false);
    };

    var loading = function (selectBox) {
        selectBox.empty();
        selectBox.append(Confluence.Blueprints.JiraReports.Dialog.loadingOption());
        selectBox.auiSelect2('val', 'loading');
    };

    var enableElement = function (element) {
        element.auiSelect2("enable", true);
    };

    var disableElement = function (element) {
        element.auiSelect2("enable", false);
    };

    var updateTabIndex = function (cssPath, indexCount) {
        AJS.$(cssPath).attr('tabindex', indexCount);
    };

    var fillVersionSelectBox = function (versions, projectId) {
        if (projectId === form.project.val()) {
            form.errorForm.empty();
            clearOption(form.version);
            form.version.append(Confluence.Blueprints.JiraReports.Dialog.versionOptions({versions: versions}));
            enableElement(form.version);
        }
    };

    var fillProjectSelectBox = function (projects, serverId) {
        if (serverId === form.server.val()) {
            form.errorForm.empty();
            clearAndAddDefaultOption(form.project);
            form.project.append(Confluence.Blueprints.JiraReports.Dialog.projectOptions({projects: projects}));
            enableElement(form.project);
        }
    };

    var bindElementEvent = function () {
        form.project.bind('servers-load-completed', Confluence.Blueprints.JiraReport.StaticDialog.loadProjects);

        form.project.change(function () {
            if (form.project.val() === DEFAULT_OPTION_VALUE) {
                clearOption(form.version);
                clearTitle();
            } else {
                removeError(form.project);
                Confluence.Blueprints.JiraReport.StaticDialog.loadVersions();
                setTitle(form.project.find('option:selected').text());
            }
        });

        form.project.on('select2-opening', function () {
            if (form.project.val() === DEFAULT_OPTION_VALUE) {
                form.project.find('option[value="' + DEFAULT_OPTION_VALUE + '"]').addClass('hidden');
            }
        });

        form.version.change(function () {
            var projectName = form.project.find('option:selected').text();
            if (!form.version.val()) {
                setTitle(projectName);
            } else if (form.version.val().length > 1) {
                setTitle(projectName);
            } else {
                var versionName = form.version.find('option:selected').text();
                setTitle(projectName + " " + versionName, true);
            }
        });

        form.title.change(function () {
            checkExistPageTitle();
        });
    };

    var clearTitle = function () {
        form.title.val("");
        removeError(form.title);
    };

    var checkExistPageTitle = function (title) {
        if (title === undefined) {
            title = form.title.val();
        }

        if (title === "") {
            return true;
        }

        if (!Confluence.Blueprint.canCreatePage(form.spaceKey, title)) {
            showError(form.title, AJS.I18n.getText('jirareports.changelog.blueprint.form.validation.title.exits'));
            return false;
        }
        removeError(form.title);
        return true;
    };

    var validateRequireSelectField = function (field, message) {
        if ($.trim(field.val()) === DEFAULT_OPTION_VALUE) {
            showError(form.project, message);
            return false;
        }
        return true;
    };

    var makeSuggestionTitle = function (title) {
        if (!Confluence.Blueprint.canCreatePage(form.spaceKey, title)) {
            var newTitle;
            for (var i = 1; i <= MAX_TIMES_CHECK_TITLE; i++) {
                newTitle = title.concat(" (", i, ")");
                if (Confluence.Blueprint.canCreatePage(form.spaceKey, newTitle)) {
                    removeError(form.title);
                    return newTitle;
                }
            }
            showError(form.title, AJS.I18n.getText('jirareports.changelog.blueprint.form.validation.title.exits'));
        } else {
            removeError(form.title);
        }
        return title;
    };

    var setTitle = function (title, isMakeSuggestionTitle) {
        if (isMakeSuggestionTitle && title.length < TITLE_MAX_LENGTH - SUGGESTION_SUFFIX_LENGTH) {
            form.title.val(makeSuggestionTitle(title));
            return;
        }

        if (title.length > TITLE_MAX_LENGTH) {
            title = title.substr(0, TITLE_MAX_LENGTH - 1) + HELLIP_CHARACTER;
        }
        form.title.val(title);
        checkExistPageTitle(title);
    };

    var showError = function (element, message) {
        element.focus().siblings(".error").text(message);
    };

    var removeError = function (element) {
        element.siblings(".error").empty();
    };

    var setupSelect2 = function (containerCSS, dropDownCSS) {
        return {
            width: "300px",
            containerCssClass: containerCSS,
            dropdownCssClass: dropDownCSS,
            formatResult: function (result, label, query) {
                label.attr("title", result.text);
                return $.fn.select2.defaults.formatResult.apply(this, arguments);
            }
        };
    };

    var setupPlaceHolder = function (dropDownCSS, text) {
        AJS.$(dropDownCSS + ' .select2-input').attr("placeholder", text);
    };

    var getErrorMessage = function (errorStatus) {
        if (errorStatus === 'timeout') {
            return AJS.I18n.getText('jirareports.changelog.blueprint.form.validation.connection.timeout');
        }
        return AJS.I18n.getText('jirareports.changelog.blueprint.form.validation.cannot.connect');
    };

    var handleRequestError = function (element, errorStatus) {
        clearAndAddDefaultOption(element);
        form.errorForm.empty();
        form.errorForm.append(Confluence.Blueprints.JiraReports.Dialog.showErrorMessage({'message': getErrorMessage(errorStatus)}));
        enableElement(element);
    };

    var getValueOption = function (columnSelectBox) {
        return columnSelectBox.select2("val");
    };

    return {
        init: function (changeLogForm) {
            form = changeLogForm;
            form.project.auiSelect2(setupSelect2('select2-project-container', 'select2-project-dropdown'));
            form.version.auiSelect2(setupSelect2('select2-version-container', 'select2-version-dropdown'));
            setupPlaceHolder('.select2-project-dropdown', AJS.I18n.getText('jirareports.changelog.blueprint.form.project.placeholder'));
            setupPlaceHolder('.select2-version-dropdown', AJS.I18n.getText('jirareports.changelog.blueprint.form.version.placeholder'));

            // update tab index for select boxes otherwise tab key won't work on them
            updateTabIndex('.select2-project-container > .aui-select2-choice', 10);
            updateTabIndex('.select2-version-container > .aui-select2-choice', 10);

            bindElementEvent(this);
            projectsCache = {};
        },

        validateForm: function (state) {
            state.$container.find(".error").empty();
            var isRequireValid = validateRequireSelectField(form.project, AJS.I18n.getText('jirareports.changelog.blueprint.form.validation.project.required'));
            var valid = checkExistPageTitle() && isRequireValid;
            if (valid) {
                state.pageData['jira-server-name'] = form.server.find("option:selected").text();
                state.pageData['jira-server-id'] = form.server.find("option:selected").val();
                return true;
            }
            return false;
        },

        onServerChange: function () {
            this.loadProjects(false);
            cleanOptionAndDisableVersion(form.version);
            clearTitle();
        },

        onAuthenticatedSuccess: function () {
            this.loadProjects(true);
            clearTitle();
        },

        loadProjects: function (isReload) {
            var serverId = form.server.val();
            disableElement(form.project);
            loading(form.project);
            if (!isReload && projectsCache[serverId]) {
                fillProjectSelectBox(projectsCache[serverId], serverId);
                cleanOptionAndDisableVersion(form.version);
            } else {
                AJS.$.ajax({
                    dataType: 'json',
                    url: JIRA_REPORT_REST_URL + 'appLink/' + serverId + '/projects',
                    timeout: AJS.Meta.getNumber('connection-timeout')
                }).done(function (projects) {
                    projectsCache[serverId] = projects;
                    fillProjectSelectBox(projects, serverId);
                }).fail(function (xhr, errorStatus) {
                    if (xhr.status === UNAUTHORIZED_STATUS) {
                        var projects = xhr.responseText ? JSON.parse(xhr.responseText) : null;
                        projectsCache[serverId] = projects;
                        fillProjectSelectBox(projects, serverId);
                    } else if (serverId === form.server.val()) {
                        handleRequestError(form.project, errorStatus);
                    }
                });
            }
        },

        loadVersions: function () {
            disableElement(form.version);
            loading(form.version);
            var projectId = form.project.val();
            AJS.$.ajax({
                dataType: 'json',
                url: JIRA_REPORT_REST_URL + 'appLink/' + form.server.val() + '/project/' + projectId + '/versions',
                timeout: AJS.Meta.getNumber('connection-timeout')
            }).done(function (versions) {
                fillVersionSelectBox(versions, projectId);
            }).fail(function (xhr, errorStatus) {
                if (xhr.status === UNAUTHORIZED_STATUS) {
                    fillVersionSelectBox(xhr.responseText ? JSON.parse(xhr.responseText) : null, projectId);
                } else if (projectId === form.project.val()) {
                    handleRequestError(form.version, errorStatus);
                }
            });
        },

        getStaticJiraQuery: function (state) {
            var jql = '';
            if (form.project.val() !== DEFAULT_OPTION_VALUE) {
                jql += "project = " + form.project.val();

                if (form.version.val()) {
                    var version = [];
                    $(form.version.val()).each(function (index, value) {
                        version.push("'" + form.version.find("option[value='" + value + "']").text() + "'");
                    });
                    jql += " and fixVersion in (" + version.join(",") + ")";
                }
            }
            return jql;
        },

        getMultiVersion: function () {
            return getValueOption(form.version).join(",");
        },

        disableForm: function () {
            clearAndAddDefaultOption(form.project);
            disableElement(form.project);
            cleanOptionAndDisableVersion(form.version);
            clearTitle();
            form.title.prop("disabled", true);
        },

        enableForm: function () {
            enableElement(form.project);
            enableElement(form.version);
            form.title.prop("disabled", false);
        }
    };
})(AJS.$);

Confluence.Blueprints.JiraReport.DynamicDialog = (function ($) {

    var form, container;
    var JQL_OPERATORS = /=|!=|~|>|<|!~| is | in | was | changed /i;
    var KEY_ISSUE_OPERATORS = /\s*([A-Z][A-Z]+)-[0-9]+\s*/;

    var cleanError = function (element) {
        element.focus().siblings(".error").empty();
    };

    var cleanInforWarning = function () {
        container.find("#inforWarning").empty();
    };

    var inforWarning = function (inforAlert) {
        cleanInforWarning();
        container.find("#inforWarning").append(inforAlert);
    };

    var processJiraParams = function (jiraParams) {
        var jql;
        if (jiraParams.serverIndex !== -1) {
            form.server.val(Confluence.Blueprints.JiraReport.servers[jiraParams.serverIndex].id);
            form.server.trigger("change");
            if (jiraParams.jqlQuery.length > 0) {
                jql = jiraParams.jqlQuery;
            }
        }
        else {
            inforWarning(Confluence.Blueprints.JiraReports.Dialog.noServerWarning({
                'isAdministrator': AJS.Meta.get("is-confluence-admin"),
                'contentPath': Confluence.getContextPath()
            }));
        }
        return jql;
    };

    var convertFilterToJQL = function (textSearch) {
        var url = decodeURIComponent(textSearch);
        var serverIndex = AJS.JQLHelper.findServerIndexFromUrl(url, Confluence.Blueprints.JiraReport.servers);
        if (serverIndex !== -1) {
            form.server.val(Confluence.Blueprints.JiraReport.servers[serverIndex].id);
            form.server.trigger("change");
            //error when convert filter to JQL
            var onError = function (xhr) {
                inforWarning(Confluence.Blueprints.JiraReports.Dialog.warning({'infor': AJS.I18n.getText("jirareports.changelog.blueprint.form.warning.badrequest")}));
                form.textSearch.val('summary ~ "' + textSearch + '" OR description ~ "' + textSearch + '"');
            };

            //success when convert filter to JQL
            var onSuccess = function (responseData) {
                if (responseData.jql) {
                    form.textSearch.val(responseData.jql);
                }
                else {
                    onError();
                }
            };
            AJS.JQLHelper.getJqlQueryFromJiraFilter(url, Confluence.Blueprints.JiraReport.servers[serverIndex].id, onSuccess, onError);
        }
        else {
            inforWarning(Confluence.Blueprints.JiraReports.Dialog.noServerWarning({
                'isAdministrator': AJS.Meta.get("is-confluence-admin"),
                'contentPath': Confluence.getContextPath()
            }));
            form.textSearch.val('summary ~ "' + textSearch + '" OR description ~ "' + textSearch + '"');
        }
    };

    //IE8 not support placeholder, we apply placeholder library to support it IE8
    //Have problem when you remove all text and get immediately value. It will return placeholder value not an empty value
    //This function will check it.
    var isIEPlaceHolderValue = function (element, placeHolderValue) {
        return element.hasClass('placeholded') && element.val() === placeHolderValue;
    };

    var convertToJQL = function (textSearch) {
        if ($.trim(textSearch) !== "" && !isIEPlaceHolderValue(form.textSearch, AJS.I18n.getText("jirareports.changelog.blueprint.form.field.jira.search.placeholder"))) {
            //convert Filter to JQL
            if (textSearch.indexOf('http') === 0 && AJS.JQLHelper.isFilterUrl(textSearch)) {
                convertFilterToJQL(textSearch);
                //convert URL/XML to JQL
            } else if (textSearch.indexOf('http') === 0 && AJS.JQLHelper.isIssueUrlOrXmlUrl(textSearch)) {
                var jiraParams = AJS.JQLHelper.getJqlAndServerIndexFromUrl(decodeURIComponent(textSearch), Confluence.Blueprints.JiraReport.servers);
                if (processJiraParams(jiraParams)) {
                    form.textSearch.val(jiraParams.jqlQuery);
                } else {
                    form.textSearch.val('summary ~ "' + textSearch + '" OR description ~ "' + textSearch + '"');
                }
            } else {
                //check is JQL
                if (textSearch.indexOf('http') !== 0 && textSearch.match(JQL_OPERATORS)) {
                    form.textSearch.val(textSearch);
                    //convert key to JQL
                } else if (textSearch.match(KEY_ISSUE_OPERATORS)) {
                    form.textSearch.val("key=" + textSearch);
                    //convert text to JQL
                } else {
                    form.textSearch.val('summary ~ "' + textSearch + '" OR description ~ "' + textSearch + '"');
                }
            }
        }
    };

    //auto convert URL/XML/Filter/Text/Key to JQL
    var autoConvert = function () {
        cleanError(form.textSearch);
        cleanInforWarning();
        setTimeout(function () {
            var textSearch = form.textSearch.val();
            convertToJQL(textSearch);
        }, 100);
    };

    var validateRequiredField = function (fieldId, error) {
        var field = container.find(fieldId);
        if ($.trim(field.val()) === "") {
            field.focus().siblings(".error").html(error);
            return false;
        }
        return true;
    };

    var checkExistTitleInSearchAdvance = function () {
        if (form.title.val() !== "" && !Confluence.Blueprint.canCreatePage(form.spaceKey, form.title.val())) {
            return false;
        }
        return true;
    };

    var resetForm = function () {
        form.textSearch.val("");
        form.textSearch.siblings(".error").empty();
        cleanInforWarning();
    };

    return {
        init: function (changeLogForm, containerState) {
            form = changeLogForm;
            container = containerState;

            form.textSearch.bind('paste', autoConvert);

            form.textSearch.bind('change', autoConvert);
        },

        onServerChange: resetForm,

        validateForm: function (state) {
            if (!validateRequiredField("#jira-query", AJS.I18n.getText("jirareports.changelog.blueprint.form.validation.jira.search.required"))) {
                return false;
            }

            if (!checkExistTitleInSearchAdvance()) {
                return false;
            }
            //set server name, fix bug for error edit marco choose right server
            state.pageData['jira-server-name'] = form.server.find("option:selected").text();
            state.pageData['jira-server-id'] = form.server.find("option:selected").val();

            if (!form.textSearch.val().match(JQL_OPERATORS)) {
                convertToJQL(form.textSearch.val());
                state.pageData['jira-query'] = form.textSearch.val();
            }

            return true;
        },

        disableForm: function () {
            resetForm();
            form.textSearch.prop("disabled", true);
        },

        enableForm: function () {
            form.textSearch.prop("disabled", false);
        }
    };
})(AJS.$);
