define(
    "cp/component/annotation/add-annotation-view",
    [
        "backbone",
        "underscore",
        "jquery",
        "ajs",
        "cp/component/utils/editor-view",
        'cp/component/utils/editor-utils',
        "MediaViewer",
        "core/template-store-singleton",
        "confluence/message-controller"
    ],
    function (
        Backbone,
        _,
        $,
        AJS,
        EditorView,
        EditorUtils,
        MediaViewer,
        templateStore,
        MessageController
    ) {

        "use strict";
        var fileChangeMode = MediaViewer.prototype.changeMode;
        var AddAnnotationView = Backbone.View.extend({
            tagName: "div",

            id: "cp-add-annotation",

            events: {},

            initialize: function(attrs) {
                this._mediaViewer = attrs.mediaViewer;
                this._annotationView = attrs.annotationView;
                this.currentUser = {
                    name: AJS.Meta.get("current-user-fullname"),
                    avatar: AJS.Meta.get("current-user-avatar-uri-reference"),
                    profile: AJS.contextPath() + "/display/~" + AJS.Meta.get("remote-user")
                };

                this._annotation = attrs.annotation;

                this.listenTo(this._mediaViewer._fileState, "cp.hideAnnotations", this.clearAnnotation);
                this.listenTo(this._mediaViewer, "fv.close", this.clearAnnotation);
                this.listenTo(this.collection, "pinSelected", this.teardown);
            },

            render: function() {
                this.editorView = this._annotationView._editorView = new EditorView({
                    editorSetup: function() {
                        this.$el.html(templateStore.get('Annotation.addAnnotation')({
                            author: this.currentUser
                        }));
                    }.bind(this),
                    container: this.$el,
                    saveHandler: _.bind(this.saveAnnotation, this),
                    cancelHandler: _.bind(this.cancelAnnotation, this),
                    content: "",
                    restoreCallback: _.bind(this.cancelAnnotation, this),
                    mediaViewer: this._mediaViewer,
                    errorCallback: _.bind(this._handleEditorError, this)
                });
                this.editorView.render();

                var _this = this;
                this._mediaViewer.changeMode = function() {
                    if (EditorUtils.confirmProcess()) {
                        _this.cancelAnnotation();
                        return fileChangeMode.apply(this, arguments);
                    }
                };

                return this;
            },

            cancelAnnotation: function(e) {
                e && e.preventDefault();
                this.removeNewAnnotation();
                this.collection.setSelected(this.collection.getCurrentOrNext());
            },

            removeNewAnnotation: function() {
                this.clearAnnotation();
                this._annotation.destroy();
                this.editorView && this.editorView.remove();
                this._mediaViewer.changeMode = fileChangeMode;
            },

            clearAnnotation: function() {
                this._mediaViewer.getView().$el
                  .find("#cp-file-control-annotate")
                  .draggable("option", "disabled", false);
            },

            saveAnnotation: function(e) {
                e && e.preventDefault();

                if(this.editorView.getContent() === "") {
                    this._generateError(AJS.I18n.getText('cp.annotations.save.empty'));
                    return;
                }

                if (AJS.Meta.get("access-mode") === 'READ_ONLY') {
                    this._handleEditorError('READ_ONLY');
                    return;
                }

                this._annotation.set("editorFormat", this.editorView.getContent(), {silent: true});
                this._annotation.set("isNew", false);
                this._annotation.save( null, {
                    error: this._handleError.bind(this),
                    success: this._handleSuccess.bind(this)
                });
            },

            teardown: function() {
                if (this._annotation.isNew()) {
                    this.removeNewAnnotation();
                } else {
                    this.clearAnnotation();
                }
                return this;
            },

            _handleEditorError: function(reason) {
                var errorMessage;
                switch (reason) {
                    case 'READ_ONLY':
                        errorMessage = AJS.I18n.getText('read.only.mode.default.error.short.message');
                        break;
                    default:
                        errorMessage = AJS.I18n.getText('cp.annotations.editor.open.fail');
                }
                this._generateError(errorMessage);
            },

            _handleSuccess: function() {
                this.editorView && this.editorView.remove();
                this.clearAnnotation();
                AJS.trigger('analyticsEvent', {
                    name: 'confluence-spaces.previews.annotation.comment',
                    data: {
                        fileType: this._mediaViewer.getCurrentFile().get("type"),
                        attachmentId: this._mediaViewer.getCurrentFile().get("id")
                    }
                });
                this.collection.setSelected(this._annotation);
                this._mediaViewer.changeMode = fileChangeMode;
            },

            _handleError: function(model, jqXHR) {
                this._generateError(MessageController.parseError(jqXHR, AJS.I18n.getText('cp.annotations.save.fail')));
            },

            _generateError: function(error) {
                $(".cp-error").empty();
                AJS.messages.error(".cp-error", {
                    body: templateStore.get('escapeError')({ text: error }),
                    closeable: false,
                    id: "cp-annotation-error"
                });
            }
        });

        return AddAnnotationView;
    });
