define("cp/component/annotation/comments",
    [
        "backbone",
        "underscore",
        "cp/component/annotation/comment"
    ],
    function (
        Backbone,
        _,
        Comment
    ) {
        "use strict";

        var matches = function (properties) {
            return function (model) {
                for (var key in properties) {
                    if (properties[key] !== model.get(key)) {
                        return false;
                    }
                }
                return true;
            };
        };

        var Comments = Backbone.Collection.extend({
            model: Comment,

            initialize: function (attrs, opts) {
                this._current = null;
                this.service = opts && opts.service;

                this._currentFilter = {resolved: false};

                if (opts && opts.fileModel) {
                    this.fileModel = opts.fileModel;
                }
                this.listenTo(this, "remove", this.selectNextIfRemoved);
            },

            comparator: function (comment1, comment2) {
                var position1 = comment1.get("position");
                var pageNumber1 = comment1.get("pageNumber");

                var position2 = comment2.get("position");
                var pageNumber2 = comment2.get("pageNumber");

                // first sort by page number
                if (pageNumber1 !== pageNumber2) {
                    return pageNumber1 > pageNumber2 ? 1 : -1;
                }

                // resolve ties by y-coordinate
                if (position1[1] !== position2[1]) {
                    return position1[1] > position2[1] ? 1 : -1;
                }

                // lastly, compare by x-coordinate
                return (
                    position1[0] > position2[0] ? 1 :
                        position1[0] < position2[0] ? -1
                            : 0);
            },

            selectNextIfRemoved: function (model) {
                if (this._current === model) {
                    this.next();
                }
            },

            fetchComments: function () {
                // Don't fetch again if we already have a list of comments.
                if (this.size() === 0) {
                    return this.fetch();
                } else {
                    return $.when();
                }
            },

            sync: function (method, collection) {
                if (!this.service) {
                    return;
                }

                if (method === "read") {
                    var commentsPromise = collection.service.getAnnotations(),
                        that = this;

                    commentsPromise && commentsPromise.done(function (comments) {
                        comments = _.map(comments, function (comment) {
                            return new Comment(comment, {
                                service: that.service
                            });
                        });
                        collection.reset(comments, {silent: true});
                        collection.trigger("sync", collection);
                    });

                    return commentsPromise;
                }
            },

            currentIndexOfWhere: function (comment, properties) {
                return properties ? this.where(properties).indexOf(comment) : this.indexOf(comment);
            },

            nextWhere: function (properties) {
                var index = this.indexOf(this._current);
                var model = this.chain()
                    .rest(index + 1)
                    .filter(matches(properties))
                    .first()
                    .value();

                model = model || this.chain()
                    .filter(matches(properties))
                    .first()
                    .value();

                this.setSelected(model);
                return model;
            },

            prevWhere: function (properties) {
                var index = this.indexOf(this._current);
                var model = this.chain()
                    .first(Math.max(index, 0))
                    .filter(matches(properties))
                    .last()
                    .value();

                model = model || this.chain()
                    .filter(matches(properties))
                    .last()
                    .value();

                this.setSelected(model);
                return model;
            },

            currentIndexOf: function (comment) {
                return this.currentIndexOfWhere(comment, this._currentFilter);
            },

            next: function () {
                return this.nextWhere(this._currentFilter);
            },

            prev: function () {
                return this.prevWhere(this._currentFilter);
            },

            getCurrent: function () {
                return this._current;
            },

            getCurrentOrNext: function () {
                if (!this._current || !this.isFiltered(this._current)) {
                    this._current = this.next();
                }
                return this.getCurrent();
            },

            isFiltered: function (model) {
                var matcher = matches(this.getFilter());
                return matcher(model);
            },

            selectCommentWithId: function (commentId, replyId) {
                var comment = this.findWhere({id: commentId});
                if (comment && comment.get('resolved')) {
                    this.setFilter();
                }
                if (replyId) {
                    comment.replies.selectReplyWithId(replyId);
                }
                this.setSelected(comment);
                this.trigger("pinSelected");
            },

            setSelected: function (model, silent) {
                this.invoke('set', {selected: false});
                this._current = null;

                if (model) {
                    model.set({
                        selected: true
                    });
                    this._current = model;
                    model.replies.fetch();
                    !silent && this.trigger("selected", model);
                } else {
                    !silent && this.trigger("unselected");
                }
            },

            setResolved: function (model) {
                model.set({
                    resolved: true
                });
            },

            getCountWhere: function (properties) {
                return properties ? this.where(properties).length : this.size();
            },

            getCount: function () {
                return this.getCountWhere(this._currentFilter);
            },

            setFilter: function (properties) {
                this._currentFilter = properties;
                this.trigger("filterUpdated");
                return this;
            },

            getFilter: function () {
                return this._currentFilter;
            }
        });

        return Comments;

    });