/**
 * Embeds the "Edit" button directly into any appropriate placeholder
 * @since 6.14
 */
define('cp/component/companion/embedded-edit-with',[
    'jquery',
    'ajs',
    'backbone',
    'document',
    'cp/component/companion/companion-view',
    'confluence/api/event',
    'confluence/api/logger',
], function (
    $,
    AJS,
    Backbone,
    document,
    CompanionView,
    Event,
    logger
) {
    'use strict';

    var embeddedEditButtons = [];
    var placeholders = [];

    /**
     * Retrieves all edit button placeholder elements on the page and inserts the function edit button (CompanionView)
     * within them.
     *
     * @returns {Array} Array of promises that resolve to the CompanionView objects that we embedded on page load
     */
    var bindEmbeddedEdit = function () {
        var allEditButtonPlaceholders = $('.companion-edit-button-placeholder[data-linked-resource-id][data-linked-resource-container-id]');
        if (allEditButtonPlaceholders.length > 0) {
            allEditButtonPlaceholders.each(function () {
                var companionViewPromise = loadEditButton(this)['catch'](function () {
                    // Ignore duplicate embedding errors
                    logger.debug('Attempted to load edit button into placeholder more than once for file', this);
                    return Promise.resolve(undefined)
                });
                embeddedEditButtons.push(companionViewPromise);
            });
        }
        Event.trigger('cp-companion-embedded-edit-buttons-loaded', getCompanionViewObjects(embeddedEditButtons));

        return embeddedEditButtons;
    };

    /**
     * Embeds CompanionView into a particular placeholder element. Note that although this function is now synchronous,
     * we must maintain the same return type (promise) as this is public API.
     *
     * @param placeholder DOM element to insert CompanionView within
     * @returns {Promise} Resolves the CompanionView object that was inserted. Rejects when the placeholder is full.
     */
    var loadEditButton = function (placeholder) {
        var $placeholder = $(placeholder);

        // Ensure that we do not embed the edit button view (CompanionView) into a placeholder more than once
        var injectedButton = $placeholder.children('.companion-edit-button');
        var placeholderEmbeddedIndex = placeholders.indexOf($placeholder[0]);
        if (injectedButton.length > 0 || placeholderEmbeddedIndex !== -1) {
            return Promise.reject('EmbeddedEditWith.loadEditButton: Companion Edit With button already loaded');
        }
        placeholders.push($placeholder[0]);

        var buttonContext = {
            pageID: $placeholder.attr("data-linked-resource-container-id"),
            attachmentId: $placeholder.attr("data-linked-resource-id"),
            templateName: $placeholder.attr("data-template-name")
        };
        return Promise.resolve(_injectFileIntoPlaceholder(buttonContext, placeholder));
    };

    var _injectFileIntoPlaceholder = function (buttonContext, placeholder) {
        var companionView = new CompanionView({
            template: buttonContext.templateName,
            el: placeholder,
            fileId: buttonContext.attachmentId
        });
        companionView.render();
        return companionView;
    };

    /**
     * Returns all of the "CompanionView" Backbone View objects that have been embedded into the page at page load.
     * The CompanionView provides the edit functionality.
     *
     * @returns {Promise} Resolves an array of CompanionView objects.
     */
    var getCompanionViewObjects = function (embeddedEditButtons) {
        // Return all companionMainView objects that are embedded on the page. Ignore promises that rejected (duplicate insertions)
        return Promise.all(embeddedEditButtons).then(function (results) {
            return results.filter(function (companionView) {
                return companionView !== undefined;
            })
        });
    };

    return {
        _init: bindEmbeddedEdit,
        loadEditButton: loadEditButton
    };
});

AJS.toInit(function() {
    try {
        require('cp/component/companion/embedded-edit-with')._init();
    } catch (e) {
        console.warn("Couldn't load edit resources", e)
    }
});
