define('cp/component/permalink/file-router', [
    'jquery',
    'backbone',
    'confluence/jsUri',
    "cp/component/versions/versions-navigation-plugin",
    "cp/service/files-service"
], function (
    $,
    Backbone,
    Uri,
    versionsNavigationPlugin,
    FilesService
) {
    'use strict';

    var encode = window.encodeURIComponent;
    var decode = window.decodeURIComponent;
    var usePushState = AJS.DarkFeatures.isEnabled('previews.sharing.pushstate') && history.pushState;

    var buildPushStateUrl = function(uri) {
        return uri.path() + uri.query() + (uri.anchor() ? "#" + uri.anchor() : '');
    };

    var buildRoute = function(file, annotation) {
        if (!file) {
            return removeRouteFromUrl();
        }

        var ownerId = file.get('ownerId');
        var attachmentId;
        if ( file.isLatestVersion && !file.isLatestVersion() ) {
            attachmentId = file.getLatestVersion().get('id');
        } else {
            attachmentId = file.get('id');
        }
        var version = file.get('version');
        var commentId = annotation && annotation.get('id');
        var src = file.get('src');
        var name = file.get('name');

        var idPart;
        if (!annotation) {
            idPart = ownerId ? (ownerId + '/' + attachmentId) : encode(src);
        }
        else {
            idPart = ownerId + '/' + attachmentId + '/' + version + '/' + commentId;
        }
        var namePart = name ? ('/' + encode(name)) : '';

        if (usePushState) {
            var previewParam = "/" + idPart + namePart;
            var uri = new Uri(window.location.href);
            uri = uri.replaceQueryParam("preview", previewParam);
            return buildPushStateUrl(uri);
        } else {
            return '#!/preview/' + idPart + namePart;
        }
    };

    var removeRouteFromUrl = function () {
        if (usePushState) {
            var uri = new Uri(window.location.href).deleteQueryParam("preview");
            return buildPushStateUrl(uri);
        } else {
            return '';
        }
    };


    var FileRouter = Backbone.Router.extend({

        // @todo: uncouple routes from Confluence
        routes: {
            '!/preview/:ownerId/:id(/:name)': 'handleViewer',
            '!/preview/:ownerId/:id/:version/:commentId(/:replyId)(/:name)': 'handleLinkForPin',
            '!/preview/*src': 'handleWebLink',
            '': 'handleCloseViewer',
            '*path': 'handleUrl'
        },

        initialize: function (options) {
            if (Backbone.History.started) { return; }

            this._enabled = true;
            this._mediaViewer = options.mediaViewer;

            // Attached files.
            this.on('route:handleViewer', function (ownerId, id) {
                this.selectFileById(ownerId, id);
            });

            // Pin drop annotations on attached files.
            this.on('route:handleLinkForPin', function (ownerId, id, version, commentId, replyId) {
                var that = this;

                var file = this._mediaViewer.getCurrentFile();

                var selectAnnotation = function(file, event) {
                    var annotations = file.get("annotations");
                    if (annotations) {
                        that._mediaViewer._fileState.trigger('cp.showAnnotations');
                        if (event) {
                            that.listenToOnce(annotations, event, function () {
                                annotations.selectCommentWithId(commentId, replyId);
                            });
                        } else {
                            annotations.selectCommentWithId(commentId, replyId);
                        }
                    }
                };

                // If the link is to the current file then no need to change the file selection.
                if (this._mediaViewer.isOpen() && file && file.get("ownerId") === ownerId && file.get("id") === id && file.get("version") == version) {
                    selectAnnotation(file);
                } else {
                    // Otherwise route to the different file then select the annotation.
                    this.selectFileById(ownerId, id, version).done(function (file) {
                        selectAnnotation(file, 'sync');
                    });
                }
            });

            // Web images.
            this.on('route:handleWebLink', function (src) {
                // Stop listening so we do not get unwanted callbacks
                this.disableListeners();

                // Make sure that viewer is visible?
                if (!this._mediaViewer.isOpen()) {
                    this._mediaViewer.open();
                }
                this._mediaViewer.showFileWithSrc(src);

                // Start listening again to events so we get callbacks for updating routes.
                this.enableListeners();
            });

            // Checks all URLs for the route information
            var handleUrl = function () {
                if (!usePushState) {
                    return;
                }

                // If we have preview query param trigger the router.
                var uri = new Uri(window.location.href);
                if (uri.getQueryParamValue("preview")) {
                    $(window).off('popstate', handleUrl);
                    if (!this._mediaViewer || !this._mediaViewer.getCurrentFile()) {
                        // If accessing from permalink, then load the url
                        Backbone.history.loadUrl("!/preview" + decode(uri.getQueryParamValue("preview")));
                    }
                } else {
                    if (options.mediaViewer.isOpen()) {
                        options.mediaViewer.close();
                    }
                    $(window).off('popstate', handleUrl);
                    $(window).on('popstate', handleUrl);
                }
            }.bind(this);

            this.on('route:handleUrl', handleUrl);

            // Close when no permalink is in the URL
            this.on('route:handleCloseViewer', function () {
                this._mediaViewer.close();
            });

            this.enableListeners();
        },

        start: function() {
            if (!Backbone.History.started) {
                Backbone.history.start({pushState: usePushState});
            }
        },

        setEnabled: function(enabled) {
            this._enabled = enabled;
        },

        selectFileById: function(ownerId, id, version) {
            // Stop listening so we do not get unwanted callbacks
            this.disableListeners();

            // Make sure that viewer is visible
            if (!this._mediaViewer.isOpen()) {
                this._mediaViewer.open();
            }

            function showFile(file) {
                this._mediaViewer._fileState.setCurrentWithCID(file.cid);

                var filePromise;

                if (!version || version == file.get('version')) {
                    filePromise = this._mediaViewer.showFile(file);
                } else {
                    filePromise = versionsNavigationPlugin.showFileForPreviousVersion(this._mediaViewer,
                        file,
                        version);
                }

                // Start listening again to events so we get callbacks for updating routes.
                this.enableListeners();

                return filePromise;
            }

            return this._selectFileById(ownerId, id, version)
                    .then(showFile.bind(this))
                    .fail(function () {
                        this._mediaViewer._fileState.setNoCurrent();
                    }.bind(this));
        },

        _selectFileById: function(ownerId, id, version) {
            var promise = $.Deferred();
            var collection = this.getCollection();
            var file = collection.findWhere({ id: id, ownerId: ownerId });

            if (file) {
                promise.resolve(file);
            } else {
                var service = new FilesService(ownerId);

                service.getFilesWithId([id]).then(function (result) {
                    collection.add(result);

                    var file = collection.findWhere({ id: id, ownerId: ownerId });

                    if (file) {
                        promise.resolve(file);
                    } else {
                        promise.reject();
                    }
                });
            }

            return promise;
        },

        getCollection: function() {
            return this._mediaViewer._fileState.collection;
        },

        enableListeners: function() {
            // Listening to when the current file changes or viewer is closed.
            this.listenTo(this._mediaViewer, 'fv.changeFile fv.close', this._setRoute);

            // Listen for when annotations sidebar is opened.  Update URL to currently selected pin.
            this.listenTo(this._mediaViewer.getView().fileSidebarView, 'initializePanel', this._setRouteOnSidebarOpen);
            // Listen for when annotations sidebar is closed.  Reset back to URL of current file.
            this.listenTo(this._mediaViewer.getView().fileSidebarView, 'teardownPanel', this._setRouteOnSidebarClose);

        },

        disableListeners: function() {
            // Stop lisenting to mediaViewer events while we update the current file.  Is to avoid an unwanted callback.
            this.stopListening(this._mediaViewer, 'fv.changeFile fv.close', this._setRoute);

            // Stop listening to annotations sidebar opening events.
            this.stopListening(this._mediaViewer.getView().fileSidebarView, 'initializePanel', this._setRouteOnSidebarOpen);
            // Stop listening to annotations sidebar closing events.
            this.stopListening(this._mediaViewer.getView().fileSidebarView, 'teardownPanel', this._setRouteOnSidebarClose);
        },

        _setRoute: function (file, annotation, replaceState) {
            if (this._enabled) {
                var route = buildRoute(file, annotation);
                this.navigate(route, {
                    trigger: !route || (usePushState && !new Uri(route).getQueryParamValue("preview")),
                    replace: !!annotation || replaceState || (file && file.isLatestVersion && !file.isLatestVersion())
                });
            }
        },

        _setRouteOnSidebarOpen: function(panelName) {
            if ('annotations' === panelName) {
                var file = this._mediaViewer.getCurrentFile();
                var annotation = file.get('annotations').getCurrent();
                annotation && this._setRoute(file, annotation);
            }
        },

        _setRouteOnSidebarClose: function(panelName) {
            if ('annotations' === panelName) {
                this._setRoute(this._mediaViewer.getCurrentFile(), null, true);
            }
        }

    });

    return FileRouter;
});
