define("cp/component/upload/attachment-uploader",
    [
        "jquery",
        "ajs",
        "confluence-drag-and-drop/upload-utils"
    ],
    function (
        $,
        AJS,
        dnd_uploadUtils
    ) {
        "use strict";

        var BatchProgress = function () {
            this._workIdToBytesUploaded = {};
            this._totalBytes = 0;
        };

        BatchProgress.prototype = {
            update: function (workId, uploaded, fileSize) {
                if (!(workId in this._workIdToBytesUploaded)) {
                    this._totalBytes += fileSize;
                }
                this._workIdToBytesUploaded[workId] = uploaded;
            },
            percentComplete: function () {
                var totalBytesUploaded = 0;
                $.each(this._workIdToBytesUploaded, function (key, value) {
                    totalBytesUploaded += value;
                });
                return Math.round(totalBytesUploaded * 100 / this._totalBytes);
            }
        };

        /**
         * This class handles and forwards upload events from uploader to UploadNewVersionDialogView.
         */
        function AttachmentUploader($browserButton, $container, attachment, uploadService) {
            /**
             * This class is used to public methods to be called from UploadNewVersionDialogView to interact with plupload
             */
            var Uploader = Backbone.Model.extend({
                initialize: function () {
                    $(window).on('resize.attachmentUploader', function () {
                        if (uploaderInstance) {
                            uploaderInstance = _createUploader();
                        }
                    });
                },
                cancelUpload: function () {
                    currentUploadingFile && currentUploadingFile.xhrConnection && currentUploadingFile.xhrConnection.abort();
                    currentUploadingFile = null;
                },
                destroy: function () {
                    $(window).off('resize.attachmentUploader');

                    if (uploaderInstance) {
                        uploaderInstance = null;
                    }
                }
            });

            var currentUploadingFile = null,
                batchProgress = null,
                uploader = new Uploader();

            function _createUploader() {

                function startUpload(filteredFile) {
                    uploader.trigger("cp.uploader.uploadStarted");
                    uploader.trigger("cp.uploader.filesAdded", filteredFile.file);

                    if (filteredFile.hasError) {
                        // currently this is only checking for file size error
                        uploader.trigger("cp.uploader.error", uploadService.handleError(filteredFile));
                        currentUploadingFile = null;
                    } else {
                        var upload_url = _handleBeforeUpload(filteredFile);
                        if (upload_url) {
                            _uploadFile(filteredFile, upload_url)
                                .then(
                                    function (result) {
                                        // upload is completed.
                                        batchProgress = null;
                                        currentUploadingFile = null;
                                        event.target.value = '';
                                    },

                                    // The second function in a .then block is specifically designed to handle the rejection
                                    // of the promise. Trying to add a .catch block threw multiple errors from the yui-compressor
                                    // maven plugin
                                    function (errorFilteredFileObj) {
                                        uploader.trigger("cp.uploader.error", uploadService.handleError(errorFilteredFileObj));
                                        currentUploadingFile = null;
                                    }
                                );
                        }
                    }
                }

                // Create an input form on the button, ensure click events propagate in correct order
                if ($browserButton) {

                    var fileInput = document.createElement('input');
                    fileInput.type = 'file';
                    fileInput.id = 'cp-preview-upload-files-input-form';
                    fileInput.multiple = false; // Don't allow multiple file selection
                    // Position and style the file input to cover the browser button
                    // this class extra is in css/preview-upload-file-input.css file.
                    fileInput.className = 'file-preview-input-file file-preview-input-file-extra';
                    var fileInputLabel = document.createElement('label');

                    // connect the input form and the label
                    fileInputLabel.setAttribute('for', 'cp-preview-upload-files-input-form');

                    fileInputLabel.id = 'cp-preview-upload-files-input-label';
                    fileInputLabel.innerText = AJS.I18n.getText('cp.upload.files.button.label');

                    fileInputLabel.className = 'file-preview-input-file-extra';

                    if (attachment) {
                        fileInput.setAttribute('accept', _getFilters(attachment.get("title")));
                    }

                    $browserButton.appendChild(fileInput);
                    $browserButton.appendChild(fileInputLabel);

                    fileInput.addEventListener('change', function (event) {
                        // Get files from the input element
                        event.preventDefault();
                        event.stopPropagation();
                        var files = Array.from(event.target.files);
                        if (files.length > 0) {
                            for (var fileIndex = 0; fileIndex < files.length; fileIndex++) {
                                var this_file = files[fileIndex];
                                // Create a filteredFile object to pass to the upload function
                                var filteredFile = {
                                    file: this_file,
                                    workId: Math.random().toString(36).toUpperCase().substr(2, 9), // Generate a unique work ID for this file
                                    hasError: false,
                                    errorCode: null,
                                    errorMessage: null
                                };

                                if (this_file.size > parseInt(uploadService.getMetaMaxSize(), 10)) {
                                    filteredFile.hasError = true;
                                    filteredFile.errorCode = dnd_uploadUtils.ErrorCode.FILE_SIZE_ERROR;
                                    filteredFile.errorMessage = AJS.I18n.getText('cp.companion.error.file.size.body');
                                }

                                startUpload(filteredFile); // Pass files to startUpload
                            }
                        }
                    });
                }

                return {
                    startUpload: startUpload,
                }
            }

            /**
             * Get filters from current previewing file
             * e.g. "image.jpg" is being previewed => return filters: ".jpg, .jpeg, .jpe"
             * @param attachmentName
             * @returns {String}
             * @private
             */
            function _getFilters(attachmentName) {
                var filters = '';
                var ext = (attachmentName.indexOf(".") !== -1) ? attachmentName.split(".").pop() : null;
                if (ext)
                {
                    var extensions = [ext];
                    var mime = dnd_uploadUtils.mimeTypes[ext.toLowerCase()];
                    extensions = (mime) ? _getExtensions(mime) : [ext];

                    filters = extensions.map(function(ext) {
                        return '.' + ext;
                    }).join(',');
                }

                return filters;
            }

            function _getExtensions (mime) {
                var _extensions = [];
                for (var _extension in dnd_uploadUtils.mimeTypes)
                {
                    if (dnd_uploadUtils.mimeTypes[_extension] === mime)
                    {
                        _extensions.push(_extension);
                    }
                }
                return _extensions;
            }

            function _handleBeforeUpload (filteredFile) {
                batchProgress = new BatchProgress();
                currentUploadingFile = filteredFile;

                var url = uploadService.getUploadUrl(),
                    params = uploadService.buildUploadParams(filteredFile.file);

                return dnd_uploadUtils.buildUrl(url, params);
            }

            function _uploadFile (filteredFile, upload_url) {
                return new Promise(function (resolve, reject) {
                    filteredFile.xhrConnection = new XMLHttpRequest();
                    filteredFile.xhrConnection.open("POST", upload_url, true);

                    // Set up progress event listener
                    filteredFile.xhrConnection.upload.onprogress = function (event) {
                        if (event.lengthComputable) {
                            var bytesUploaded = event.loaded;
                            try {
                                batchProgress.update(filteredFile.workId, bytesUploaded, filteredFile.file.size);
                                var percent = batchProgress.percentComplete() / 100;

                                uploader.trigger("cp.uploader.uploadProgress", percent);

                            } catch (error) {
                                logger.log("Error updating progress:", error);
                                filteredFile.errorCode = dnd_uploadUtils.ErrorCode.GENERIC_ERROR;
                                filteredFile.errorMessage = error.errorMessage;
                                filteredFile.hasError = true;
                                reject(filteredFile);
                            }
                        }
                    };

                    // Set up load event listener for completion or HTTP errors
                    filteredFile.xhrConnection.onload = function () {
                        try {
                            if (filteredFile.xhrConnection.status>= 200 && filteredFile.xhrConnection.status < 300) {
                                uploader.trigger("cp.uploader.fileUploaded", filteredFile.file, filteredFile.xhrConnection.response);
                                resolve(filteredFile.xhrConnection.response);
                            } else {
                                filteredFile.errorCode = dnd_uploadUtils.ErrorCode.HTTP_ERROR;
                                filteredFile.errorMessage = filteredFile.xhrConnection.response.responseText;
                                filteredFile.hasError = true;
                                reject(filteredFile);
                            }
                        } catch (error) {
                            logger.log("Error completing progress:", error);
                            filteredFile.errorCode = dnd_uploadUtils.ErrorCode.GENERIC_ERROR;
                            filteredFile.errorMessage = error.errorMessage;
                            filteredFile.hasError = true;
                            reject(filteredFile);
                        }
                    };

                    // Set up error event listener
                    filteredFile.xhrConnection.onerror = function () {
                        filteredFile.errorCode = dnd_uploadUtils.ErrorCode.HTTP_ERROR;
                        filteredFile.errorMessage = filteredFile.xhrConnection.response.responseText;
                        filteredFile.hasError = true;
                        reject(filteredFile);
                    };

                    // Prepare and send the file
                    filteredFile.xhrConnection.setRequestHeader('Content-Type', 'application/octet-stream');
                    filteredFile.xhrConnection.send(filteredFile.file);
                });
            }

            var uploaderInstance = _createUploader();

            return uploader;
        }

        return AttachmentUploader;
});