define('cp/component/utils/editor',
    [
        'jquery',
        'underscore',
        'ajs',
        'tinymce',
        'cp/component/utils/editor-utils',
        'exports',
        'MediaViewer'
    ],
    function(
        $,
        _,
        AJS,
        tinymce,
        EditorUtils,
        exports,
        MediaViewer
        ) {

        "use strict";

        var fileClose;
        var fileNext;
        var filePrev;
        var fileNextWithCid;
        var fileShowFile;

        function preActivate(preActivate, mediaViewer) {
            if (!AJS.Confluence.EditorLoader.resourcesLoaded()) {
                this.$form.find('.loading-container').spin('small');
            }

            /**
             * We need to wrap the MediaViewer functions with our editor checks, before the Editor is initialised.
             * Original functionality will be restored
             */
            fileClose = MediaViewer.prototype.close;
            mediaViewer.close = function() {
                if(EditorUtils.confirmProcess()) {
                    remove();
                    return fileClose.apply(this, arguments);
                }
            };

            fileNext = MediaViewer.prototype.showFileNext;
            mediaViewer.showFileNext = function() {
                if(EditorUtils.confirmProcess()) {
                    remove();
                    return fileNext.apply(this, arguments);
                }
            };

            filePrev = MediaViewer.prototype.showFilePrev;
            mediaViewer.showFilePrev = function() {
                if(EditorUtils.confirmProcess()) {
                    remove();
                    return filePrev.apply(this, arguments);
                }
            };

            //for the footer view
            fileNextWithCid = MediaViewer.prototype.showFileWithCID;
            mediaViewer.showFileWithCID = function() {
                if(EditorUtils.confirmProcess()) {
                    remove();
                    return fileNextWithCid.apply(this, arguments);
                }
            };

            fileShowFile = MediaViewer.prototype.showFile; //for version nav
            mediaViewer.showFile = function() {
                if(EditorUtils.confirmProcess()) {
                    remove();
                    return fileShowFile.apply(this, arguments);
                }
            };

            preActivate && preActivate();
        }

        function preInitialise(preInitialise) {
            // Make sure the content-type is set to "comment" when activating a quick comment, see CONFDEV-10526 for more detail
            AJS.Meta.set('content-type', 'comment');

            // Set autoresize-min-height for editor,
            // This code can be removed when CONFDEV-27777 is resolved
            AJS.Meta.set('min-editor-height', 80);

            AJS.Meta.set('use-inline-tasks', 'false'); // disable Inline Task

            preInitialise && preInitialise();
        }

        function postInitialise(callback, options) {
            // Hide the loading
            options.$form.find('.loading-container').hide();
            // Hide unneeded buttons
            options.$form.find('#rte-button-preview').hide();

            // Format Save button as aui link button
            options.$form.find('#rte-button-publish').text(AJS.I18n.getText('cp.save')).removeAttr('title');
            options.$form.find('#rte-spinner').parent().addClass('rte-button-spinner').appendTo('#rte-savebar .toolbar-split .toolbar-split-right'); // move this spinner to the right since we align all button on the left.
            options.$form.find("#toolbar-hints-draft-status").hide();
            options.$form.find('#wysiwygTextarea_ifr').height(AJS.Meta.get('min-editor-height')); // Set autoresize-min-height for editor
            AJS.Meta.set('min-editor-height', undefined); // reset this value

            if (this.hideCancelButton) {
                options.$form.find("#rte-button-cancel").hide();
            }

            callback && callback();

            /**
             * From Editor API:
             * Disables quick edit page handlers. Intended to be used when another editor has been
             * activated (so that the keyboard shortcut to quick-edit a page doesn't kick-in, among other things).
             *
             * If quick edit was previously enabled, calling this function directly will effectively disable
             * the 'edit' keyboard shortcut for quick and slow edit (because the slow edit shortcut handler
             * is disabled by quick edit, and is only re-bound when another editor instance is deactivated).
             */
            AJS.Confluence.QuickEdit.QuickEditPage.disable();
        }

        function postDeactivate(postDeactivate, mediaViewer) {
            // restore the original callbacks
            mediaViewer.close = fileClose;
            mediaViewer.showFileNext = fileNext;
            mediaViewer.showFilePrev = filePrev;
            mediaViewer.showFileWithCID = fileNextWithCid;
            mediaViewer.showFile = fileShowFile;

            postDeactivate && postDeactivate();
        }

        /**
         * Setup RTE in sidebar
         * @param option.container {object} The container containing the necessary structure to activate the editor within.
         * @param option.form {object} The target editor form.
         * @param option.saveHandler {function} Save handler.
         * @param option.cancelHandler {function} Cancel handler.
         * @param option.fetchContent {function} Deferred object that will return editor's content
         * @param option.postDeactivate {function} This function will be executed after an RTE is deactivated completely
         *
         * @returns promise tracking editor activation
         */
        function init(option) {
            return AJS.Confluence.QuickEdit.activateEditor({
                preActivate : _.partial(preActivate, option.preActivate, option._mediaViewer), // Gets executed before the call to fetch resources.
                preInitialise: _.partial(preInitialise, option.preInitialise), // Gets executed after the resources are fetched but before the editor gets loaded.
                postInitialise: _.partial($.proxy(postInitialise, option), option.postInitialise), // Gets executed after editor has loaded.
                toolbar: false,
                $container : option.container,
                $form : option.form,
                saveHandler: option.saveHandler,
                cancelHandler: option.cancelHandler,
                fetchContent: option.fetchContent(),
                closeAnyExistingEditor: true,
                postDeactivate: _.partial(postDeactivate, option.postDeactivate, option._mediaViewer),
                plugins: EditorUtils.getSupportedRtePlugins(),
                excludePlugins: EditorUtils.getUnsupportedRtePlugins()
            });
        }

        /**
         * Remove editor's dom and relative things.
         * @returns Deferred object
         */
        function remove() {
            if (AJS.Rte && AJS.Rte.getEditor()) {
                if(!$('.quick-comment-body .editor-container').length) {
                    return AJS.Confluence.QuickEdit.deactivateEditor();
                }
            } else {
                return $.Deferred().resolve();
            }
        }

        /**
         * Get content from current Editor
         * @returns HTML
         */
        function getContent() {
            return tinymce.activeEditor.getContent();
        }

        function setEditorBusy(isBusy) {
            $('.ic-sidebar .rte-button-spinner').toggleClass('aui-icon-wait', isBusy);
        }

        exports.init = init;
        exports.remove = remove;
        exports.getContent = getContent;
        exports.setEditorBusy = setEditorBusy;
    });
