define("cp/confluence/preview",
    [
        "underscore",
        "jquery",
        "cp/confluence/preview-support",
        "ajs",
        "confluence/jsUri"
    ],
    function (
        _,
        $,
        PreviewSupport,
        AJS,
        Uri
    ) {
            "use strict";
            var loader,
                dfd;

            $(document).ready(bindConfluencePreviews);

            function bindConfluencePreviews() {
                var $files = $(PreviewSupport.getFileSelectorString());

                // Unbind fancybox from the images on the page.
                $files.off('click.fb');

                // Bind our new lightbox onto the files.
                $(document.body).off("click.filePreviews");
                $(document.body).on("click.filePreviews keydown.filePreviews", PreviewSupport.getFileSelectorString(), previewTriggerClicked);
                $(document.body).on("click.filePreviews", PreviewSupport.SELECTOR_STRINGS.FILE_OVERLAY, _onOverlayClick);

                // check on popstate for a previewer URL (only if previewer is not open yet)
                $(window).on('popstate', function() {
                    if (!dfd && AJS.DarkFeatures.isEnabled('previews.sharing.pushstate')) {
                        detectPreviewUrl();
                    }
                });

                // Check if page currently contains a preview URL and if so load the previer.
                detectPreviewUrl();
            }

            function previewTriggerClicked(event) {
                if (event.type === 'keydown' && event.key !== 'Enter' && event.key !== ' ')  {
                    return;
                }
                // Don't open if it is a child of an <a> or if it's in the draft changes dialog or it's an image embedded
                // in an annotation, or if user holds alt | ctrl | command key.
                // If we run into this again, let's make a more generic "data-file-preview = false"
                if (!event.altKey && !event.ctrlKey && !event.metaKey
                    && !$(this).parent().closest("a, #draft-changes-dialog, #cp-annotations, .diff-block-context, .diff-block-target").length) {
                        event.preventDefault();
                        loadConfluencePreviews(this);
                }
            }

            function _onOverlayClick(event) {
                event.preventDefault();

                var $target = $(event.target);
                var wrapper = $target.closest("span.confluence-embedded-file-wrapper");

                loadConfluencePreviews(wrapper.find(PreviewSupport.SELECTOR_STRINGS.IMAGE + "," + PreviewSupport.SELECTOR_STRINGS.FILE),
                        undefined, undefined, $target.closest(".comment-count-overlay").length > 0);
            }

            function loadConfluencePreviews(file, viewMode, source, shouldAutoShowAnnotations) {
                if (!dfd) {
                    // We have never opened the lightbox prior.
                    var cssKey = 'confluence-previews-css';
                    var mediaViewerContextKey = 'media-viewer'; // css resource can be batched into file-viewer context
                    var viewAttachmentsContext = 'viewattachments'; // css resource can be batched in to viewattachments context
                    WRM.require('wr!com.atlassian.confluence.plugins.confluence-previews:' + cssKey);
                    var cssDeferred = $.Deferred(), cssPromise = cssDeferred.promise();

                    // Poll and resolve the deferred when the CSS is loaded
                    // We don't use the normal WRM.require().done() here as it blocks until the page is fully loaded,
                    // meaning that the previewer wouldn't open until all images on the page had finished downloading.
                    var checkCss = setInterval(function() {
                        for (var i = 0; i < document.styleSheets.length; i++) {
                            var styleSheet = document.styleSheets[i];
                            if (styleSheet.href
                                && (styleSheet.href.indexOf(cssKey) !== -1
                                || styleSheet.href.indexOf(mediaViewerContextKey) !== -1 || styleSheet.href.indexOf(viewAttachmentsContext) !== -1)) {
                                cssResolved();
                                return;
                            }

                            // This can be removed when we no longer support IE9
                            if (document.styleSheets[i].imports && document.all && !window.atob) {
                                for (var j = 0; j < document.styleSheets[i].imports.length; j++) {
                                    if (document.styleSheets[i].imports[j].href.indexOf(cssKey) !== -1) {
                                        cssResolved();
                                        return;
                                    }
                                }
                            }
                        }

                    }, 100);

                    var cssResolved = function () {
                        cssDeferred.resolve();
                        clearInterval(checkCss);
                    }

                    var staticPrefix = AJS.Meta.get("static-resource-url-prefix");
                    var webResource = 'com.atlassian.confluence.plugins.confluence-previews:mediaviewer-chunks';
                    var mediaviewerBasePath = staticPrefix + '/download/resources/' + webResource + '/';


                    dfd = $.when(WRM.require(['wr!com.atlassian.confluence.plugins.confluence-previews:confluence-previews-js', 'wrc!media-viewer']), cssPromise);

                    dfd.done(function() {
                        loader = require("cp/confluence/file-preview-loader");
                        showPreviewerForFile(file, viewMode, source, shouldAutoShowAnnotations);
                    });

                    Confluence.PageLoadingIndicator($("body")).showUntilResolved(dfd, AJS.I18n.getText('cp.error.load'));
                } else {
                    dfd.done(_.partial(showPreviewerForFile, file, viewMode, source, shouldAutoShowAnnotations));
                }
                return dfd;
            }

            var showPreviewerForFile = function(file, viewMode, source, shouldAutoShowAnnotations) {
                var $pageContent = $('#content');
                var $commentContainer = $(file).closest(".comment,.cq-content,.ic-content");

                if (!viewMode) {
                    if (!($pageContent.hasClass('page') || $pageContent.hasClass('blogpost'))) {
                        // CONFDEV-28895: Previewer simple mode is enabled unless you are on a page or blogpost
                        viewMode = PreviewSupport.VIEW_MODE.SIMPLE;
                    } else if ($commentContainer.length) {
                        viewMode = PreviewSupport.VIEW_MODE.COMMENT;
                    }
                }

                if (viewMode === PreviewSupport.VIEW_MODE.COMMENT) {
                    loader.showPreviewerForComment(file, $commentContainer, viewMode, shouldAutoShowAnnotations);
                } else if (viewMode === PreviewSupport.VIEW_MODE.SIMPLE) {
                    loader.showPreviewerForSingleFile(file, viewMode, source);
                } else {
                    viewMode = PreviewSupport.VIEW_MODE.FULL;
                    loader.showPreviewer(undefined, file, viewMode, shouldAutoShowAnnotations);
                }
            };

            function detectPreviewUrl() {
                if ( isPreviewUrl() ) {
                    // if the URL has a preview anchor then load immediately
                    var uri = new Uri(window.location.href);
                    var supportsPushState = window.history.pushState && AJS.DarkFeatures.isEnabled('previews.sharing.pushstate');

                    // If we have a 'preview' query pram extract and set this as the URLs anchor for Non-Push State browsers and below.
                    if (uri.getQueryParamValue("preview") && !supportsPushState ) {
                        var preview = "#!/preview"+uri.getQueryParamValue("preview");
                        var previewUrl = decodeURIComponent(uri.deleteQueryParam("preview").setAnchor(preview).toString());
                        if (window.history.replaceState) {
                            window.history.replaceState({}, "", previewUrl);
                        } else {
                            window.location = previewUrl;
                        }
                    }
                    // if URL contains a #! preview URL and our browser supports push state.  Convert to be param.
                    else if (uri.anchor().indexOf("!/preview") === 0 && supportsPushState) {
                        var previewUrl;
                        if (uri.getQueryParamValue("preview")) {
                            // We want to ignore anchor if preview param already set.
                            previewUrl = uri.setAnchor("");
                        }
                        else {
                            // Take the anchor and set it as the preview URL param.
                            previewUrl = uri.addQueryParam("preview", uri.anchor().substr("!/preview".length, uri.anchor().length)).setAnchor("");
                        }
                        window.history.replaceState({}, "", previewUrl);
                    }

                    loadConfluencePreviews();
                }
            }

            /**
             * Upon loading a confluence page this logic checks the current URL to determine if the previewer
             * should be loaded. URL formats that will trigger the previewer to load are:
             *
             *  URLs that start with the #preview anchor
             *  Pretty URLs that contain /previews e.g. /confluence/display/space/page/preview/589830/819201/image.jpg
             *  Non-pretty URLs that contain a previews param e.g. /confluence/pages/viewpage.action?preview=%2Fpreview%2F589830%2F819201%2Fmilky_way.jpg
             */
            function isPreviewUrl() {
                var uri = new Uri(window.location.href);
                return AJS.DarkFeatures.isEnabled('previews.sharing') &&
                        (uri.getQueryParamValue("preview") || (uri.anchor() && uri.anchor().indexOf("!/preview") === 0));
            }

            return {
                loadConfluencePreviews: loadConfluencePreviews
            };
    });
