define("cp/service/upload-service",
    [
        "ajs",
        "jquery",
        "cp/service/files-service",
        "confluence-drag-and-drop/upload-utils",
        "confluence-drag-and-drop/drag-and-drop-utils"
    ],
    function(
        AJS,
        $,
        FilesService,
        dnd_uploadUtils,
        dnd_dragAndDropUtils
        ) {
        'use strict';

        var DEFAULT_MIME_TYPE = "application/x-upload-data";

        var contextPath = AJS.contextPath();

        var dragAndDropEntityId = AJS.Meta.get("drag-and-drop-entity-id"),
            atlToken = AJS.Meta.get("atl-token"),
            metaMaxFileSize = AJS.Meta.get("global-settings-attachment-max-size");

        /**
         * UploadService is used to upload new file version
         * @param previewingAttachment
         * @constructor
         */
        function UploadService(previewingAttachment) {
            this.previewingAttachment = previewingAttachment;
            this.filesService = new FilesService(previewingAttachment.get("ownerId"));
        }

        UploadService.prototype.getMetaMaxSize = function() {
            return metaMaxFileSize;
        };

        UploadService.prototype.getUploadUrl = function() {
            return contextPath + "/plugins/drag-and-drop/upload.action";
        };

        UploadService.prototype.buildUploadParams = function(file) {
            var params = {},
                extension = file.name.split('.').pop();

            params.pageId = this.previewingAttachment.get("ownerId");
            params.filename = this.previewingAttachment.get("title");
            params.size = file.size;

            if (dragAndDropEntityId) {
                params.dragAndDropEntityId = dragAndDropEntityId;
            }

            params.mimeType = dnd_uploadUtils.mimeTypes[extension.toLowerCase()] || DEFAULT_MIME_TYPE;
            params.atl_token = atlToken;
            params.withEditorPlaceholder = false;

            return params;
        };

        UploadService.prototype.parseResponse = function(resp) {
            return resp ? JSON.parse(resp).data : null;
        };

        UploadService.prototype.handleError = function(errorFilteredFile) {
            var message = "";
            var response = null;
            if (errorFilteredFile && errorFilteredFile.xhrConnection) {
                response = errorFilteredFile.xhrConnection.response;
            }

            if (response) {
                try {
                    //only http errors
                    var result = JSON.parse(errorFilteredFile.xhrConnection.response);
                    message = result.actionErrors[0];
                } catch (e) {
                    message = AJS.I18n.getText('cp.error.invalid.response.from.server');
                }
            } else {
                message = errorFilteredFile.errorMessage;
                if (errorFilteredFile.errorCode === dnd_uploadUtils.ErrorCode.FILE_SIZE_ERROR) {
                    var fileName = errorFilteredFile.file.name;
                    var fileSizeLimit = dnd_dragAndDropUtils.niceSize(metaMaxFileSize).toString();
                    message = AJS.I18n.getText("dnd.validation.file.too.large.withname", fileName, fileSizeLimit);
                }
                else if (errorFilteredFile.errorCode === dnd_uploadUtils.ErrorCode.FILE_EXTENSION_ERROR) {
                    message = AJS.I18n.getText("cp.validation.file.extension.error");
                }
            }

            return message;
        };

        UploadService.prototype.promiseFileModel = function(uploadResponse) {
            var latestVersionId = (this.previewingAttachment.getLatestVersion ? this.previewingAttachment.getLatestVersion() : this.previewingAttachment).get("id");
            return this.filesService.getFileWithId(latestVersionId);
        };

        UploadService.prototype.addVersionChangeComment = function(comment, attachment, successCallback, failedCallback) {
            var id = attachment.get("id");
            var url = contextPath + "/rest/api/content/" + attachment.get("ownerId") + "/child/attachment/" + id;
            var data = {id: id,
                version: {number: attachment.get("version") + 1},
                metadata: {comment: comment}};
            $.ajax({
                url: url,
                type: "PUT",
                data: JSON.stringify(data),
                dataType: "json",
                contentType: "application/json; charset=utf-8"}
            ).done(successCallback)
            .fail(function(data){
                failedCallback(JSON.parse(data.responseText).message);
            });
        };

        return UploadService;
    }
);