define('diagnostics/detail',
    [
        'aui',
        'jquery',
        'diagnostics/common',
        'diagnostics/field-filter',
        'diagnostics/interval-filter',
        'diagnostics/paging',
        'diagnostics/date-format',
        'exports'
    ],
    function (AJS, $, Common, FieldFilter, IntervalFilter, paging, dateFormat, exports) {
        'use strict';

        /**
         * @param {jQuery|HTMLElement} $tableContainer
         * @param {jQuery|HtmlElement} $header
         * @param {jQuery|HtmlElement} $description
         * @param String issueId
         * @param String pluginKey
         */
        function AlertDetailPage($tableContainer, $header, $description, issueId, pluginKey) {
            this.data = {};
            this.issueId = issueId;
            this.pluginKey = pluginKey;
            this.$container = $($tableContainer);
            this.nodeFilter = new FieldFilter($('#node-filter'), $('#node-filter-trigger'), this.loadData.bind(this));
            this.intervalFilter = new IntervalFilter($('#interval-filter'), this.loadData.bind(this));
            this.$paging = $("#diagnostics-paging");

            JSON.parse($tableContainer.attr("data-nodes")).forEach(function(node) {
                this.nodeFilter.addValue(node)
            }.bind(this));

            paging.setChangeCallback(this.internalLoadData.bind(this));

            this.renderFilters();
        }

        AlertDetailPage.prototype.loadData = function() {
            // this only gets called via the callbacks, so in doing so, we know that we've had a change in the date
            // selected. Therefore our paging data is invalid. We must abandon it.
            paging.setPageData(0, paging.getPageData().limit);

            this.internalLoadData();
        };

        AlertDetailPage.prototype.internalLoadData = function() {
            var interval = this.intervalFilter.getInterval() || {
                from: 0,
                to: Date.now()
            };

            const url = new URL(window.location.href);
            url.pathname = AJS.contextPath() + '/rest/diagnostics/latest/alerts/details/' + this.issueId + '/' + this.pluginKey;
            url.search = new URLSearchParams({
                since: interval.from,
                until: interval.to,
                node: Common.encodeSearchParamValueSet(this.nodeFilter.getSelected()),
                start: paging.getPageData().start,
                limit: paging.getPageData().limit
            });
            fetch(url.toString()).then(response => response.json()).then((data) => {
                this.data = data;
                this.render();
            });
        };

        AlertDetailPage.prototype.renderFilters = function() {
            this.nodeFilter.render();
            this.intervalFilter.render();
        };

        AlertDetailPage.prototype.render = function() {
            $('.loading-banner-content').remove();

            if (this.data.size) {
                var $tbody = this.$container.find('table tbody');
                $tbody.empty();

                var count = 0;
                this.data.values.forEach(function(alert) {
                    var $row = $(diagnostics.detail.row({
                        time: dateFormat.toFormattedDate(alert.timestamp),
                        module: alert.trigger.module || '',
                        node: alert.nodeName,
                        details: alert.details
                    }));

                    count += (alert.elisions ? alert.elisions.count : 1);

                    var $dialog = $(diagnostics.detail.detailsDialog({
                        content: JSON.stringify(alert.details, null, 2).replace(/\\n/g, "\n\t\t")
                    }));

                    $row.find('.issue-details').on('click', function() {
                        $dialog.find('.cancel-button').click(function() {
                            AJS.dialog2($dialog).hide();
                        });

                        $("#details-dialog-container").html($dialog);
                        AJS.dialog2($dialog).show();
                    });

                    $tbody.append($row)
                });

                var textForBlame = "";
                if (this.data.plugin.key !== "not-detected") {
                    textForBlame = " " + AJS.I18n.getText('atl.diagnostics.elisions.cause', this.data.plugin.key);
                }
                $('#elisions-copy')
                    .text(AJS.I18n.getText('atl.diagnostics.elisions.count', count, textForBlame));

                paging.renderHtmlContainer(this.$paging, this.data);

            } else if (this.data.size === 0) {
                this.$container.find('table tbody').html(diagnostics.alertReport.noDataFoundRow());
                this.$paging.empty();
            } else {
                this.$container.append(diagnostics.alertReport.loadingContent());
                this.$paging.empty();
            }
        };

        exports.onReady = function(issueId, pluginKey, helpPath) {
            var reportTable = new AlertDetailPage($('#diagnostics-container'),
                $("#detail-header"),
                $("#detail-description"),
                issueId, pluginKey, helpPath);
            reportTable.internalLoadData();
            reportTable.render();
            AJS.trigger('analyticsEvent', {name: "diagnostics.page.detail"});
        };
    }
);

AJS.toInit(function() {
  const data = $('#diagnostics-container').data();
  require('diagnostics/detail').onReady(data.issueId, data.pluginKey, data.helpPath);
});
