define('diagnostics/interval-filter',
    [
        'aui',
        'jquery',
        'diagnostics/relative-date',
        'diagnostics/date-range'
    ],
    function(AJS, $, RelativeDate, DateRange) {
        'use strict';

        var SELECTION_TYPE = {
            NONE: 'NONE',
            RELATIVE: 'RELATIVE',
            RANGE: 'RANGE',
            CLEAR: 'CLEAR'
        };

        /**
         * @param {Object} container
         * @param {Function} onChangedFunction
         **/
        function IntervalFilter(container, onChangedFunction) {
            this.$container = $(container);
            this.onChangedFunction = onChangedFunction;

            this.filterOptions = [
                {
                    id: 'day',
                    type: SELECTION_TYPE.RELATIVE,
                    label: AJS.I18n.getText('atl.diagnostics.day'),
                    from: new Date().setDate(new Date().getDate() - 1)
                },
                {
                    id: 'week',
                    type: SELECTION_TYPE.RELATIVE,
                    label: AJS.I18n.getText('atl.diagnostics.week'),
                    from: new Date().setDate(new Date().getDate() - 7)
                },
                {
                    id: 'month',
                    type: SELECTION_TYPE.RELATIVE,
                    label: AJS.I18n.getText('atl.diagnostics.month'),
                    from: new Date().setMonth(new Date().getMonth() - 1)
                },
                {
                    id: 'custom-range',
                    type: SELECTION_TYPE.RANGE,
                    label: AJS.I18n.getText('atl.diagnostics.custom')
                }
            ];

            this.selection = {
                type: SELECTION_TYPE.NONE
            };

            this.$container.on('click', 'button.relative', this.updateRelativeTime.bind(this));

            this.render()
        }

        IntervalFilter.prototype.clear = function() {
            this.selection = {
                type: SELECTION_TYPE.NONE
            };
            this.onChangedFunction();
            this.render();
        };

        IntervalFilter.prototype.updateRelativeTime = function(e) {
            var button = e.target;
            var id = button.getAttribute("data-id");

            if (this.selection.id === id) {
                // we've clicked on a previously selected item
                this.clear();
            } else {

                var relative = this.filterOptions.filter(function(opt) {
                    return opt.id === id
                })[0];

                this.selection.type = SELECTION_TYPE.RELATIVE;
                this.selection.id = id;
                this.selection.payload = {
                    from: relative.from
                };
                this.onChangedFunction();
                this.render()
            }
        };

        IntervalFilter.prototype.updateRangedTime = function(id, from, to) {
            if (id === undefined) {
                this.clear();
            } else {
                this.selection.type = SELECTION_TYPE.RANGE;
                this.selection.id = id;
                this.selection.payload = {
                    from: from,
                    to: to
                };
                this.onChangedFunction();
                this.render();
            }
        };

        IntervalFilter.prototype.createChild = function(childData) {
            var selected = this.selection.id ? this.selection.id === childData.id : false;

            if(childData.type === SELECTION_TYPE.RELATIVE) {
                return RelativeDate(childData.id, childData.label, childData.from, selected)
            } else if(childData.type === SELECTION_TYPE.RANGE) {
                return DateRange(childData.id, childData.label, this.updateRangedTime.bind(this), selected, this.selection)
            }
        };

        IntervalFilter.prototype.render = function() {
            this.$container.html(diagnostics.alertReport.relativeDateSelector({
                label: AJS.I18n.getText('atl.diagnostics.relativeselect')
            }));

            var $filterContainer = this.$container.find('.relative-dates');
            this.filterOptions.map(this.createChild.bind(this)).forEach(function(elem) {
                $filterContainer.append(elem);
            })
        };

        IntervalFilter.prototype.getInterval = function() {
            if (!this.selection.id) {
                return null;
            }

            return {
                to: this.selection.payload.to || Date.now(),
                from: this.selection.payload.from
            }
        };

        return IntervalFilter;
    }
);
