AJS.namespace("Atlassian.Notifications.AddNotificationView");

Atlassian.Notifications.AddNotificationView = Backbone.View.extend({
    tagName: "div",

    initialize: function (attributes, options) {
        _.bindAll(this, 'showThrobber', 'hideThrobber', 'renderInputs', 'getCustomContainer', 'resetCustomParameterForm', 'handleRemoveRecipient', 'showGroupWarning', 'hideGroupWarning', '_formatResponse', 'handleRecipientSelect', 'addRecipientWithParam', 'renderRecipients', 'renderRecipientInput', 'renderEventInput', 'handleSubmit', 'handleSaveError', 'clearForm', 'render');
        this.model.on("change:selectedRecipients", this.renderRecipients);
        this.model.on("saveError", this.handleSaveError);

        this.update = options && options.update;

        var dialogTitle = this.update ? AJS.I18n.getText("notifications.plugin.update.notification") : AJS.I18n.getText("notifications.plugin.add.notification");
        var dialogId = this.update ? "edit-" + this.model.get("notification").id + "-notification-dialog" : "add-notification-dialog";

        //if we're editing remove any previous dialogs to prevent double ids etc
        if (this.update) {
            AJS.$("#" + dialogId).remove();
        }

        this.dialog = new AJS.Dialog({
            width: 810,
            height: 500,
            id: dialogId
        });

        this.dialog.addHeader(dialogTitle);
        this.dialog.addPanel("add-notification-page");

        this.panel = this.dialog.getPanel(0, 0);
        this.$el = AJS.$(Atlassian.Templates.Notifications.addNotificationForm());
        this.errors = this.$el.find(".messages");
        this.panel.html(this.$el);

        this.$el.on("submit", "form", this.handleSubmit);
        var buttonLabel = options && options.update ? AJS.I18n.getText("notifications.plugin.common.forms.update") : AJS.I18n.getText("notifications.plugin.common.forms.add");
        this.dialog.addButton(buttonLabel, this.handleSubmit, "notification-submit-button aui-button");
        this.dialog.addLink(AJS.I18n.getText("notifications.plugin.common.forms.cancel"), this.clearForm);
    },

    showThrobber: function () {
        if (!this.model.get("loaded")) {
            AJS.$("#scheme-admin .throbber").addClass("loading");
        }
    },

    hideThrobber: function () {
        AJS.$("#scheme-admin .throbber").removeClass("loading");
    },

    renderInputs: function () {
        this.renderEventInput();
        this.renderRecipients();
        this.renderRecipientInput();
    },

    getCustomContainer: function () {
        return this.$el.find(".custom-container");
    },

    resetCustomParameterForm: function () {
        this.getCustomContainer().empty();

    },

    handleRemoveRecipient: function (e) {
        var recipientKey = AJS.$(e.target).attr("rel");
        this.model.removeRecipient(recipientKey);
    },

    showGroupWarning: function () {
        this.$el.find("#group-warning").removeClass("hidden");
    },

    hideGroupWarning: function () {
        this.$el.find("#group-warning").addClass("hidden");
    },

    _formatResponse: function (data) {
        var results = {};

        if (AJS.$.isArray(data)) {
            AJS.$.each(data, function (i, item) {
                var val = item.id || item.value;
                var name = item.name || item.displayName;

                if (val !== undefined) {
                    results[val] = name;
                }
            });
        } else {
            //jira user picker.  Probably should extract this somehow into a x-product dep
            if (data.users !== undefined) {
                AJS.$.each(data.users, function (i, item) {
                    var val = item.name;
                    var name = item.displayName;

                    if (val !== undefined) {
                        results[val] = name;
                    }
                });
            }
        }

        return results;
    },

    handleRecipientSelect: function (e) {
        this.resetCustomParameterForm();
        var val = AJS.$(e.target).val();
        if (val !== undefined && val.length !== 0) {
            var recipient = this.model.getRecipient(val);
            if (recipient.individual) {
                this.hideGroupWarning();
            } else {
                this.showGroupWarning();
            }
            if (recipient.type !== undefined && recipient.parameterConfig) {
                if (recipient.parameterConfig.html !== undefined) {
                    this.getCustomContainer().append(recipient.parameterConfig.html);
                } else if (recipient.parameterConfig.options !== undefined) {
                    var optionGroups = [{options: recipient.parameterConfig.options}];
                    var clientSelect = AJS.$(Atlassian.Templates.Notifications.selectList({
                        "name": val,
                        "optionGroups": optionGroups
                    }));
                    this.getCustomContainer().append(clientSelect);
                    clientSelect.chosen();
                } else if (recipient.parameterConfig.url !== undefined) {
                    var $select = AJS.$(Atlassian.Templates.Notifications.selectList({
                        "name": val, "optionGroups": [
                            {options: [{value: "", name: ""}]}]
                    }));
                    this.getCustomContainer().append($select);
                    var url = recipient.parameterConfig.url;
                    $select.ajaxChosen({
                        type: 'GET',
                        url: AJS.contextPath() + url,
                        dataType: 'json',
                        minTermLength: 3,
                        jsonTermKey: "query",
                        keepTypingMsg: AJS.I18n.getText("notifications.plugin.ajax.chosen.typing"),
                        lookingForMsg: AJS.I18n.getText("notifications.plugin.ajax.chosen.looking")
                    }, this._formatResponse);
                }
            }
        }
    },

    addRecipientWithParam: function (e) {
        if (e && e.preventDefault) {
            e.preventDefault();
        }
        var recipientId = this.recipientSelect.val();
        var model = new Atlassian.Notifications.RecipientModel(this.model.getRecipient(recipientId));
        var customInput = this.$el.find(".custom-container select");
        if (customInput.length === 0) {
            customInput = this.$el.find(".custom-container input");
        }
        if (customInput.length !== 0) {
            var value = customInput.val();

            if (value === null || value === "") {
                return;
            }

            //multiselect controls can return an array. But they are still single selects.
            if (AJS.$.isArray(value)) {
                value = value[0];
            }
            var option = customInput.find("option[value='" + value + "']");
            model.set({
                "paramValue": value,
                "paramDisplay": option.length === 1 ? option.text() : value
            });
        }
        this.model.addRecipient(model);
        this.resetCustomParameterForm();
        this.recipientSelect.focus().change();
    },

    renderRecipients: function () {
        var list = AJS.$("<ul class='lozenge-list recipient-list'/>");
        list.append(Atlassian.Templates.Notifications.recipientsList({
            recipients: this.model.getConfiguredRecipients(),
            deletable: true
        }));
        this.$el.find("#recipients-list").html(list);
        list.on("click", "a.icon-close", this.handleRemoveRecipient);

    },

    renderRecipientInput: function () {
        var $container = this.$el.find("#recipients-input");
        var recipientOptions = _.map(this.model.get("recipients"), function (item) {
            return {value: item.type, name: item.name};
        });
        var serverOptions = _.map(this.model.get("servers"), function (item) {
            return {value: item.serverId, name: item.name};
        });
        var optGroups = [
            {label: AJS.I18n.getText('notifications.plugin.recipients'), options: recipientOptions},
            {label: AJS.I18n.getText('notifications.plugin.group.notifications'), options: serverOptions}
        ];

        var $select = AJS.$(Atlassian.Templates.Notifications.selectList({
            name: this.update ? "edit-recipient-" + this.model.get("notification").id : "recipient",
            multiple: false,
            optionGroups: optGroups,
            selectedEventIds: this.model.get("selectedEvents")
        }));
        $container.html($select);
        $select.chosen().change(this.handleRecipientSelect).change();
        this.recipientSelect = $select;
        this.$el.find("#submit-param").unbind("click").click(this.addRecipientWithParam);
    },

    renderEventInput: function () {
        var eventOptions = _.map(this.model.get("events"), function (item) {
            return {value: item.eventKey, name: item.name};
        });

        var allEventOptions = _.filter(eventOptions, function (item) {
            return item.value === "all_events";
        });
        var remainingEventOptions = _.filter(eventOptions, function (item) {
            return item.value !== "all_events";
        });

        var optGroups = [
            {label: AJS.I18n.getText("notifications.plugin.group.all.events"), options: allEventOptions},
            {label: AJS.I18n.getText("notifications.plugin.common.words.events"), options: remainingEventOptions}
        ];
        var $select = AJS.$(Atlassian.Templates.Notifications.selectList({
            name: this.update ? "edit-events-" + this.model.get("notification").id : "events",
            multiple: true,
            placeholder: AJS.I18n.getText("notifications.plugin.select.events"),
            optionGroups: optGroups,
            selectedValues: this.model.getSelectedEventIds()
        }));
        this.$el.find("#events-input").html($select);
        $select.chosen();

        $select.bind("selected", this.handleAddEvent);
        $select.bind("unselect", this.handleRemoveEvent);

        if (this.model.get("filterHtml") !== undefined) {
            this.$el.find(".custom-filter").html(this.model.get("filterHtml"));
        }
    },

    handleSubmit: function (e) {
        if (e && e.preventDefault !== undefined) {
            e.preventDefault();
        }

        //hack: close the jql autocomplete dropdown if it's still there in JIRA.
        this.$el.find("#jql").click();

        this.errors.empty();
        var params = this.$el.find(".custom-filter").serializeArray();

        var events = this.$el.find("#events-input select").val();
        this.model.addEvents(events);
        this.model.get("notification").set({
            "filterConfiguration": {"params": params}
        });
        this.model.saveOnServer();
    },

    handleSaveError: function (errors) {
        this.errors.empty();
        var $recipientList = this.$el.find("#recipients-list");
        AJS.$(".error", $recipientList).removeClass("error");

        var div = AJS.$("<p/>");
        _.each(errors.errorMessages, function (errorMessage) {
            div.append(errorMessage + "<br/>");
        });
        _.each(errors.errors, function (errorMessage, key) {
            AJS.$("#" + key, $recipientList).addClass("error");
        });
        AJS.messages.error(".messages", {
            title: AJS.I18n.getText("common.words.errors"),
            body: div.html(),
            closeable: false
        });

    },

    clearForm: function () {
        this.errors.empty();
        this.dialog.hide();
    },

    render: function () {
        this.renderInputs();

        this.hideThrobber();

        this.dialog.show();
        this.dialog.popup.element.addClass("jira-dialog-content-ready");

        if (this.model.get("filterHtml") !== undefined) {
            //init custom filter js controls once the dialog has been shown.
            AJS.trigger("notification-custom-filter-added", [this.$el]);
        }


        this.model.get("notification").on("saveSuccess", this.clearForm);

        return this.$el;
    }
});