/**
 * @module confluence/easyuser
 */
define('confluence/easyuser', [
    'jquery',
    'ajs'
], function(
    $,
    AJS
) {
    'use strict';

    var $sendInvitationsForm;
    var resetButton;
    var inviteLinkField;
    var inviteEmail;
    var $messageContainer;
    var $sendInvitationsSubmitButton;

    var EasyUser = {
        init: function() {
            $sendInvitationsForm = $('#easyuser-send-invites-form');
            resetButton = $('#reset-invite-token');
            inviteLinkField = $('#invite-link');
            inviteEmail = $('#inviteEmail');
            $messageContainer = $('#invite-user-errors');
            $sendInvitationsSubmitButton = $sendInvitationsForm.find('input[type="submit"]');

            AJS.MessageHandler.closeOnNew(true);
            resetButton.click(function() {
                resetButton.prop('disabled', true);
                EasyUser._postJson({
                    url: '/rest/easyuser/latest/refreshToken',
                    data: null,
                    success: function(data) {
                        inviteLinkField.attr('href', data.signupUrl).text(data.signupUrl);
                        EasyUser._msg(AJS.I18n.getText('easyuser.refresh.token.success'));
                        $messageContainer.find('.aui-message-success').first().append(
                            $('<a></a>').prop('href', '#').text(AJS.I18n.getText('easyuser.undo.link')).prop('id', 'undo-reset-token')
                                .click(EasyUser._undoTokenReset));
                    },
                    error: function(xhr, statusText, error) {
                        EasyUser._errorMsg(AJS.I18n.getText('easyuser.refresh.token.error'), status, error);
                    },
                    loadingMessage: AJS.I18n.getText('easyuser.saving.changes')
                })
                    .always(function() {
                        resetButton.removeProp('disabled');
                    });
                return false;
            });

            inviteEmail.click(function() {
                inviteEmail.prop('disabled', true);
                var notify = inviteEmail.prop('checked');
                EasyUser._postJson({
                    url: '/rest/easyuser/latest/sendEmailOnInvite',
                    data: $.toJSON(notify),
                    success: function(notifySaved) {
                        var text = notifySaved ? AJS.I18n.getText('easyuser.notify.admin.on.signup.on')
                            : AJS.I18n.getText('easyuser.notify.admin.on.signup.off');
                        EasyUser._msg(text);
                    },
                    error: function(xhr, statusText, error) {
                        EasyUser._errorMsg(AJS.I18n.getText('easyuser.update.notifications.error'), status, error);
                    },
                    loadingMessage: AJS.I18n.getText('easyuser.saving.changes')
                })
                    .always(function() {
                        inviteEmail.removeProp('disabled');
                    });
            });

            $sendInvitationsSubmitButton.click(function() {
                var emailMessage = $('#invite-user-email-content').val();
                var recipients = $.trim($('#recipient-list').val());
                if (!recipients) {
                    EasyUser._errorMsg(AJS.I18n.getText('easyuser.send.invitations.no.emails.error'));
                    return false;
                }
                recipients = recipients.split(',');
                EasyUser._postJson({
                    url: '/rest/easyuser/latest/sendUserInvites',
                    data: $.toJSON({
                        emailMessage: emailMessage,
                        recipientList: recipients
                    }),
                    success: EasyUser._sendInvitationsSuccessHandler,
                    error: function(xhr, statusText, error) {
                        EasyUser._errorMsg(AJS.I18n.getText('easyuser.send.invitations.error'), status, error);
                    },
                    loadingMessage: AJS.I18n.getText('easyuser.sending.emails')
                });
                return false;
            });
        },

        _sendInvitationsSuccessHandler: function(response) {
            // Show number of sent emails OR a mix of email addresses sent and failed
            var happyText = response.sentAddresses.length && '<span id="sent-invitations-count">' + response.sentAddresses.length + '</span>';
            var sadText = response.failedAddresses.length && '<span id="failed-invitations">' + AJS.escapeHtml(response.failedAddresses.join(', ')) + '</span>';
            if (!sadText) {
                EasyUser._msg(AJS.I18n.getText('easyuser.send.invitations.success', happyText));
            } else if (!happyText) {
                EasyUser._errorMsg(AJS.I18n.getText('easyuser.send.invitations.failure', sadText));
            } else {
                EasyUser._errorMsg(AJS.I18n.getText('easyuser.send.invitations.mixed.success', happyText, sadText));
            }
        },

        _msg: function(text, type) {
            var $messageContainer = $('#invite-user-errors');
            AJS.MessageHandler.message($messageContainer, text, type);
        },

        _errorMsg: function(text, statusText, error) {
            var $messageContainer = $('#invite-user-errors');
            AJS.MessageHandler.error($messageContainer, text);
            if (statusText) {
                AJS.log(text + ': [statusText = ' + statusText + '], [error = ' + error + ']');
            }
        },

        _postJson: function(options) {
            EasyUser._msg(options.loadingMessage, 'info');
            return $.ajax({
                url: AJS.contextPath() + options.url,
                type: 'POST',
                dataType: 'json',
                data: options.data,
                contentType: 'application/json',
                success: options.success,
                error: options.error,
                statusCode: {
                    401: function() {
                        EasyUser._errorMsg(AJS.I18n.getText('easyuser.authentication.error'));
                    }
                }
            });
        },

        _undoTokenReset: function() {
            resetButton.prop('disabled', true);
            EasyUser._postJson({
                url: '/rest/easyuser/latest/undoTokenReset',
                data: null,
                success: function(data) {
                    var previousUrl = data.signupUrl;
                    inviteLinkField.attr('href', previousUrl).text(previousUrl);
                    EasyUser._msg(AJS.I18n.getText('easyuser.undo.token.refresh.success'));
                },
                error: function(xhr, statusText, error) {
                    EasyUser._errorMsg(AJS.I18n.getText('easyuser.undo.token.refresh.error'), status, error);
                },
                loadingMessage: AJS.I18n.getText('easyuser.saving.changes')
            })
                .always(function() {
                    resetButton.removeProp('disabled');
                });
        }
    };
    return EasyUser;
});

require('confluence/module-exporter').safeRequire('confluence/easyuser', function(EasyUser) {
    'use strict';

    require('ajs').toInit(EasyUser.init);
});
