import React from 'react';
import Modal, {
    ModalBody,
    ModalFooter,
    ModalHeader,
    ModalTitle,
    type Appearance as ModalAppearance,
} from '@atlaskit/modal-dialog';
import Button, { type Appearance } from '@atlaskit/button/new';
import { ButtonGroup } from '@atlaskit/button';
import Form from '@atlaskit/form';
import { WidthNames } from '@atlaskit/modal-dialog/dist/types/types';

export type AtlassianOauth2UiDialogFormState = {
    redirectUri: string;
    incomingScopes: string[];
};

interface AtlassianOauth2UiDialogProps {
    onSubmit: (formState: AtlassianOauth2UiDialogFormState) => void;
    onClose: () => void;
    id?: string;
    titleText: string;
    children: React.ReactNode;
    primaryButtonText: string;
    closeButtonText: string;
    primaryButtonIsHidden?: boolean;
    primaryButtonIsLoading?: boolean;
    primaryButtonAppearance?: Appearance;
    modalAppearance?: ModalAppearance | null;
    modalWidth?: WidthNames;
}

export function AtlassianOauth2UiDialog(props: AtlassianOauth2UiDialogProps) {
    const {
        onSubmit,
        onClose,
        id,
        titleText,
        children,
        primaryButtonText,
        closeButtonText,
        primaryButtonIsHidden,
        primaryButtonIsLoading,
        primaryButtonAppearance = 'primary',
        modalAppearance = 'warning',
    } = props;

    function handleSubmit(formState: AtlassianOauth2UiDialogFormState) {
        onSubmit(formState);
    }

    function closeModal() {
        onClose();
    }

    return (
        <Modal testId={'oauth2-dialog-modal'} onClose={onClose}>
            {(({ children }) => (
                <Form
                    onSubmit={(formState: AtlassianOauth2UiDialogFormState) => {
                        handleSubmit(formState);
                    }}
                >
                    {({ formProps }) => (
                        <form id={id} {...formProps}>
                            {children}
                        </form>
                    )}
                </Form>
            ))({
                children: (
                    <React.Fragment>
                        <ModalHeader>
                            <ModalTitle appearance={modalAppearance || undefined}>
                                {titleText}
                            </ModalTitle>
                        </ModalHeader>
                        <ModalBody>{children}</ModalBody>
                        <ModalFooter>
                            <ButtonGroup>
                                {primaryButtonIsHidden ? null : (
                                    <Button
                                        appearance={primaryButtonAppearance}
                                        type="submit"
                                        testId="submit-oauth2-dialog"
                                        isLoading={primaryButtonIsLoading}
                                    >
                                        {primaryButtonText}
                                    </Button>
                                )}
                                <Button
                                    appearance="subtle"
                                    onClick={closeModal}
                                    testId="close-oauth2-dialog"
                                >
                                    {closeButtonText}
                                </Button>
                            </ButtonGroup>
                        </ModalFooter>
                    </React.Fragment>
                ),
            })}
        </Modal>
    );
}
