import React, { StrictMode, useEffect, useReducer } from 'react';
import { createRoot } from 'react-dom/client';
import { ModalTransition } from '@atlaskit/modal-dialog';

import {
    createIncomingLinkDialogInitialState,
    createIncomingLinkDialogReducer,
    CreateIncomingLinkDialog,
} from './CreateIncomingLinkDialog/CreateIncomingLinkDialog';
import {
    incomingLinkDetailsDialogInitialState,
    incomingLinkDetailsDialogReducer,
    IncomingLinkDetailsDialog,
} from './IncomingLinkDetailsDialog/IncomingLinkDetailsDialog';
import { CreateOutgoingLinkDialog } from './CreateOutgoingLinkDialog/CreateOutgoingLinkDialog';
import {
    createOutgoingLinkDialogInitialState,
    createOutgoingLinkDialogReducer,
} from './CreateOutgoingLinkDialog/CreateOutgoingLinkReducer';
import { ResetCredentialsDialog } from './ResetCredentialsDialog/ResetCredentialsDialog';
import {
    resetCredentialsDialogInitialState,
    resetCredentialsDialogReducer,
} from './ResetCredentialsDialog/ResetCredentialsDialogReducer';
import { RotateCredentialsDialog } from './RotateCredentialsDialog/RotateCredentialsDialog';
import { RevokeCredentialsDialog } from './RevokeCredentialsDialog/RevokeCredentialsDialog';
import {
    rotateCredentialsDialogInitialState,
    rotateCredentialsDialogReducer,
} from './RotateCredentialsDialog/RotateCredentialsDialogReducer';
import {
    revokeCredentialsDialogInitialState,
    revokeCredentialsDialogReducer,
} from './RevokeCredentialsDialog/RevokeCredentialsDialogReducer';
import { FlagsProvider } from '@atlaskit/flag';

type AtlassianOauth2UiAction = { type: string; detail?: unknown };

const initialState = {
    createIncomingLinkDialog: createIncomingLinkDialogInitialState,
    incomingLinkDetailsDialog: incomingLinkDetailsDialogInitialState,
    createOutgoingLinkDialog: createOutgoingLinkDialogInitialState,
    resetCredentialsDialog: resetCredentialsDialogInitialState,
    rotateCredentialsDialog: rotateCredentialsDialogInitialState,
    revokeCredentialsDialog: revokeCredentialsDialogInitialState,
};

export type AtlassianOauth2UiState = typeof initialState;

type AtlassianOauth2UiSubstateReducer = (state: any, action: any) => any;

const subreducers: {
    key: keyof AtlassianOauth2UiState;
    handle: AtlassianOauth2UiSubstateReducer;
}[] = [
    { key: 'createIncomingLinkDialog', handle: createIncomingLinkDialogReducer },
    { key: 'incomingLinkDetailsDialog', handle: incomingLinkDetailsDialogReducer },
    { key: 'createOutgoingLinkDialog', handle: createOutgoingLinkDialogReducer },
    { key: 'resetCredentialsDialog', handle: resetCredentialsDialogReducer },
    { key: 'rotateCredentialsDialog', handle: rotateCredentialsDialogReducer },
    { key: 'revokeCredentialsDialog', handle: revokeCredentialsDialogReducer },
];

function reducer(
    state: AtlassianOauth2UiState,
    action: AtlassianOauth2UiAction
): AtlassianOauth2UiState {
    let newState = state;
    for (const subreducer of subreducers) {
        const { key, handle } = subreducer;
        newState = {
            ...newState,
            [key]: handle(newState[key], action),
        };
    }
    return newState;
}

export function AtlassianOauth2Ui() {
    const [state, dispatch] = useReducer(reducer, initialState);

    useEffect(() => {
        const openCreateIncomingLinkDialog = (event: CustomEvent) => {
            dispatch({
                type: 'cil:open',
                detail: event.detail,
            });
        };

        const openIncomingLinkDetailsDialog = (event: CustomEvent) => {
            dispatch({
                type: 'ild:openFromEvent',
                detail: event.detail,
            });
        };

        const openCreateOutgoingLinkDialog = (event: CustomEvent) => {
            dispatch({
                type: 'col:open',
                detail: event.detail,
            });
        };
        const openResetCredentialsDialog = (event: CustomEvent) => {
            dispatch({
                type: 'rcd:openResetFromEvent',
                detail: event.detail,
            });
        };

        const openRotateCredentialsDialog = (event: CustomEvent) => {
            dispatch({
                type: 'rcd:openRotateFromEvent',
                detail: event.detail,
            });
        };
        const openRevokeCredentialsDialog = (event: CustomEvent) => {
            dispatch({
                type: 'rcd:openRevokeFromEvent',
                detail: event.detail,
            });
        };

        document.addEventListener(
            'AtlassianOauth2Ui:openCreateIncomingLinkDialog',
            openCreateIncomingLinkDialog as EventListener
        );

        document.addEventListener(
            'AtlassianOauth2Ui:openIncomingLinkDetailsDialog',
            openIncomingLinkDetailsDialog as EventListener
        );

        document.addEventListener(
            'AtlassianOauth2Ui:openCreateOutgoingLinkDialog',
            openCreateOutgoingLinkDialog as EventListener
        );

        document.addEventListener(
            'AtlassianOauth2Ui:openResetCredentialsDialog',
            openResetCredentialsDialog as EventListener
        );

        document.addEventListener(
            'AtlassianOauth2Ui:openRotateCredentialsDialog',
            openRotateCredentialsDialog as EventListener
        );
        document.addEventListener(
            'AtlassianOauth2Ui:openRevokeCredentialsDialog',
            openRevokeCredentialsDialog as EventListener
        );

        return () => {
            document.removeEventListener(
                'AtlassianOauth2Ui:openCreateIncomingLinkDialog',
                openCreateIncomingLinkDialog as EventListener
            );
            document.removeEventListener(
                'AtlassianOauth2Ui:openIncomingLinkDetailsDialog',
                openIncomingLinkDetailsDialog as EventListener
            );
            document.removeEventListener(
                'AtlassianOauth2Ui:openCreateOutgoingLinkDialog',
                openCreateOutgoingLinkDialog as EventListener
            );
            document.removeEventListener(
                'AtlassianOauth2Ui:openResetCredentialsDialog',
                openResetCredentialsDialog as EventListener
            );

            document.removeEventListener(
                'AtlassianOauth2Ui:openRotateCredentialsDialog',
                openRotateCredentialsDialog as EventListener
            );
            document.removeEventListener(
                'AtlassianOauth2Ui:openRevokeCredentialsDialog',
                openRevokeCredentialsDialog as EventListener
            );
        };
    }, [dispatch]);

    return (
        <div id="atlassian-oauth2-ui-from-react">
            <FlagsProvider>
                <ModalTransition>
                    {state.createIncomingLinkDialog.isOpen && (
                        <CreateIncomingLinkDialog
                            onClose={() => dispatch({ type: 'cil:close' })}
                            substate={state['createIncomingLinkDialog']}
                            dispatch={dispatch}
                        />
                    )}
                    {state.incomingLinkDetailsDialog.isOpen && (
                        <IncomingLinkDetailsDialog
                            onClose={() => dispatch({ type: 'ild:close' })}
                            substate={state['incomingLinkDetailsDialog']}
                            dispatch={dispatch}
                        />
                    )}
                    {state.createOutgoingLinkDialog.isOpen && (
                        <CreateOutgoingLinkDialog
                            applinkId={state['createOutgoingLinkDialog'].applinkId}
                            onClose={() => dispatch({ type: 'col:close' })}
                        />
                    )}
                    {state.resetCredentialsDialog.isOpen && (
                        <ResetCredentialsDialog
                            onClose={() => dispatch({ type: 'rcd:resetClose' })}
                            substate={state['resetCredentialsDialog']}
                            dispatch={dispatch}
                        />
                    )}
                    {state.rotateCredentialsDialog.isOpen && (
                        <RotateCredentialsDialog
                            onClose={() => dispatch({ type: 'rcd:rotateClose' })}
                            substate={state['rotateCredentialsDialog']}
                            dispatch={dispatch}
                        />
                    )}
                    {state.revokeCredentialsDialog.isOpen && (
                        <RevokeCredentialsDialog
                            onClose={() => dispatch({ type: 'rcd:revokeClose' })}
                            substate={state['revokeCredentialsDialog']}
                            dispatch={dispatch}
                        />
                    )}
                </ModalTransition>
            </FlagsProvider>
        </div>
    );
}

export function initAtlassianOauth2Ui(rootElement: HTMLElement) {
    const root = createRoot(rootElement);
    root.render(
        <StrictMode>
            <AtlassianOauth2Ui />
        </StrictMode>
    );
}
